///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

// Launcher for Open Inventor / RemoteViz Demos
//
// NOTE: This is just a launcher, not the actual demo!
//       It is not necessary to use this launcher.
//       User can start the service and load the HTML directly.
//
// Strategy:
//   - Check that RemoteService executable exists.
//   - Check that HTML demo file exists.
//   - Start the RemoteService executable.
//   - Wait a few seconds for the service to be ready to listen for connections.
//   - Display the HTML demo file using the default HTML browser.
//
// Notes:
//   - Depends on OIVHOME and OIVARCH environment variables being defined.
//
//   - Depends on the system having a default HTML browser defined.
//     Usually a good assumption.
//
//   - It's possible the wait will not be long enough.
//     In this case the user can simply click the reload button in the browser.
//
//   - Eventually might be possible to detect when the service is ready.

#include <Windows.h>
#include <shellapi.h>

#include <iostream>
#include <string>
#include <stdlib.h>  // getenv

const std::string Demo_Name = "medicalImageViewerRemote";

const std::string Exe_Name = "medicalImageViewerService.exe";

const std::string HTML_Path1 = "/examples/source/Medical/Web/";
const std::string HTML_Path2 = "/Clients/HTML5/index.html";

static bool FileExists( const std::string filename )
{
  DWORD fileAttr = ::GetFileAttributes( filename.c_str() );
  return (fileAttr != INVALID_FILE_ATTRIBUTES);
}

#pragma warning(disable: 4996)

void main( int argc, char** argv )
{
  // Get the correct environment
  std::string rootDir = getenv( "OIVHOME" );

  std::string archStr = getenv( "OIVARCH" );
  if (archStr.empty()) {
    archStr = "arch-Windows-x86_64-msvc11-Release";
  }

  // Get the paths we need to start
  // - Example exe path : C:\OIVMEDemo\examples\bin\arch-Windows-x86_64-msvc11-Release\Medical\medicalSimpleRemoteService.exe
  // - Example html path: C:\OIVMEDemo\examples\Medical\Web\medicalSimpleRemote\Clients\HTML5\index.html

#ifdef EXECUTABLE_PATH
  std::string fullExePath = rootDir + "/" + EXECUTABLE_PATH + "/" + Exe_Name;
#else
  std::string fullExePath = rootDir + "/examples/bin/" + archStr + "/Medical/" + Exe_Name;
#endif
  if (! FileExists(fullExePath)) {
    std::string msg = "Could not find RemoteViz service executable:\n" + fullExePath;
    ::MessageBox( NULL, msg.c_str(), "Launcher Error", MB_OK );
    return;
  }

  std::string fullHtmlPath = rootDir + HTML_Path1 + Demo_Name + HTML_Path2;
  if (! FileExists(fullExePath)) {
    std::string msg = "Could not find RemoteViz client HTML:\n" + fullHtmlPath;
    ::MessageBox( NULL, msg.c_str(), "Launcher Error", MB_OK );
    return;
  }

  //---------------------------------------------------------------------------
  // Start the service.
  // For this we use CreateProcess so we can check the result.
  STARTUPINFO si;
  PROCESS_INFORMATION pi;
  ZeroMemory( &si, sizeof(si) );
  si.cb = sizeof(si);
  ZeroMemory( &pi, sizeof(pi) );
  std::string cmdLineString = "\"" + fullExePath + "\""; // Quotes in case of space in pathname
  char* cmdLine = (char*)cmdLineString.c_str();          // CreateProcess will not accept "const char*"
  //std::cout << "Starting: " << cmdLineString << std::endl;

  if ( !CreateProcess( NULL,   // No module name (use command line)
        cmdLine,        // Command line
        NULL,           // Process handle not inheritable
        NULL,           // Thread handle not inheritable
        FALSE,          // Set handle inheritance to FALSE
        0,              // No creation flags
        NULL,           // Use parent's environment block
        NULL,           // Use parent's starting directory 
        &si,            // Pointer to STARTUPINFO structure
        &pi )           // Pointer to PROCESS_INFORMATION structure
    ) 
  {
    std::string msg = "CreateProcess failed for RemoteViz service:\n" + cmdLineString;
    ::MessageBox( NULL, msg.c_str(), "Launcher Error", MB_OK );
    return;
  }

  // See if the service has exited.
  // If it did something went wrong...
  DWORD dwExitCode = WaitForSingleObject( pi.hProcess, 0 );

  if (   (dwExitCode == WAIT_FAILED   )
      || (dwExitCode == WAIT_OBJECT_0 )
      || (dwExitCode == WAIT_ABANDONED) )
  {
    std::string msg = "RemoteService service failed to start:\n" + cmdLineString;
    ::MessageBox( NULL, msg.c_str(), "Launcher Error", MB_OK );

    // Service stopped so close the handles
    CloseHandle( pi.hProcess );
    CloseHandle( pi.hThread );
    return;
  }

  // This message seems to appear in the service's console window.
  std::cout << "Launcher: Started RemoteViz service: " << Demo_Name << "\n\n";

  // Wait for service to get started before we try to launch the HTML.
  // In theory this call will not return until the service is running (has
  // loaded all its DLLs and so on) and is waiting in its message loop.
  // It's not clear if this really works...
  //
  // Return values are:
  //   - 0 : wait satisfied normally    (probably running now)
  //   - WAIT_TIMEOUT : timeout expired (still not running?)
  //   - WAIT_FAILED  : error occured   (probably not running)
  DWORD result = WaitForInputIdle( pi.hProcess, 3000 );

  // Just to be safe let's give it a little more time.
  Sleep( 2000 );

  //---------------------------------------------------------------------------
  // Launch the HTML page.
  // For this we use 'ShellExecute' because passing an HTML file should
  // automatically start the user's default HTML browser (just like
  // double-clicking an HTML file in Windows Explorer).
  std::string htmlLineString = "\"" + fullHtmlPath + "\""; // Quotes in case of space in pathname
  char* htmlLine = (char*)htmlLineString.c_str(); // CreateProcess will not accept "const char*"
  std::cout << "\nLauncher: Starting RemoteViz client.\n\n";

  ShellExecuteA( NULL, "open", htmlLine, NULL, NULL, SW_SHOW );

  // We don't need to wait for this.  The user will close the service when they're done.
}

