///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include "RemoteRenderAreaListener.h"

#include <RemoteViz/Rendering/RenderArea.h>
#include <RemoteViz/Rendering/Connection.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/SoSceneManager.h>

///////////////////////////////////////////////////////////////////////////////
// Constructor
RemoteRenderAreaListener::RemoteRenderAreaListener() : m_dataLoaded(false)
{
}

///////////////////////////////////////////////////////////////////////////////
// Utility method to send a key press event to the ServiceListener.
static void
doKeyPress(std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea, SoKeyboardEvent::Key key )
{
    SoKeyboardEvent keyEvent;
    keyEvent.setKey( key );
    keyEvent.setState( SoButtonEvent::State::DOWN );
    renderArea->getSceneManager()->processEvent( &keyEvent );
}

///////////////////////////////////////////////////////////////////////////////
// Triggered when the connection and the frame encoders are initialized successfully.
void
RemoteRenderAreaListener::onInitializedConnection( std::shared_ptr<RemoteViz::Rendering::RenderArea>,  
                                                   std::shared_ptr<RemoteViz::Rendering::Connection>,  
                                                   std::shared_ptr<const RemoteViz::Rendering::FrameEncoders> )
{
  // Just connected - no data is loaded yet.
  m_dataLoaded = false;
}

///////////////////////////////////////////////////////////////////////////////
// Triggered when a rendering is done. 
void
RemoteRenderAreaListener::onPostRender( std::shared_ptr<RemoteViz::Rendering::RenderArea>  renderArea )
{
  // If data has not been loaded yet, do it now.
  // We wait until after the first render, so the "Loading" message is visible.
  if (! m_dataLoaded) {
    doKeyPress( renderArea, SoKeyboardEvent::Key::D );
    m_dataLoaded = true;
  }
}

///////////////////////////////////////////////////////////////////////////////
// Triggered when a message is received from the client.
// Typically this is a string generated by some JavaScript user interface.
// In this case we implement the request by sending the corresponding key
// press event to the ServiceListener.

void RemoteRenderAreaListener::onReceivedMessage( std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea, 
                                                  std::shared_ptr<RemoteViz::Rendering::Connection> sender, 
                                                  const std::string& message)
{

  if (message == "reset camera")
  {
    doKeyPress( renderArea, SoKeyboardEvent::Key::R );
  }
  else if (message == "reset window")
  {
    doKeyPress( renderArea, SoKeyboardEvent::Key::W );
  }
  else if (message == "reset image#")
  {
    doKeyPress( renderArea, SoKeyboardEvent::Key::I );
  }

  else if (message == "axial")
  {
    doKeyPress( renderArea, SoKeyboardEvent::Key::A );
  }
  else if (message == "coronal")
  {
    doKeyPress( renderArea, SoKeyboardEvent::Key::C );
  }
  else if (message == "sagittal")
  {
    doKeyPress( renderArea, SoKeyboardEvent::Key::S );
  }
}
