///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** DICOM Image Viewer
** Author      : Mike Heck (March 2016)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to render DICOM images with interactive features.
Description : 
This example shows a number of display and interaction techniques combined
to create a (very) basic DICOM image stack viewer.  Features include:

  - DICOM information (patient, study, series, etc) is displayed using
    screen locked annotation text.
  - The image number and number of images are displayed
  - The window level and width values are displayed.
  - Orientation markers are (letters near edge of window) are displayed.
  - Dynamic scale bars are displayed.
  - Mouse buttons
      - In Selection mode (the default):
          - No buttons: Voxel position and value are displayed as mouse moves.
          - Button 1: Image number changes as mouse moves up and down.
          - Button 2: Window level and width change as mouse moves.
          - Wheel   : Changes image number.
      - In Viewing mode (press ESC to toggle):
          - Button 1: Zoom in and out as mouse moves.
          - Button 2: Pan image as mouse moves.
          - Wheel   : Zoom
  - Touch
      - In Selection mode (the default):
          - 1 finger drag : Image number
          - 2 finger drag : Window center/width
          - Long tap      : Switch to viewing mode
      - In Viewing mode (press ESC to toggle):
          - 1 finger drag : Rotate
          - 2 finger drag : Zoom
          - Long tap      : Switch to selection mode
  - Hot keys:
      - A : Switch to Axial (Transverse) view.
      - C : Switch to Coronal view.
      - S : Switch to Sagittal view.

      - I : Reset slice number
      - R : Reset pan and zoom (default view).
      - W : Reset window level and width.

  - By changing the '#define REMOTE_VIZ' macro, this example can be built
    as either a desktop application or a cloud application.
-------------------------------------------------------------------------*/
// Notes:
//   - Basic touch interaction
//     The SceneExaminer node automatically converts 1-finger touch and drag
//     events to mouse button and mouse move events. So we can handle basic
//     touch interaction in the slice views using the onMouseXXXSlice methods.
//
//   - LongTap gesture
//     By default the SceneExaminer class handles this event and toggles between
//     viewing and selection modes.  In this demo, the LongTap event is handled
//     in the onOtherEvent() method and does nothing.  This is because currently a
//     LongTap gesture event can easily be generated by accident.  For example,
//     if the user is rotating the camera with one finger and pauses with finger
//     still on screen, eventually a LongTap event will be generated.
//     That can be very confusing.  Since there is no visible indication that
//     the SceneExaminer is no longer in viewing mode, the user may conclude
//     that the camera is "frozen". --> Currently this demo ignores LongTap.
//
//   - DoubleTap gesture
//     By default the SceneExaminer class handles this event and calls the viewAll()
//     method. --> In this demo, DoubleTap is handled in the onOtherEvent() method
//     and causes the current view to be maximized or minimized.  If you want the
//     behavior of viewAll(), note that calling MedicalHelper::orientView() works
//     better for medical images because it makes the image fill the viewport.
//
//   - Double-Click event
//     It would be convenient to handle this event as equivalent to a Double-Tap.
//     However some touch screen devices automatically convert Double-Tap events
//     to Double-Click events in the driver. This results in the application seeing
//     two events (Double-Click and Double-Tap) for the same user action and the
//     user seeing the view maximize then immediately minimize. Currently this demo
//     handles Double-Tap but ignores Double-Click events (use 'M' to maximize when
//     there is no touch screen).
//
//   - 2-finger gestures
//     In viewing mode (default in the volume view), the behavior is not always
//     intuitive and the application may want to do its own handling.
//     One issue is that the Open Inventor gesture recognizers run simultaneously
//     and the SceneExaminer does not enforce "one gesture at a time". As a result,
//     when 2 fingers are down the SceneExaminer is effectively doing orbit, pan,
//     rotate and zoom at the same time.

///////////////////////////////////////////////////////////////////////////////
// Set REMOTE_VIZ = 0 to build a desktop application.
//                = 1 to build a RemoteViz cloud application
//
// REMOTE_VIZ macro is defined as 1 in the MedicalRemoteMPRService project
///////////////////////////////////////////////////////////////////////////////

#if !defined(REMOTE_VIZ)
#define REMOTE_VIZ 1
#endif

#define ALLOW_DOUBLE_CLICK 0

#if REMOTE_VIZ
#include "RemoteServiceListener.h"
#include "RemoteRenderAreaListener.h"
#include <RemoteViz/Rendering/RenderArea.h>
#include <Inventor/touch/SoTouchManager.h>

#include <IvTune/SoIvTune.h>
#else
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h> //HACK
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>    //HACK
#include <Inventor/errors/SoDebugError.h>
#include <Inventor/Win/SoConsole.h>
#ifdef WIN32
#include <Inventor/touch/devices/SoWinTouchScreen.h>
#include <Inventor/touch/SoTouchManager.h>
#endif
#endif

#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoText3.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/gestures/events/SoScaleGestureEvent.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/details/SoOrthoSliceDetail.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <DialogViz/dialog/SoMessageDialog.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>
#include <Medical/nodes/SliceOrientationMarkers.h>
#include <Medical/nodes/SliceScaleBar.h>

#include <Inventor/ViewerComponents/nodes/SceneExaminer.h>

#include <iostream>

///////////////////////////////////////////////////////////////////////////////

// Data Set
const SbString VOLUME_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles.dcm";
const SbString VOLUME_FILENAME2 = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME   = "$OIVHOME/examples/data/Medical/dicomSample/CVH279.dcm";

///////////////////////////////////////////////////////////////////////////////
// Application state

#if REMOTE_VIZ
// Definition of a RenderArea is different for desktop vs cloud.
static std::shared_ptr<RemoteViz::Rendering::RenderArea> m_renderArea = NULL;
#else
static SoXtRenderArea* m_renderArea  = NULL;
static bool            m_touchScreen = true;
#endif

// Viewer
static SceneExaminer*  m_examiner = NULL;

// Common volume info
static SbVec3i32     m_volDim(0,0,0);
static SbBox3f       m_volExt(0,0,0,0,0,0);
static SbVec3f       m_voxelSize(0,0,0);

// Nodes we might need to query from (or modify) during execution.
static SoVolumeData* m_volData   = NULL;
static SoDataRange*  m_dataRange = NULL;
static SoOrthoSlice* m_sliceNode = NULL;
static SoTransferFunction* m_cmap= NULL;

static SbVec2f m_resetWinCtrWidth(0,0); // Init this when data set is loaded.

// TextBox to display dynamic info and corresponding line numbers.
static TextBox* m_infoDisplay = NULL;
static int m_line_sliceNum = 0;
static int m_line_ctrWidth = 1;
static int m_line_voxValue = 2;

// Slice properties needed for the user interface
static int m_sliceAxis = MedicalHelper::AXIAL;
static int m_sliceNum  = 0;
static int m_numSlices = 0;

// Mouse button state and enable display of voxel values
enum MouseMode {
  MOUSE_NOOP,           // Moving the mouse does nothing
  MOUSE_SHOW_VALUE,     // Show voxel pos/value as mouse moves (default when no button)
  MOUSE_SCROLL_IMAGE,   // Change image number as mouse moves up/down (bound to mouse 1)
  MOUSE_CHANGE_WINCW    // Change window center/level as mouse moves  (bound to mouse 2)
};
static bool      m_2FingerDrag= false;
static bool      m_mouse1Down = false;
static bool      m_mouse2Down = false;
static SbVec2s   m_mousePosition(0,0);
static MouseMode m_mouseMode = MOUSE_SHOW_VALUE;

// Loading message
static SoSwitch* m_messageSwitch = NULL;

///////////////////////////////////////////////////////////////////////////////

static void buildAnnotations( SoGroup* root );

// Event handlers
static void onKeyPress   ( void* data, SoEventCallback* node );
static void onMouseButton( void* data, SoEventCallback* node );
static void onMouseMove  ( void* data, SoEventCallback* node );
static void onMouseWheel ( void* data, SoEventCallback* node );
static void onTouchEvent ( void* data, SoEventCallback* node );

// Change to a different image
static void goToNewImage( int upOrDown );
// Change to a different axis
static void goToNewAxis ( MedicalHelper::Axis axis );

// User interface updaters
static void ui_updateSliceNum( int sliceNum, int numSlices );
static void ui_updateWinCtrWidth( float center, float width );
static void ui_updateVoxelPosVal( const SbVec3i32* ijkPos, float value );

// Sensor to trigger data loading after render area is established
static void sensorCB( void* data, SoSensor* sensor );
static void loadData();
static void traversalCB( void *userData, SoAction *action );

#if ! REMOTE_VIZ
// Catch Open Inventor error reports.
void errorCB( const SoError* error, void *data )
{
  // Print messages except the useless one about touch screen not supported.
  const SbString& msg = error->getDebugString();
  if (msg.contains(SbString("SoWinTouchScreen"))) {
    m_touchScreen = false;
  }
  else {
    SoWin::getErrorConsole()->printMessage( msg );
  }
}
#endif

///////////////////////////////////////////////////////////////////////////////
#if REMOTE_VIZ
// Triggered when a client is connected to the service. */
void RemoteServiceListener::onConnectedClient( const std::string& clientId, std::shared_ptr<RemoteViz::Rendering::NetworkPerformance> )
{
  std::cout << "onConnectedClient id: " << clientId << std::endl;
}

void RemoteServiceListener::onInstantiatedRenderArea( std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea )
{
  std::cout << "\nCreated render area: " << renderArea->getId() << std::endl;

  // Instantiate the class overriding the renderAreaListener class to manage the renderArea events.
  std::shared_ptr<RemoteRenderAreaListener> renderAreaListener(new RemoteRenderAreaListener());

  // Add the renderAreaListener instance as renderArea listener
  renderArea->addListener(renderAreaListener);

  // Add recognizers for gesture events
  // Adds SoScaleGestureRecognizer, SoRotateGestureRecognizer,
  //      SoLongTapGestureRecognizer and SoDoubleTapGestureRecognizer
  // SceneExaminer handles the first two for us.
  renderArea->getTouchManager()->addDefaultRecognizers();

  m_renderArea = renderArea;

#else
void main(int argc, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  SoDialogViz::init();  // For SoMessageDialog
  SoVolumeRendering::init();

  // Create render area
  m_renderArea = new SoXtRenderArea( myWindow );
    m_renderArea->setTitle( "MPR Viewer" );
    m_renderArea->setSize( MedicalHelper::exampleWindowSize() );

#ifdef WIN32
  // Enable touch screen events (if supported) and add default recognizers.
  // - We need an error handler to detect if touch screen is supported.
  // - Default recognizers are SoScaleGestureRecognizer, SoRotateGestureRecognizer
  //   SoLongTapGestureRecognizer and SoDoubleTapGestureRecognizer.
  //   SceneExaminer handles the first two for us.
  SoDebugError::setHandlerCallback( errorCB, (void*)NULL );
  SoWinTouchScreen touchScreenDevice( myWindow );
  if (m_touchScreen) {
    m_renderArea->registerDevice( &touchScreenDevice );
    touchScreenDevice.getTouchManager()->addDefaultRecognizers();
  }
#endif
#endif

  // Initialize Open Inventor medical extensions
  InventorMedical::init();

  // Instructions
  std::cout << "Hot keys:\n"
               "   A : Switch to Axial (Transverse) view.\n"
               "   C : Switch to Coronal view.\n"
               "   S : Switch to Sagittal view.\n"
               "   I : Reset slice number.\n"
               "   R : Reset camera (pan and zoom).\n"
               "   W : Reset window level and width.\n"
               "Selection mode (default):\n"
               "  Mouse 1   / 1 finger : Image number\n"
               "  Mouse 2   / 2 finger : Window center/width\n"
               "  Long tap             : Switch to viewing mode\n"
               "Viewing mode:\n"
               "  Mouse 1   / 1 finger : Orbit\n"
               "  Mouse 2              : Pan\n"
               "  Mouse 1+2 / 2 finger : Zoom\n"
               "  Long tap             : Switch to selection mode\n";

  // Create root of scene graph and assign to render area
  SoRef<SoSeparator> renderRoot = new SoSeparator();
  m_renderArea->getSceneManager()->setSceneGraph( renderRoot.ptr() );

  // Event handling
  // We have to install the key press handler _above_ the SceneExaminer
  // node or we can't get the 'S' key press.  (bug 58247)
  SoEventCallback* eventNode = new SoEventCallback();
    eventNode->addEventCallback( SoKeyboardEvent::getClassTypeId()   , onKeyPress );
    renderRoot->addChild( eventNode );

  // Camera controller for this slice is a "2D" camera.
  // Note that we start in picking/selection mode so the SceneExaminer will *not*
  // respond to events by moving the camera. Instead we use the events to change
  // the image number, window center/width, etc.
  m_examiner = new SceneExaminer();
    m_examiner->setCameraMode( SceneExaminer::ORTHOGRAPHIC );
    m_examiner->setNavigationMode( SceneExaminer::PLANE );
    m_examiner->setInteractionMode(SceneExaminer::SELECTION);
    m_examiner->enableSeek( false );   // Not useful in an image viewer
    m_examiner->enableRotate( false ); // Not useful in an image viewer
    renderRoot->addChild( m_examiner );

  // Create scene
  SoSeparator* root = new SoSeparator();
  m_examiner->addChild( root );

  // Event handling part 2
  // We handle mouse events after the SceneExaminer because we want it
  // to handle these events when in "viewing" mode.
  // Note that the SceneExaminer automatically converts 1-finger touch and drag
  // events to mouse button and mouse move events. So we can handle basic touch
  // interaction using the onMouseXXX methods.
  eventNode = new SoEventCallback();
    eventNode->addEventCallback( SoLocation2Event::getClassTypeId()   , onMouseMove );
    eventNode->addEventCallback( SoMouseWheelEvent::getClassTypeId()  , onMouseWheel );
    eventNode->addEventCallback( SoMouseButtonEvent::getClassTypeId() , onMouseButton );
    eventNode->addEventCallback( SoScaleGestureEvent::getClassTypeId(), onTouchEvent );
    root->addChild( eventNode );

  //---------------------------------------------------------------------------
  // Display loading message initially
  m_messageSwitch = new SoSwitch();
    m_messageSwitch->whichChild = 0;   // Initially visible
    root->addChild( m_messageSwitch );

  TextBox* loadingMessage = new TextBox();
    loadingMessage->position.setValue( 0,0,0 );
    loadingMessage->alignmentH = TextBox::AlignmentH::CENTER;
    loadingMessage->alignmentV = TextBox::AlignmentV::MIDDLE;
    loadingMessage->addLine( "LOADING..." );
    loadingMessage->getFontNode()->size = 32;
    m_messageSwitch->addChild( loadingMessage );

  // Volume visualization stuff
  SoSeparator* volSep = new SoSeparator();
    m_messageSwitch->addChild( volSep );

    // Data node
    SoVolumeData* volData = new SoVolumeData();
      //volData->setReader( *volReader );
      volData->fileName = IMAGE_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild(volData);

    // Data range (from data file by default)
    SoDataRange* dataRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( dataRange, volData );
      volSep->addChild( dataRange );

    // Base material.
    // By default Open Inventor uses gray 0.8 to leave room for lighting to
    // brighten the image. For slice viewing it's better to use full intensity.
    SoMaterial* volMatl = new SoMaterial();
      volMatl->diffuseColor.setValue( 1, 1, 1 );
      volSep->addChild( volMatl );

    // Transfer function (gray scale)
    SoTransferFunction* transFunc = new SoTransferFunction();
      transFunc->predefColorMap = SoTransferFunction::INTENSITY;
      MedicalHelper::dicomCheckMonochrome1( transFunc, volData );
      volSep->addChild( transFunc );

    // Otho slice rendering
    // If we loaded a volume (more than 1 slice) then pick an "interesting"
    // slice in the middle for demoing.  Else we may have loaded a single image.
    int numSlices = volData->data.getSize()[MedicalHelper::AXIAL];
    SoOrthoSlice* orthoSlice = new SoOrthoSlice();
      orthoSlice->axis          = MedicalHelper::AXIAL;
      orthoSlice->sliceNumber   = (numSlices > 1) ? (numSlices / 2) : 0;
      orthoSlice->interpolation = SoSlice::MULTISAMPLE_12;
      volSep->addChild( orthoSlice );

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Slice orientation markers.
    // Connect the axis field to the slice so the node will update automatically.
    SoSeparator* markerSep = new SoSeparator();
      root->addChild( markerSep );
      SoMaterial* sorientMat = new SoMaterial();  // Color for slice orientation markers
        sorientMat->diffuseColor.setValue( 1, 1, 0.25f );
        markerSep->addChild( sorientMat );
      SliceOrientationMarkers* sliceOrient = new SliceOrientationMarkers();
        sliceOrient->axis.connectFrom( &orthoSlice->axis );
        markerSep->addChild( sliceOrient );

    // DICOM annotation (including our image number display and so on).
    SoSwitch* dicomAnnoSwitch = new SoSwitch(); // Allows to toggle annotation visibility
      dicomAnnoSwitch->whichChild = SO_SWITCH_ALL;
      root->addChild( dicomAnnoSwitch );
      buildAnnotations( dicomAnnoSwitch );

    // Dynamic scale bars
    SoSeparator* scaleSep = new SoSeparator();
      root->addChild( scaleSep );
      SoMaterial* scaleColor = new SoMaterial();
        scaleColor->diffuseColor.setValue( 1, 0.25f, 0.25f );
        scaleSep->addChild( scaleColor );
      SoDrawStyle* scaleStyle = new SoDrawStyle();
        scaleStyle->lineWidth = 2;
        scaleSep->addChild( scaleStyle );
      SliceScaleBar* scale1 = new SliceScaleBar(); // Horizontal
        scale1->position.setValue( 0, -0.99f );
        scale1->length = 100; // 10 cm
        scale1->trackedCamera = m_examiner->getCamera();
        scale1->label = "10cm";
        scaleSep->addChild( scale1 );
      SliceScaleBar* scale2 = new SliceScaleBar(); // Vertical
        scale2->position.setValue( -0.99f, 0 );
        scale2->length = 100; // 10 cm
        scale2->orientation = SliceScaleBar::VERTICAL;
        scale2->trackedCamera = m_examiner->getCamera();
        scale2->label = "10cm";
        scaleSep->addChild( scale2 );

#if REMOTE_VIZ

  MedicalHelper::orientView( MedicalHelper::Axis::AXIAL, m_examiner->getCamera(), volData );
  m_examiner->getCameraInteractor()->pushCamera();            // Save default view

  // Set up variables for UI
  m_volData   = volData;
  m_dataRange = dataRange;
  m_sliceNode = orthoSlice;
  m_cmap      = transFunc;
  m_sliceAxis = MedicalHelper::AXIAL;
  m_sliceNum  = m_sliceNode->sliceNumber.getValue();
  m_numSlices = volData->data.getSize()[m_sliceAxis];
  m_resetWinCtrWidth = MedicalHelper::dicomGetWindowCenterWidth( m_dataRange );
  ui_updateSliceNum( m_sliceNum, m_numSlices );
  ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );

#else
    // Set up viewer.
    SoXtPlaneViewer *viewer = new SoXtPlaneViewer(myWindow);
    viewer->setDecoration( FALSE );
    viewer->setViewing( FALSE );
    viewer->setTitle( "DICOM Image Viewer" );
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    viewer->setBackgroundColor( SbColor(0,0.1f,0.1f) );
    viewer->setSceneGraph(root.ptr());

    // Initialize the camera view
    MedicalHelper::orientView( MedicalHelper::AXIAL, viewer->getCamera(), volData );
    viewer->saveHomePosition();
    viewer->show();

    // Set up variables for UI
    m_viewer    = viewer;
    m_volData   = volData;
    m_dataRange = dataRange;
    m_sliceNode = orthoSlice;
    m_sliceAxis = MedicalHelper::AXIAL;
    m_sliceNum  = m_sliceNode->sliceNumber.getValue();
    m_numSlices = volData->data.getSize()[m_sliceAxis];
    m_resetWinCtrWidth = MedicalHelper::dicomGetWindowCenterWidth( m_dataRange );
    ui_updateSliceNum( m_sliceNum, m_numSlices );
    ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );

    SoXt::show(myWindow);
    SoXt::mainLoop();

    // Clean up.
    //delete viewer;
    renderRoot = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
#endif
}

///////////////////////////////////////////////////////////////////////////////
// Handle mouse button events
// -- Affects what happens when the mouse cursor is moved.
// -- SceneExaminer automatically converts 1-finger down/up touch events to
//    mouse button down/up events.  So we are also handling touch here.

void onMouseButton( void* data, SoEventCallback* node )
{
  const SoMouseButtonEvent* evt = (const SoMouseButtonEvent*)node->getEvent();

  if (SoMouseButtonEvent::isButtonPressEvent(evt,SoMouseButtonEvent::BUTTON1)) {
    m_mouse1Down = true;
    if (! m_mouse2Down) // Don't change if we're already in a non-default mode.
      m_mouseMode = MOUSE_SCROLL_IMAGE;
  }
  else if (SoMouseButtonEvent::isButtonReleaseEvent(evt,SoMouseButtonEvent::BUTTON1)) {
    m_mouse1Down = false;
    if (! m_mouse2Down)
      m_mouseMode = MOUSE_SHOW_VALUE; // Default when no button pressed
  }
  else if (SoMouseButtonEvent::isButtonPressEvent(evt,SoMouseButtonEvent::BUTTON2)) {
    m_mouse2Down = true;
    if (! m_mouse1Down)
      m_mouseMode = MOUSE_CHANGE_WINCW;
  }
  else if (SoMouseButtonEvent::isButtonReleaseEvent(evt,SoMouseButtonEvent::BUTTON2)) {
    m_mouse2Down = false;
    if (! m_mouse1Down)
      m_mouseMode = MOUSE_SHOW_VALUE; // Default when no button pressed
  }
  else if (SoMouseButtonEvent::isButtonPressEvent(evt,SoMouseButtonEvent::BUTTON3)) {
    node->setHandled();
  }
  else if (SoMouseButtonEvent::isButtonReleaseEvent(evt,SoMouseButtonEvent::BUTTON3)) {
    node->setHandled();
  }
  m_mousePosition = evt->getPosition();
}

///////////////////////////////////////////////////////////////////////////////
// Handle mouse move events
// Behavior depends on which mouse buttons are pressed (if any).

void onMouseMove ( void* data, SoEventCallback* node )
{
  // If user is dragging, that should be handled in onTouchEvent_Slice.
  if (m_2FingerDrag)
    return;

  const SoLocation2Event* evt = (const SoLocation2Event*)node->getEvent();

  // Check what mode we are in.
  if (m_mouseMode == MOUSE_SHOW_VALUE) {
    //---------------------------------------------------------------
    // Check if the cursor is over any data objects.
    const SoPickedPoint* pickedPt = node->getPickedPoint();
    if (pickedPt != NULL) {
      // The cursor is over something interesting (probably the ortho slice
      // because we set all the annotation to be unpickable).
      const SoOrthoSliceDetail* detail =
        dynamic_cast<const SoOrthoSliceDetail*>(pickedPt->getDetail());
      if (detail != NULL) {
        // Get picked voxel
        const SbVec3i32& ijkPos = detail->getValueDataPos();
        // Get value of voxel
        const float value = (float)detail->getValueD();
        // Update the user interface
        ui_updateVoxelPosVal( &ijkPos, value );
        return;
      }
    }
    ui_updateVoxelPosVal( NULL, 0 );
  }
  else if (m_mouseMode == MOUSE_SCROLL_IMAGE) {
    //---------------------------------------------------------------
    SbVec2s newPos = evt->getPosition();
    int delta = newPos[1] - m_mousePosition[1];
    // Sometimes we get a stream of mouse move events with the same event position.
    // This is probably a glitch in the mouse and/or Windows. Ignore this case.
    if (delta != 0)
      goToNewImage( delta );
  }
  else if (m_mouseMode == MOUSE_CHANGE_WINCW) {
    //---------------------------------------------------------------
    SbVec2s newPos = evt->getPosition();
    int deltaX = newPos[0] - m_mousePosition[0];
    int deltaY = newPos[1] - m_mousePosition[1];
    SbVec2f winCW = MedicalHelper::dicomGetWindowCenterWidth( m_dataRange );
    winCW[0] += deltaY;
    winCW[1] += deltaX;
    MedicalHelper::dicomSetWindowCenterWidth( m_dataRange, winCW );
    ui_updateWinCtrWidth( winCW[0], winCW[1] );
  }

  // Remember new position
  m_mousePosition = evt->getPosition();
}

///////////////////////////////////////////////////////////////////////////////
// Handle mouse wheel events --> Change the image number.
// For consistency with other applications:
//   - Mouse wheel forward decreases image number.
//   - Mouse wheel backward increases image number.

void onMouseWheel( void* data, SoEventCallback* node )
{
  const SoMouseWheelEvent* evt = (SoMouseWheelEvent*)node->getEvent();
  int delta = evt->getDelta();
  if (delta != 0)
    goToNewImage( delta );
}

///////////////////////////////////////////////////////////////////////////////
// Handle key press events.
//
// Note:
//   - When slice axis changes the SliceOrientation object will be updated
//     automatically because its field is connected from the slice's field.
void
onKeyPress( void* data, SoEventCallback* node )
{
  const SoKeyboardEvent* evt = (SoKeyboardEvent*)node->getEvent();

  //-----------------------------------------------------------------
  // A : Axial view
  if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::A )) {
    goToNewAxis( MedicalHelper::AXIAL );
  }
  //-----------------------------------------------------------------
  // C : Coronal view
  else if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::C )) {
    goToNewAxis( MedicalHelper::CORONAL );
  }
  // ---------- 'D'ata loading ------------------------------------------------
  // We just use this in the RemoteViz configuration as as easy way for the
  // RenderAreaListener to start data loading after the initial rendering.
  // In that configuration the SoIdleSensor technique doesn't work reliably.
  else if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::Key::D )) {
    loadData();
    node->setHandled();
  }
  //-----------------------------------------------------------------
  // I : Reset Image number
  else if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::I )) {
    // Reset slice node to center of current axis
    m_sliceNum = m_volDim[m_sliceAxis] / 2;
    m_sliceNode->sliceNumber = m_sliceNum;
    // Update image number display
    ui_updateSliceNum( m_sliceNum, m_numSlices );
    // This (temporarily) invalidates the voxel value display
    ui_updateVoxelPosVal( NULL, 0 );
  }
  //-----------------------------------------------------------------
  // R : Home (reset view)
  else if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::R )) {
    m_examiner->getCameraInteractor()->popCamera();   // Restore default view
    m_examiner->getCameraInteractor()->pushCamera();  // Re-save default view
  }
  //-----------------------------------------------------------------
  // S : Sagittal view
  else if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::S )) {
    std::cout << "Pressed 'S'\n";
    goToNewAxis( MedicalHelper::SAGITTAL );
  }
  //-----------------------------------------------------------------
  // W : Reset window center/width to original values
  else if (SoKeyboardEvent::isKeyPressEvent( evt, SoKeyboardEvent::W )) {
    MedicalHelper::dicomSetWindowCenterWidth( m_dataRange, m_resetWinCtrWidth );
    ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );
  }

#if REMOTE_VIZ
  // ---------- Shift-F12 --------------------------------
  // RemoteViz does not automatically start IvTune when Shift-F12 is pressed
  // but this can be very convenient for debugging.
  else if (SoKeyboardEvent::isKeyPressEvent(evt,SoKeyboardEvent::Key::F12)) {
    if (evt->wasShiftDown()) {
      SoNode* root = m_renderArea->getSceneManager()->getSceneGraph();
      if (root != NULL) {
        SoIvTune::start( root );
      }
      node->setHandled();
    }
  }
#endif
}

///////////////////////////////////////////////////////////////////////////////
// Handle touch events for slices
// -- Affects what happens when the mouse cursor is moved.
// -- SceneExaminer automatically converts 1-finger down/up touch events to
//    mouse button down/up events. Handled in onMouseButton_Slice.

void onTouchEvent( void* data, SoEventCallback* node )
{
  const SoEvent* theEvent = node->getEvent();

  // --------- 2-Finger Drag event ----------
  // Open Inventor doesn't currently provide a 2-finger-drag gesture.
  // We could easily write a recognizer for this simple gesture, but since we
  // don't need an actual scale gesture in the slice views, we can use the
  // this event to tell us when two fingers are down.
  if (theEvent->isOfType(SoScaleGestureEvent::getClassTypeId())) {
    const SoGestureEvent* gesEvent = (SoGestureEvent*)theEvent;
    SoGestureEvent::GestureState state = gesEvent->getGestureState();
    if (state == SoGestureEvent::GestureState::BEGIN) {
      m_2FingerDrag = true;     // Two fingers are down
      m_mouseMode = MOUSE_NOOP; // Don't handle mouseMove for now
    }
    else if (state == SoGestureEvent::GestureState::END) {
      m_2FingerDrag = false;  // Fingers are up
    }
    else {
      // Drag - Same code as onMouseMove_Slice
      SbVec2s newPos = theEvent->getPosition();
      int deltaX = newPos[0] - m_mousePosition[0];
      int deltaY = newPos[1] - m_mousePosition[1];
      SbVec2f winCW = MedicalHelper::dicomGetWindowCenterWidth( m_dataRange );
      winCW[0] += deltaY;
      winCW[1] += deltaX;
      MedicalHelper::dicomSetWindowCenterWidth( m_dataRange, winCW );
      ui_updateWinCtrWidth( winCW[0], winCW[1] );
    }
    // Remember new position
    m_mousePosition = theEvent->getPosition();
    node->setHandled();
  }
}

///////////////////////////////////////////////////////////////////////////////
void
buildAnnotations( SoGroup* root )
{
  SoMaterial* dicomMatl = new SoMaterial();
    dicomMatl->diffuseColor.setValue( 0.8f, 0.8f, 0.5f );
    root->addChild( dicomMatl );

  DicomInfo* upperLeft = new DicomInfo();
    upperLeft->fileName = IMAGE_FILENAME;
    upperLeft->position.setValue( -0.99f, 0.99f, 0 );
    upperLeft->alignmentV = DicomInfo::TOP;
    upperLeft->displayDicomInfo( "", 0x0010, 0x0010 ); // Patient Name
    upperLeft->displayDicomInfo( "", 0x0010, 0x0030 ); // Patient Birth Date
    upperLeft->displayDicomInfo( "", 0x0008, 0x1030 ); // Study Description
    upperLeft->displayDicomInfo( "", 0x0008, 0x103E ); // Series Description
    root->addChild( upperLeft );

  DicomInfo* upperRight = new DicomInfo();
    upperRight->fileName = IMAGE_FILENAME;
    upperRight->position.setValue( 0.99f, 0.99f, 0 );
    upperRight->alignmentH = DicomInfo::RIGHT;
    upperRight->alignmentV = DicomInfo::TOP;
    upperRight->textAlignH = DicomInfo::RIGHT;
    upperRight->displayDicomInfo( "", 0x0008, 0x0080 ); // Institution
    upperRight->displayDicomInfo( "", 0x0008, 0x0090 ); // Physician
    upperRight->displayDicomInfo( "", 0x0008, 0x1090 ); // Model name
    root->addChild( upperRight );

  DicomInfo* lowerRight = new DicomInfo();
    lowerRight->fileName = IMAGE_FILENAME;
    lowerRight->position.setValue( 0.99f, -0.99f, 0 );
    lowerRight->alignmentH = DicomInfo::RIGHT;
    lowerRight->alignmentV = DicomInfo::BOTTOM;
    lowerRight->textAlignH = DicomInfo::RIGHT;
    lowerRight->displayDicomInfo( ""    , 0x0008, 0x0060 ); // Modality
    lowerRight->displayDicomInfo( "mA: ", 0x0018, 0x1151 ); // X-Ray Tube Current
    lowerRight->displayDicomInfo( "kV: ", 0x0018, 0x0060 ); // KVP (Kilo Voltage Peak)
    lowerRight->displayDicomInfo( ""    , 0x0008, 0x0022 ); // Acquisition date
    root->addChild( lowerRight );

  // This next TextBox is where we'll display dynamic info like image number,
  // voxel value, etc.  So give it a different color to be more visible.
  SoMaterial* dynInfoMatl = new SoMaterial();
    dynInfoMatl->diffuseColor.setValue( 0.8f, 0.5f, 0.5f );
    root->addChild( dynInfoMatl );

  TextBox* lowerLeft = new TextBox();
    lowerLeft->fontSize = 17; // Slightly larger than default
    lowerLeft->position.setValue( -0.99f, -0.94f, 0 ); // Leave room for OIV logo
    lowerLeft->alignmentV = TextBox::BOTTOM;
    lowerLeft->addLine( "" ); // Placeholder for image number
    lowerLeft->addLine( "" ); // Placeholder for window center/width
    lowerLeft->addLine( "" ); // Placeholder for voxel pos/value
    root->addChild( lowerLeft );

  // For user interface updates
  m_infoDisplay = lowerLeft;
}

///////////////////////////////////////////////////////////////////////////////
// Change to the next image in stack.
// If delta is negative, increment the image number.
// Else decrement the image number.
void
goToNewImage( int delta )
{
  if (delta < 0) {
    if (m_sliceNum < (m_numSlices-1)) {
      m_sliceNum++;
      m_sliceNode->sliceNumber = m_sliceNum;
    }
  }
  else {
    if (m_sliceNum > 0) {
      m_sliceNum = m_sliceNum--;
      m_sliceNode->sliceNumber = m_sliceNum;
    }
  }
  // Update image number display
  ui_updateSliceNum( m_sliceNum, m_numSlices );
  // This (temporarily) invalidates the voxel value display
  ui_updateVoxelPosVal( NULL, 0 );
}

///////////////////////////////////////////////////////////////////////////////
// Change to a different axis
void goToNewAxis ( MedicalHelper::Axis axis )
{
  // If we only have 1 slice there is no way to view from other axes. :-)
  if (m_numSlices == 1) {
    std::cerr << "Only 1 slice loaded. Not possible to view other axes.\n";
    return;
  }
  // In each case:
  //   - Remember the new axis.
  //   - Flip the slice to the new axis.
  //   - Orient the camera to the new axis and save new home position.
  //   - Force slice number to be valid for the new axis.
  if (axis == MedicalHelper::AXIAL) {
    m_sliceAxis = MedicalHelper::AXIAL;
    m_sliceNode->axis = m_sliceAxis;
    m_examiner->getCameraInteractor()->popCamera();   // Restore default view
    MedicalHelper::orientView( MedicalHelper::AXIAL, m_examiner->getCamera(), m_volData );
    m_examiner->getCameraInteractor()->pushCamera();  // Re-save default view
    m_numSlices = m_volData->data.getSize()[m_sliceAxis];
    if (m_sliceNum > (m_numSlices-1))
      m_sliceNum = m_numSlices - 1;
  }
  else if (axis == MedicalHelper::CORONAL) {
    m_sliceAxis = MedicalHelper::CORONAL;
    m_sliceNode->axis = m_sliceAxis;
    m_examiner->getCameraInteractor()->popCamera();   // Restore default view
    MedicalHelper::orientView( MedicalHelper::CORONAL, m_examiner->getCamera(), m_volData );
    m_examiner->getCameraInteractor()->pushCamera();  // Re-save default view
    m_numSlices = m_volData->data.getSize()[m_sliceAxis];
    if (m_sliceNum > (m_numSlices-1))
      m_sliceNum = m_numSlices - 1;
  }
  else if (axis == MedicalHelper::SAGITTAL) {
    m_sliceAxis = MedicalHelper::SAGITTAL;
    m_sliceNode->axis = m_sliceAxis;
    m_examiner->getCameraInteractor()->popCamera();   // Restore default view
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, m_examiner->getCamera(), m_volData );
    m_examiner->getCameraInteractor()->pushCamera();  // Re-save default view
    m_numSlices = m_volData->data.getSize()[m_sliceAxis];
    if (m_sliceNum > (m_numSlices-1))
      m_sliceNum = m_numSlices - 1;
  }

  // Update slice number display and invalidate voxel value display.
  ui_updateSliceNum( m_sliceNum, m_numSlices );
  ui_updateVoxelPosVal( NULL, 0 );
}

///////////////////////////////////////////////////////////////////////////////
// Update wherever the UI displays the slice number.
// Remember that OIV numbers slices starting at 0, but medical applications
// typically display the image number starting at 1.
void
ui_updateSliceNum( int sliceNum, int numSlices )
{
  SbString str;
  str.sprintf( "Image  %d  /  %d", sliceNum + 1, numSlices );
  m_infoDisplay->setLine( str, m_line_sliceNum );
}

///////////////////////////////////////////////////////////////////////////////
// Update wherever the UI displays the window center/width.
void
ui_updateWinCtrWidth( float center, float width )
{
  SbString str;
  str.sprintf( "WL: %g  WW: %g", center, width );
  m_infoDisplay->setLine( str, m_line_ctrWidth );
}

///////////////////////////////////////////////////////////////////////////////
// Update wherever the UI displays the voxel position and value.
// If values are not valid, clear the display.
void
ui_updateVoxelPosVal( const SbVec3i32* ijkPos, float value )
{
  SbString str("");
  if (ijkPos != NULL) {
    // We have valid values, but which voxel coords are relevant depends on
    // the current slice orientation (Axial, Coronal, Sagittal).
    int i = 0; // Valid for Axial case
    int j = 1;
    if (m_sliceAxis == MedicalHelper::CORONAL) {
      i = 0; // X
      j = 2; // Z
    }
    else if (m_sliceAxis == MedicalHelper::SAGITTAL) {
      i = 1; // Y
      j = 2; // Z
    }
    str.sprintf( "Pos: %d , %d  Val: %g", (*ijkPos)[i], (*ijkPos)[j], value );
  }
  m_infoDisplay->setLine( str, m_line_voxValue );
}

///////////////////////////////////////////////////////////////////////////////
// Should be called after the render area is established.
// This allows us to display messages to the user while we're busy.
void sensorCB( void* data, SoSensor* sensor )
{
  loadData();
}

///////////////////////////////////////////////////////////////////////////////
// Load the image stack.
// Should be called after the render area is established.
// This allows us to display messages to the user while we're busy.
void loadData()
{
  std::cout << "Loading...\n";

  // Check that we can access the data
  SbBool useShortList = SoPreferences::getBool( "DICOM_SHORT_LIST", FALSE );
  SbString filename = useShortList ? VOLUME_FILENAME2 : VOLUME_FILENAME;
  if (! SbFileHelper::isAccessible( filename )) {
    filename = VOLUME_FILENAME;
    if (! SbFileHelper::isAccessible( filename )) {
      SoMessageDialog* dialog = new SoMessageDialog( filename, "Unable to open:", SoMessageDialog::MD_ERROR );
      return;
    }
  }

  // Global initialization -- Load the volume data
  m_volData->fileName = filename;
  MedicalHelper::dicomAdjustVolume( m_volData );

  // Remember volume characteristics for user interface updates.
  // Note that volume extent min is the outer edge of the first voxel,
  // DICOM origin is the _center_ of the first voxel (difference of 1/2 voxel).
  m_volDim = m_volData->data.getSize();
  m_volExt = m_volData->extent.getValue();
  SbVec3f volSize = m_volExt.getSize();
  m_voxelSize.setValue( volSize[0]/m_volDim[0], volSize[1]/m_volDim[1], volSize[2]/m_volDim[2] );
  SbVec3f dicomOrigin = m_volExt.getMin() + (0.5f * m_voxelSize);
  double volMin, volMax;
  m_volData->getMinMax( volMin, volMax );

  // Set data range for slices from the volume's window center/width.
  // Save that initial center/width and update the UI.
  MedicalHelper::dicomAdjustDataRange( m_dataRange, m_volData );
  m_resetWinCtrWidth = MedicalHelper::dicomGetWindowCenterWidth( m_dataRange );
  ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );

  std::cout << "\nVolume dims: " << m_volDim << std::endl;
  std::cout << "       extent: " << m_volExt.getMin() << " " << m_volExt.getMax() << std::endl;
  std::cout << "  DICOMorigin: " << dicomOrigin << std::endl;
  std::cout << "     PhysSize: " << volSize << std::endl;
  std::cout << "    voxelSize: " << m_voxelSize << std::endl;
  std::cout << "  scalarRange: " << volMin << " to " << volMax << std::endl;
  std::cout << "  windowCtrWid " << m_resetWinCtrWidth[0] << " / " << m_resetWinCtrWidth[1] << std::endl;

  // Slice rendering color map.
  // Usually INTENSITY (black to white), but check for MONOCHROME1 case.
  m_cmap->predefColorMap = SoTransferFunction::PredefColorMap::INTENSITY;
  MedicalHelper::dicomCheckMonochrome1( m_cmap, m_volData );

  // View initialization...
  m_examiner->getCameraInteractor()->popCamera();
  MedicalHelper::orientView( MedicalHelper::Axis::AXIAL, m_examiner->getCamera(), m_volData );
  m_examiner->getCameraInteractor()->pushCamera();

  // UI initialization...
  m_sliceAxis = MedicalHelper::AXIAL;
  m_sliceNode->sliceNumber = m_volDim[m_sliceAxis] / 2;
  m_sliceNum  = m_sliceNode->sliceNumber.getValue();
  m_numSlices = m_volData->data.getSize()[m_sliceAxis];

  m_resetWinCtrWidth = MedicalHelper::dicomGetWindowCenterWidth( m_dataRange );
  ui_updateSliceNum( m_sliceNum, m_numSlices );
  ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );

  // Finally, disable the loading message
  m_messageSwitch->whichChild = 1; // Disabled
}
