///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** DICOM MPR Viewer
** Author      : Mike Heck
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to render DICOM data in classical four views.

Description : 
This example shows a number of display and interaction techniques combined
to create a very basic DICOM MPR viewer.  Features include:

  - DICOM information (patient, study, series, etc) is displayed using
    screen locked annotation text.
  - The image number and number of images are displayed
  - The window level and width values are displayed.
  - Orientation markers are (letters near edge of window) are displayed.
  - The camera and window level/width for all slice views are synchronized.
  - Hot keys:
      - I : Reset image numbers (slice views)
      - L : Toggle move-as-low-res (volume view)
      - M : Maximize     (current view)
      - R : Reset camera (all views)
      - W : Reset window center/width (slice views)
      - 1 : 2x2 square layout
      - 2 : 1x3 side layout
      - 3 : 1x3 below layout
  - Mouse buttons
      - In Volume View (viewing mode by default):
          - Button 1  : Rotate
          - Button 2  : Pan
          - Button 1+2: Zoom
          - Wheel     : Zoom
      - In Slice Views (selection mode by default):
          - Button 1: Image number
          - Button 2: Window center/width
          - Wheel   : Image number
  - Touch
      - In Volume View (viewing mode by default):
          - 1 finger drag : Rotate
          - 2 finger drag : Zoom
          - Double tap    : Maximize
      - In Slice View (selection mode by default):
          - 1 finger drag : Image number
          - 2 finger drag : Window center/width
          - Double tap    : Maximize

  - By changing the '#define REMOTE_VIZ' macro, this example can be built
    as either a desktop application or a cloud application.
-------------------------------------------------------------------------*/

// Notes:
//   - Basic touch interaction
//     The SceneExaminer node automatically converts 1-finger touch and drag
//     events to mouse button and mouse move events. So we can handle basic
//     touch interaction in the slice views using the onMouseXXXSlice methods.
//
//   - LongTap gesture
//     By default the SceneExaminer class handles this event and toggles between
//     viewing and selection modes.  In this demo, the LongTap event is handled
//     in the onOtherEvent() method and does nothing.  This is because currently a
//     LongTap gesture event can easily be generated by accident.  For example,
//     if the user is rotating the camera with one finger and pauses with finger
//     still on screen, eventually a LongTap event will be generated.
//     That can be very confusing.  Since there is no visible indication that
//     the SceneExaminer is no longer in viewing mode, the user may conclude
//     that the camera is "frozen". --> Currently this demo ignores LongTap.
//
//   - DoubleTap gesture
//     By default the SceneExaminer class handles this event and calls the viewAll()
//     method. --> In this demo, DoubleTap is handled in the onOtherEvent() method
//     and causes the current view to be maximized or minimized.  If you want the
//     behavior of viewAll(), note that calling MedicalHelper::orientView() works
//     better for medical images because it makes the image fill the viewport.
//
//   - Double-Click event
//     It would be convenient to handle this event as equivalent to a Double-Tap.
//     However some touch screen devices automatically convert Double-Tap events
//     to Double-Click events in the driver. This results in the application seeing
//     two events (Double-Click and Double-Tap) for the same user action and the
//     user seeing the view maximize then immediately minimize. Currently this demo
//     handles Double-Tap but ignores Double-Click events (use 'M' to maximize when
//     there is no touch screen).
//
//   - 2-finger gestures
//     In viewing mode (default in the volume view), the behavior is not always
//     intuitive and the application may want to do its own handling.
//     One issue is that the Open Inventor gesture recognizers run simultaneously
//     and the SceneExaminer does not enforce "one gesture at a time". As a result,
//     when 2 fingers are down the SceneExaminer is effectively doing orbit, pan,
//     rotate and zoom at the same time.

///////////////////////////////////////////////////////////////////////////////
// Set REMOTE_VIZ = 0 to build a desktop application.
//                = 1 to build a RemoteViz cloud application
//
// REMOTE_VIZ macro is defined as 1 in the MedicalRemoteMPRService project
///////////////////////////////////////////////////////////////////////////////

#if !defined(REMOTE_VIZ)
#define REMOTE_VIZ 1
#endif

#define ALLOW_DOUBLE_CLICK 0

#if REMOTE_VIZ
#include "RemoteMPRServiceListener.h"
#include "RemoteMPRRenderAreaListener.h"
#include <RemoteViz/Rendering/RenderArea.h>
#include <Inventor/touch/SoTouchManager.h>

#include <IvTune/SoIvTune.h>
#else
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/errors/SoDebugError.h>
#include <Inventor/Win/SoConsole.h>
#ifdef WIN32
#include <Inventor/touch/devices/SoWinTouchScreen.h>
#include <Inventor/touch/SoTouchManager.h>
#endif
#endif

#include <Inventor/nodes/SoBBox.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoViewport.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoMouseWheelEvent.h>

#include <Inventor/touch/events/SoTouchEvent.h>
#include <Inventor/gestures/events/SoDoubleTapGestureEvent.h>
#include <Inventor/gestures/events/SoLongTapGestureEvent.h>
#include <Inventor/gestures/events/SoRotateGestureEvent.h>
#include <Inventor/gestures/events/SoScaleGestureEvent.h>

#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/sensors/SoIdleSensor.h>
#include <Inventor/sensors/SoOneShotSensor.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoROI.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/details/SoOrthoSliceDetail.h>

#include <DialogViz/dialog/SoMessageDialog.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/SceneView.h>
#include <Medical/nodes/SliceOrientationMarkers.h>
#include <Medical/nodes/SliceScaleBar.h>
#include <Medical/nodes/TextBox.h>
#include <Medical/nodes/ViewManager.h>

#include "MySceneExaminer.h"  // SceneExaminer with a bug fix missing in OIV 9.7.0
#define SceneExaminer MySceneExaminer

#include <iostream>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles.dcm";
const SbString VOLUME_FILENAME2 = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME   = "$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm";
const SbString COLORMAPFILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

// Common volume info
static SoVolumeData* m_volData = NULL;
static SbVec3i32     m_volDim(0,0,0);
static SbBox3f       m_volExt(0,0,0,0,0,0);
static SbVec3f       m_voxelSize(0,0,0);

// Volume rendering info
static SoDataRange*              m_volRange = NULL;
static SoTransferFunction*       m_volCmap  = NULL;
static SoVolumeRenderingQuality* m_volQual  = NULL;
static SoVolumeRender*           m_volRend  = NULL;

// Slice rendering info (all slices share same data range etc)
static SoDataRange*        m_sliceRange = NULL;
static SoTransferFunction* m_sliceCmap  = NULL;
static SoMaterial*         m_sliceMatl  = NULL;
static SoOrthoSlice*       m_sliceNodes[3] = {NULL,NULL,NULL};
static SoSwitch*           m_sliceSBars[3] = {NULL,NULL,NULL};

// View info (the child of each view is a SceneExaminer)
static ViewManager* m_viewManager = NULL;
static SceneView*   m_views[4]    = {NULL,NULL,NULL,NULL};
static MedicalHelper::Axis ViewAxes[] = { // Orientation axis for each view
    MedicalHelper::Axis::CORONAL,
    MedicalHelper::Axis::SAGITTAL,
    MedicalHelper::Axis::CORONAL,
    MedicalHelper::Axis::AXIAL  };
static float Viewports[][4] = {    // Default viewport for each view (x,y,w,h)
    {0   , 0.5f, 0.5f, 0.5f},      // Upper left
    {0.5f, 0.5f, 0.5f, 0.5f},      // Upper right
    {0   , 0   , 0.5f, 0.5f},      // Lower left
    {0.5f, 0   , 0.5f, 0.5f}       // Lower right
};

#if REMOTE_VIZ
// Definition of a RenderArea is different for desktop vs cloud.
static std::shared_ptr<RemoteViz::Rendering::RenderArea> m_renderArea = NULL;
#else
static SoXtRenderArea* m_renderArea  = NULL;
static bool            m_touchScreen = true;
#endif

// User interface state
static TextBox* m_sliceTextUL[3];
static TextBox* m_sliceTextLL[3];
static int      m_maximizedViewIndex = -1; // Remember which view is maximized
static SbVec4f  m_maximizedViewSave;       // Remember orig/size of maximized view
static SbVec2f  m_resetWinCtrWidth(0,0);   // Init this when data set is loaded.

// Mouse button state and enable display of voxel values
enum MouseMode {
  MOUSE_NOOP,           // Moving the mouse does nothing
  MOUSE_SHOW_VALUE,     // Show voxel pos/value as mouse moves (default when no button)
  MOUSE_SCROLL_IMAGE,   // Change image number as mouse moves up/down (bound to mouse 1)
  MOUSE_CHANGE_WINCW    // Change window center/level as mouse moves  (bound to mouse 2)
};
static bool      m_mouse1Down = false;
static bool      m_mouse2Down = false;
static bool      m_2FingerDrag= false;
static SbVec2s   m_mousePosition(0,0);
static MouseMode m_mouseMode = MOUSE_SHOW_VALUE;

// Interactive mode tracking
static SoInteractiveComplexity* m_interactNode = NULL; // We use this to set SoInteractionElement
static bool      m_useLowRes   = true;  // Use low-res for volRend when interacting
static bool      m_interacting = false; // Work around for SoInteractionElement

// Loading message
static SoSwitch* m_messageSwitch = NULL;

///////////////////////////////////////////////////////////////////////////////

// Create volume rendering scene graph in specified view
static void createVolumeView( int viewId );

// Create the slice views and their shared data nodes
static void createSliceViews();

// Create slice rendering scene graph in specified view
static void createSliceView( int viewId, MedicalHelper::Axis axis, int sliceNumber );

// Event handlers for all views
static void onKeyPress( void* userData, SoEventCallback* node );
static void onOtherEvent( void* userData, SoEventCallback* node );

// Event handlers for slice views
static void onMouseButton_Slice( void* data, SoEventCallback* node );
static void onMouseMove_Slice  ( void* data, SoEventCallback* node );
static void onMouseWheel_Slice ( void* data, SoEventCallback* node );
static void onTouchEvent_Slice ( void* data, SoEventCallback* node );

// Change to a different viewport layout, maximized view, etc
static void viewportLayout( int layout );
static int  maximizeView( int viewIndex );
static void updateWindowCW( const SbVec2s& newPos );

// Change to a different image in a slice view.
static void goToNewImage( int viewIndex, int upOrDown );

// User interface updaters
static void ui_updateSliceNum( int viewIndex );
static void ui_updateWinCtrWidth( float center, float width );

// Sensor to trigger data loading after render area is established
static void sensorCB( void* data, SoSensor* sensor );
static void loadData();
static void traversalCB( void *userData, SoAction *action );

#if ! REMOTE_VIZ
// Catch Open Inventor error reports.
void errorCB( const SoError* error, void *data )
{
  // Print messages except the useless one about touch screen not supported.
  const SbString& msg = error->getDebugString();
  if (msg.contains(SbString("SoWinTouchScreen"))) {
    m_touchScreen = false;
  }
  else {
    SoWin::getErrorConsole()->printMessage( msg );
  }
}
#endif

///////////////////////////////////////////////////////////////////////////////
#if REMOTE_VIZ
// Triggered when a client is connected to the service. */
void RemoteMPRServiceListener::onConnectedClient( const std::string& clientId, std::shared_ptr<RemoteViz::Rendering::NetworkPerformance>)
{
  std::cout << "onConnectedClient id: " << clientId << std::endl;
}

void RemoteMPRServiceListener::onInstantiatedRenderArea( std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea)
{
  std::cout << "\nCreated render area: " << renderArea->getId() << std::endl;

  // Instantiate the class overriding the renderAreaListener class to manage the renderArea events.
  std::shared_ptr<RemoteMPRRenderAreaListener> renderAreaListener(new RemoteMPRRenderAreaListener());

  // Add the renderAreaListener instance as renderArea listener
  renderArea->addListener(renderAreaListener);

  // Add recognizers for gesture events
  // Adds SoScaleGestureRecognizer, SoRotateGestureRecognizer,
  //      SoLongTapGestureRecognizer and SoDoubleTapGestureRecognizer
  // SceneExaminer handles the first two for us.
  renderArea->getTouchManager()->addDefaultRecognizers();

  m_renderArea = renderArea;

#else
void main(int argc, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  SoDialogViz::init();  // For SoMessageDialog
  SoVolumeRendering::init();

  // Create render area
  m_renderArea = new SoXtRenderArea( myWindow );
    m_renderArea->setTitle( "MPR Viewer" );
    m_renderArea->setSize( MedicalHelper::exampleWindowSize() );

#ifdef WIN32
  // Enable touch screen events (if supported) and add default recognizers.
  // - We need an error handler to detect if touch screen is supported.
  // - Default recognizers are SoScaleGestureRecognizer, SoRotateGestureRecognizer
  //   SoLongTapGestureRecognizer and SoDoubleTapGestureRecognizer.
  //   SceneExaminer handles the first two for us.
  SoDebugError::setHandlerCallback( errorCB, (void*)NULL );
  SoWinTouchScreen touchScreenDevice( myWindow );
  if (m_touchScreen) {
    m_renderArea->registerDevice( &touchScreenDevice );
    touchScreenDevice.getTouchManager()->addDefaultRecognizers();
  }
#endif
#endif

  // Initialize Open Inventor medical extensions
  InventorMedical::init();

  ViewManager::initClass();
  SceneView::initClass();

  // Help info
  std::cout << "Press:\n"
               "  'M' to maximize view\n"
               "  'R' to reset camera\n"
               "  'W' to reset window level/width\n"
               "  '1' for 2x2 square layout\n"
               "  '2' for 1x3 side layout\n"
               "  '3' for 1x3 below layout\n"
               "Volume view:\n"
               "  Mouse 1   / 1 finger : Orbit\n"
               "  Mouse 2              : Pan\n"
               "  Mouse 1+2 / 2 finger : Zoom\n"
               "  Double tap           : Maximize\n"
               "Slice View:\n"
               "  Mouse 1   / 1 finger : Image number\n"
               "  Mouse 2   / 2 finger : Window center/width\n"
               "  Double tap           : Maximize\n";

  // Create root of scene graph and assign to render area
  SoRef<SoSeparator> root = new SoSeparator();
  m_renderArea->getSceneManager()->setSceneGraph( root.ptr() );

  // Handle events common to all views.
  // Note this node is above the SceneExaminer nodes so we can "filter"
  // what that node sees and override some built-in behaviors.
  SoEventCallback* eventNode = new SoEventCallback;
    eventNode->addEventCallback( SoKeyboardEvent::getClassTypeId(), onKeyPress, NULL );
    eventNode->addEventCallback( SoMouseButtonEvent::getClassTypeId(), onOtherEvent, NULL );
    eventNode->addEventCallback( SoTouchEvent::getClassTypeId(), onOtherEvent, NULL );
    eventNode->addEventCallback( SoDoubleTapGestureEvent::getClassTypeId(), onOtherEvent, NULL );
    eventNode->addEventCallback( SoLongTapGestureEvent::getClassTypeId(), onOtherEvent, NULL );
    eventNode->addEventCallback( SoRotateGestureEvent::getClassTypeId(), onOtherEvent, NULL );
    eventNode->addEventCallback( SoScaleGestureEvent::getClassTypeId(), onOtherEvent, NULL );
    root->addChild( eventNode );

  // Specify settings to change when rendering in interactive mode.
  // Setting interactive mode also triggers low res rendering mode in SoVolumeRender.
  // Changing these settings helps maintain frame rate while (for example) moving the camera.
  // We also use this node to explicitly set "interactive mode" because the SceneExaminer
  // does not currently do that automatically. See the onOtherEvent() method.
  m_interactNode = new SoInteractiveComplexity();
    m_interactNode->fieldSettings.set1Value( 0, "SoComplexity value 0.3 0.9" );
    m_interactNode->fieldSettings.set1Value( 1, "SoVolumeRender interpolation LINEAR CUBIC" );
    //m_interactNode->fieldSettings.set1Value( 2, "SoVolumeRender lowScreenResolutionScale 2 1" );
    m_interactNode->interactiveMode = SoInteractiveComplexity::InteractiveMode::FORCE_STILL;
    m_interactNode->refinementDelay = 0;
    root->addChild( m_interactNode );
    root->addChild(new SoComplexity());  // Must have an SoComplexity node to be affected by the settings.

  // --------------------------------------------------------------------------
  // Volume data common to all views.
  // This node will be instanced in each view, but data range and color map are view specific.
  // It might seem logical to add the volume data node to the scene here, above all the views,
  // since it is shared, but that makes it inconvenient to do "viewAll" in each view since the
  // viewAll needs the volume data node to correctly compute the scene's bounding box.
  m_volData = new SoVolumeData();
    m_volData->extent.setValue( 0,0,0, 1,1,1 ); // Temporary until data loaded

  // --------------------------------------------------------------------------
  // Create four views in a 2x2 layout (we can change that later).
  // Create view manager
  m_viewManager = new ViewManager();
    root->addChild( m_viewManager );

  // Upper left: volume rendering
  m_views[0] = new SceneView();
    m_views[0]->setName("UpperLeft");
    m_views[0]->setViewport( SbVec4f(Viewports[0]) );
    m_viewManager->addView( m_views[0] );

  // Upper right
  m_views[1] = new SceneView();
    m_views[1]->setName( "UpperRight" );
    m_views[1]->setViewport( SbVec4f(Viewports[1]) );
    m_viewManager->addView( m_views[1] );

  // Lower left
  m_views[2] = new SceneView();
    m_views[2]->setName( "LowerLeft" );
    m_views[2]->setViewport( SbVec4f(Viewports[2]) );
    m_viewManager->addView( m_views[2] );

  // Lower right
  m_views[3] = new SceneView();
    m_views[3]->setName( "LowerRight" );
    m_views[3]->setViewport( SbVec4f(Viewports[3]) );
    m_viewManager->addView( m_views[3] );

  // --------------------------------------------------------------------------
  // Create rendering scene graphs
  createVolumeView( 0 ); // In view 0 (upper left)
  createSliceViews();

  // Display loading message in volume view
  SceneExaminer* exam = (SceneExaminer*)m_viewManager->getView(0)->getChild(0);

  m_messageSwitch = new SoSwitch();
    m_messageSwitch->whichChild = 0;   // Initially visible
    exam->addChild( m_messageSwitch );

  TextBox* loadingMessage = new TextBox();
    loadingMessage->position.setValue( 0,0,0 );
    loadingMessage->alignmentH = TextBox::AlignmentH::CENTER;
    loadingMessage->alignmentV = TextBox::AlignmentV::MIDDLE;
    loadingMessage->addLine( "LOADING..." );
    loadingMessage->getFontNode()->size = 32;
    m_messageSwitch->addChild( loadingMessage );

#if REMOTE_VIZ
#else
  // Finally, schedule a sensor to load data after the render area is established.
  // Note this is not reliable when using RemoteViz. In that case we use the
  // RenderAreaListener callbacks to trigger loading after the first render.
  SoIdleSensor* sensor = new SoIdleSensor();
    sensor->setFunction( sensorCB );
    sensor->schedule();

    // Loop, then cleanup
  SoXt::show(myWindow);
  SoXt::mainLoop();

  root = NULL;
  SceneView::exitClass();
  ViewManager::exitClass();
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
#endif
}

///////////////////////////////////////////////////////////////////////////////
// Create scene graph for volume rendering.
void createVolumeView( int viewId )
{
  SceneExaminer* exam = new SceneExaminer();
    m_views[viewId]->addChild( exam );

  SoSeparator* volSep = new SoSeparator();
    exam->addChild( volSep );

  // Add an instance of the volume data
  volSep->addChild( m_volData );

  // We already selected a range for the demo data set.
  m_volRange = new SoDataRange();
    m_volRange->min = 176;
    m_volRange->max = 476;
    volSep->addChild( m_volRange );

  // We already created a color map for the demo data set.
  m_volCmap = new SoTransferFunction();
    m_volCmap->loadColormap( COLORMAPFILENAME );
    volSep->addChild( m_volCmap );

  // Volume rendering options (lighting, etc)
  m_volQual = new SoVolumeRenderingQuality();
    m_volQual->interpolateOnMove = TRUE;
    m_volQual->preIntegrated     = TRUE;
    m_volQual->deferredLighting  = TRUE;
    m_volQual->ambientOcclusion  = TRUE;
    volSep->addChild( m_volQual );

  // The actual rendering node.
  // Note the lowResMode is only used when Open Inventor is in interactive mode.
  m_volRend = new SoVolumeRender();
    m_volRend->numSlicesControl = SoVolumeRender::NumSlicesControl::AUTOMATIC;
    m_volRend->numSlices = -1;
    m_volRend->samplingAlignment = SoVolumeRender::SamplingAlignment::BOUNDARY_ALIGNED;
    m_volRend->opacityThreshold  = 0.01f;
    m_volRend->lowResMode = SoVolumeRender::LowResMode::DECREASE_SCREEN_RESOLUTION;
    m_volRend->lowScreenResolutionScale = 2;
    volSep->addChild( m_volRend );

  // Adjust camera (note orientView() also does a viewAll)
  exam->setCameraMode( SceneInteractor::CameraMode::ORTHOGRAPHIC );
  MedicalHelper::orientView( MedicalHelper::Axis::CORONAL, exam->getCamera(), m_volData );
  exam->getCameraInteractor()->pushCamera();            // Save default view

  // Orientation indicator
  exam->addChild( new Gnomon() );

  // OIV Logo
  exam->addChild( MedicalHelper::exampleLogoNode() );

  // Annotation
  DicomInfo* upperLeft = new DicomInfo();
    upperLeft->fileName = IMAGE_FILENAME;
    upperLeft->position.setValue( -0.99f, 0.99f, 0 );
    upperLeft->addLine( "Volume" );
    upperLeft->displayDicomInfo("", 0x0008, 0x1090); // Model name
    upperLeft->displayDicomInfo("", 0x0008, 0x1030); // Study Description
    upperLeft->displayDicomInfo("", 0x0008, 0x103E); // Series Description
    exam->addChild( upperLeft );

  DicomInfo* upperRight = new DicomInfo();
    upperRight->fileName = IMAGE_FILENAME;
    upperRight->position.setValue(0.99f, 0.99f, 0);
    upperRight->alignmentH = DicomInfo::AlignmentH::RIGHT;
    upperRight->textAlignH = DicomInfo::AlignmentH::RIGHT;
    upperRight->displayDicomInfo("", 0x0008, 0x0080); // Institution
    upperRight->displayDicomInfo("", 0x0010, 0x0010); // Patient Name
    upperRight->displayDicomInfo("", 0x0008, 0x0022); // Acquisition date
    upperRight->displayDicomInfo("", 0x0008, 0x0032); // Acquisition time
    exam->addChild(upperRight);

  DicomInfo* lowerRight = new DicomInfo();
    lowerRight->fileName = IMAGE_FILENAME;
    lowerRight->position.setValue(0.99f, -0.98f, 0);
    lowerRight->alignmentH = DicomInfo::AlignmentH::RIGHT;
    lowerRight->alignmentV = DicomInfo::AlignmentV::BOTTOM;
    lowerRight->textAlignH = DicomInfo::AlignmentH::RIGHT;
    lowerRight->displayDicomInfo("", 0x0008, 0x0060); // Modality
    lowerRight->displayDicomInfo("mA: ", 0x0018, 0x1151); // X-Ray Tube Current
    lowerRight->displayDicomInfo("kV: ", 0x0018, 0x0060); // KVP (Kilo Voltage Peak)
    exam->addChild(lowerRight);
}

///////////////////////////////////////////////////////////////////////////////
// Create all slice views
void createSliceViews()
{
  // Shared slice data range
  m_sliceRange = new SoDataRange();
    MedicalHelper::dicomAdjustDataRange( m_sliceRange, m_volData );

  // Shared slice rendering material (full intensity)
  m_sliceMatl = new SoMaterial();
    m_sliceMatl->diffuseColor.setValue( 1, 1, 1 );

  // Shared slice rendering colormap (opaque gray scale).
  // Note we will check the Photometric Interpretation value when we actually load some data.
  m_sliceCmap = new SoTransferFunction();
    m_sliceCmap->predefColorMap = SoTransferFunction::PredefColorMap::INTENSITY;

  // Create slice rendering scene graphs
  createSliceView( 1, MedicalHelper::Axis::SAGITTAL, m_volDim[MedicalHelper::Axis::SAGITTAL]/2 );
  createSliceView( 2, MedicalHelper::Axis::CORONAL , m_volDim[MedicalHelper::Axis::CORONAL ]/2 );
  createSliceView( 3, MedicalHelper::Axis::AXIAL   , m_volDim[MedicalHelper::Axis::AXIAL   ]/2 );

  // Update UI
  for (int viewIndex = 1; viewIndex < 4; ++viewIndex)
    ui_updateSliceNum( viewIndex );
  m_resetWinCtrWidth = MedicalHelper::dicomGetWindowCenterWidth( m_sliceRange );
  ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );
}

///////////////////////////////////////////////////////////////////////////////
// Create a specific slice view
void createSliceView( int viewId, MedicalHelper::Axis axis, int sliceNumber )
{
  // Camera controller for this slice is a "2D" camera.
  // Note that we start in picking/selection mode so the SceneExaminer will *not*
  // respond to events by moving the camera. Instead we use the events to change
  // the image number, window center/width, etc.
  SceneExaminer* exam = new SceneExaminer();
    exam->setCameraMode( SceneExaminer::ORTHOGRAPHIC );
    exam->setNavigationMode( SceneExaminer::PLANE );
    exam->setInteractionMode( SceneExaminer::SELECTION );
    m_views[viewId]->addChild( exam );

  // Handle events specific to slices
  // Note that the SceneExaminer automatically converts 1-finger touch and drag
  // events to mouse button and mouse move events. So we can handle basic touch
  // interaction using the onMouseXXX methods.
  SoEventCallback* eventNode = new SoEventCallback;
    eventNode->addEventCallback( SoMouseButtonEvent::getClassTypeId() , onMouseButton_Slice );
    eventNode->addEventCallback( SoLocation2Event::getClassTypeId()   , onMouseMove_Slice );
    eventNode->addEventCallback( SoMouseWheelEvent::getClassTypeId()  , onMouseWheel_Slice );
    eventNode->addEventCallback( SoScaleGestureEvent::getClassTypeId(), onTouchEvent_Slice );
    exam->addChild( eventNode );

  // Force bounding box to be extent of the volume.
  // Note that SceneExaminer does not automatically adjust near/far clipping planes
  // unless the camera is modified (unlike the legacy ExaminerViewer).  Changing to
  // a different slice means the slice geometry is at a different Z value and that
  // could move the geometry outside the current clip planes.  Setting the bbox to
  // the full volume extent ensures all slices are visible.
  SoBBox* bboxNode = new SoBBox();
    bboxNode->mode = SoBBox::Mode::USER_DEFINED;
    bboxNode->boundingBox.connectFrom( &m_volData->extent );
    exam->addChild( bboxNode );

  // Keep the volume rendering separate from geometry rendering.
  SoSeparator* volSep = new SoSeparator();
    exam->addChild( volSep );

  // Add an instance of the shared volume data
  volSep->addChild( m_volData );

  // Add shared slice rendering property nodes.
  volSep->addChild( m_sliceRange );
  volSep->addChild( m_sliceMatl  );
  volSep->addChild( m_sliceCmap  );

  // Add slice rendering node with high quality interpolation.
  SoOrthoSlice* slice = new SoOrthoSlice();
    slice->axis = axis;
    slice->sliceNumber = sliceNumber;
    slice->interpolation = SoSlice::Interpolation::MULTISAMPLE_12;
    volSep->addChild( slice );
    m_sliceNodes[axis] = slice;

  // Adjust camera
  MedicalHelper::orientView( axis, exam->getCamera(), m_volData );  // Standard orientation
  exam->getCameraInteractor()->pushCamera();                        // Save default view

  // Add standard slice orientation markers
  exam->addChild( MedicalHelper::buildSliceOrientationMarkers( slice ) );

  // Add slice scale bars
  m_sliceSBars[axis] = new SoSwitch();
    m_sliceSBars[axis]->whichChild = -1;  // Turned off when viewport is small
    m_sliceSBars[axis]->addChild( MedicalHelper::buildSliceScaleBars( exam->getCamera() ));
    exam->addChild( m_sliceSBars[axis] );

  static char* AXIS_NAMES[] = { "Sagittal", "Coronal", "Axial" };

  // Upper left annotation (View name, slice number, slice position)
  TextBox* textUL = new TextBox();
    textUL->position.setValue( -0.98f, 0.98f, 0 );
    textUL->addLine( AXIS_NAMES[axis] );
    textUL->addLine( "Image: " );
    exam->addChild( textUL );

  // Lower left annotation (window level/width)
  TextBox* textLL = new TextBox();
    textLL->position.setValue( -0.98f, -0.98f, 0 );
    textLL->alignmentV = TextBox::AlignmentV::BOTTOM;
    textLL->addLine( "WL / WW: " );
    exam->addChild( textLL );

  int sliceId = viewId - 1; // Slice views are view indices 1,2,3
  m_sliceTextUL[sliceId] = textUL;
  m_sliceTextLL[sliceId] = textLL;
}

///////////////////////////////////////////////////////////////////////////////
// For a slice view, change to the next/previous image in the stack.
void
goToNewImage( int viewIndex, int delta )
{
  if (viewIndex < 1 || viewIndex > 3)
    return;
  int sliceIndex = viewIndex - 1; // Slices are in views 1..3
  SoOrthoSlice* sliceNode = m_sliceNodes[sliceIndex];
  int curSlice  = (int)sliceNode->sliceNumber.getValue();
  int axis      = (int)sliceNode->axis.getValue();
  int numSlices = m_volDim[axis];

  // If delta is negative, increment slice number, else decrement.
  // This arithmetic seems "backward" but it's the expected behavior.
  int newSlice = curSlice - delta;

  // Update if necessary.
  // SoOrthoSlice will "clamp" invalid values, so this check is not strictly
  // necessary, but it avoids triggering an unnecessary redraw.
  if (newSlice < 0) {
    newSlice = 0;
  }
  else if (newSlice >= numSlices) {
    newSlice = numSlices - 1;
  }
  if (newSlice != curSlice) {
    sliceNode->sliceNumber = newSlice;
  }
}

///////////////////////////////////////////////////////////////////////////////
// Update wherever the UI displays the slice number.
//
// NOTE! OIV numbers slices starting at 0, but medical applications
// typically display the image number starting at 1.
void
ui_updateSliceNum( int viewIndex )
{
  // Note slice views are 1..3
  if (viewIndex < 1 || viewIndex > 3)
    return;
  int sliceIndex = viewIndex - 1;  // Slices are in views 1..3
  const SoOrthoSlice* sliceNode = m_sliceNodes[sliceIndex];
  int sliceNum  = (int)sliceNode->sliceNumber.getValue();
  int sliceAxis = (int)sliceNode->axis.getValue();
  int numSlices = m_volDim[sliceAxis];
  TextBox* textBox = m_sliceTextUL[sliceIndex];

  // Update image number
  SbString str;
  str.sprintf( "Image  %d  of  %d", sliceNum + 1, numSlices );
  textBox->setLine( str, 1 );

  // Update image position
  float pos = m_volExt.getMin()[sliceAxis] + (sliceNum * m_voxelSize[sliceAxis]);
  str.setNum( pos );
  str += SbString(" mm");
  textBox->setLine( str, 2 );
}

///////////////////////////////////////////////////////////////////////////////
// Update wherever the UI displays the window center/width.
void
ui_updateWinCtrWidth( float center, float width )
{
  // Note slice views are 1..3
  for (int viewIndex = 1; viewIndex < 4; ++viewIndex) {
    int sliceIndex = viewIndex - 1;
    TextBox* textBox = m_sliceTextLL[sliceIndex];

    SbString str;
    str.sprintf( "%g / %g", center, width );
    textBox->setLine( str, 1 );
  }
}

///////////////////////////////////////////////////////////////////////////////
// Switch to a different viewport layout
void viewportLayout( int layout )
{
  // Undo any "maximized" view.
  maximizeView( -1 );

  // --------------------------------------------------------------------------
  // ---------- Layout 2
  if (layout == 2) {
    // 1 + column layout
    float oneThird = 1 / 3.0f;
    float twoThird = 2 / 3.0f;
    float midSize  = oneThird;
    // The height of the render area may not be evenly divisible by 3.  In that case
    // the middle viewport must be 1 pixel taller to avoid a gap.
    // TODO: To handle the user resizing the render area we should re-compute on size
    //       change. For now, pressing the number key again will fix any gaps.
    const SbVec2i32& viewport = m_renderArea->getSceneManager()->getViewportRegion().getViewportSizePixels_i32();
    if (((viewport[1]/3)*3) != viewport[1]) { // Not divisible by 3
      midSize += 1 / (float)viewport[1];      // Add 1 pixel in normalized coordinates
    }
    m_viewManager->getView( 0 )->setViewport( 0, 0, twoThird, 1 );
    m_viewManager->getView( 1 )->setViewport( twoThird, twoThird, oneThird, oneThird );
    m_viewManager->getView( 2 )->setViewport( twoThird, oneThird, oneThird, midSize  );
    m_viewManager->getView( 3 )->setViewport( twoThird,        0, oneThird, oneThird );
  }

  // --------------------------------------------------------------------------
  // ---------- Layout 3
  else if (layout == 3) {
    // 1 + row layout
    float oneThird = 1 / 3.0f;
    float twoThird = 2 / 3.0f;
    float midSize  = oneThird;
    // The width of the render area may not be evenly divisible by 3.  In that case
    // the middle viewport must be 1 pixel wider to avoid a gap.
    // TODO: To handle the user resizing the render area we should re-compute on size
    //       change. For now, pressing the number key again will fix any gaps.
    const SbVec2i32& viewport = m_renderArea->getSceneManager()->getViewportRegion().getViewportSizePixels_i32();
    if (((viewport[0]/3)*3) != viewport[0]) { // Not divisible by 3
      midSize += 1 / (float)viewport[0];      // Add 1 pixel in normalized coordinates
    }
    m_viewManager->getView( 0 )->setViewport(        0, oneThird,        1, twoThird );
    m_viewManager->getView( 1 )->setViewport(        0,        0, oneThird, oneThird );
    m_viewManager->getView( 2 )->setViewport( oneThird,        0, midSize , oneThird );
    m_viewManager->getView( 3 )->setViewport( twoThird,        0, oneThird, oneThird );
  }

  else {
    // 4 square layout
    m_viewManager->getView( 0 )->setViewport( SbVec4f(Viewports[0]) );
    m_viewManager->getView( 1 )->setViewport( SbVec4f(Viewports[1]) );
    m_viewManager->getView( 2 )->setViewport( SbVec4f(Viewports[2]) );
    m_viewManager->getView( 3 )->setViewport( SbVec4f(Viewports[3]) );
  }
}

///////////////////////////////////////////////////////////////////////////////
// Maximize (or restore) a view.
//
// Call with any invalid view index to force maximized view to be restored.
int maximizeView( int viewIndex )
{
  int rc = 0;
  int numViews = m_viewManager->getNumChildren();

  // If a view is currrently maximized...
  if (m_maximizedViewIndex >= 0) {
    // Restore maximized view / Re-activate all other views
    for (int i = 0; i < numViews; ++i) {
      if (i == m_maximizedViewIndex)
        m_viewManager->getView( m_maximizedViewIndex )->setViewport( m_maximizedViewSave );
      else
        m_viewManager->getView( i )->active = TRUE;
    }
    if (m_maximizedViewIndex > 0) {
      // Slice view: Remove scale bars while small
      m_sliceSBars[m_maximizedViewIndex-1]->whichChild = -1;
    }
    m_maximizedViewIndex = -1;
  }

  // Maximize specified view (if valid) / De-activate other views
  else if (viewIndex >= 0 && viewIndex < numViews) {
    for (int i = 0; i < numViews; ++i) {
      if (i == viewIndex) {
        m_maximizedViewIndex = i;
        m_maximizedViewSave  = m_viewManager->getView(i)->getViewport();
        m_viewManager->getView( i )->setViewport( 0, 0, 1, 1 );
      }
      else
        m_viewManager->getView( i )->active = FALSE;
    }
    if (m_maximizedViewIndex > 0) {
      // Slice view: Display scale bars while large
      m_sliceSBars[m_maximizedViewIndex-1]->whichChild = 0;
    }
  }

  else {
    rc = -1;
  }
  return rc;
}

///////////////////////////////////////////////////////////////////////////////
// Update Window Center/Width based on change in event position.
// Called from onMouseMove_Slice and onTouchEvent_Slice.
//
void updateWindowCW( const SbVec2s& newPos )
{
  int deltaX = newPos[0] - m_mousePosition[0];
  int deltaY = newPos[1] - m_mousePosition[1];
  // Sometimes we get a stream of mouse move events with the same event position.
  // This is probably a glitch in the mouse and/or Windows. Ignore this case.
  if (deltaX != 0 || deltaY != 0) {
    SbVec2f winCW = MedicalHelper::dicomGetWindowCenterWidth( m_sliceRange );
    winCW[0] += deltaY;
    winCW[1] += deltaX;
    if (winCW[1] < 1)
      winCW[1] = 1;
    MedicalHelper::dicomSetWindowCenterWidth( m_sliceRange, winCW );
    ui_updateWinCtrWidth( winCW[0], winCW[1] );
  }
}

///////////////////////////////////////////////////////////////////////////////
// Handle mouse button events for slices
// -- Affects what happens when the mouse cursor is moved.
// -- SceneExaminer automatically converts 1-finger down/up touch events to
//    mouse button down/up events.  So we are also handling touch here.

void onMouseButton_Slice( void* data, SoEventCallback* node )
{
  const SoMouseButtonEvent* theEvent = (const SoMouseButtonEvent*)node->getEvent();

  if (SoMouseButtonEvent::isButtonPressEvent(theEvent,SoMouseButtonEvent::BUTTON1)) {
    m_mouse1Down = true;
    if (! m_mouse2Down) // Don't change if we're already in a non-default mode.
      m_mouseMode = MOUSE_SCROLL_IMAGE;
  }
  else if (SoMouseButtonEvent::isButtonReleaseEvent(theEvent,SoMouseButtonEvent::BUTTON1)) {
    m_mouse1Down = false;
    if (! m_mouse2Down)
      m_mouseMode = MOUSE_SHOW_VALUE; // Default when no button pressed
  }
  else if (SoMouseButtonEvent::isButtonPressEvent(theEvent,SoMouseButtonEvent::BUTTON2)) {
    m_mouse2Down = true;
    if (! m_mouse1Down)
      m_mouseMode = MOUSE_CHANGE_WINCW;
  }
  else if (SoMouseButtonEvent::isButtonReleaseEvent(theEvent,SoMouseButtonEvent::BUTTON2)) {
    m_mouse2Down = false;
    if (! m_mouse1Down)
      m_mouseMode = MOUSE_SHOW_VALUE; // Default when no button pressed
  }
  else if (SoMouseButtonEvent::isButtonPressEvent(theEvent,SoMouseButtonEvent::BUTTON3)) {
    node->setHandled();
  }
  else if (SoMouseButtonEvent::isButtonReleaseEvent(theEvent,SoMouseButtonEvent::BUTTON3)) {
    node->setHandled();
  }

  // Remember position
  m_mousePosition = theEvent->getPosition();
}

///////////////////////////////////////////////////////////////////////////////
// Handle mouse move events for slices.
// Behavior depends on which mouse buttons are pressed (if any).

void onMouseMove_Slice( void* data, SoEventCallback* node )
{
  // If user is dragging, that should be handled in onTouchEvent_Slice.
  if (m_2FingerDrag)
    return;

  const SoLocation2Event* theEvent = (const SoLocation2Event*)node->getEvent();

  // Check what mode we are in.
  if (m_mouseMode == MOUSE_SCROLL_IMAGE) {
    //---------------------------------------------------------------
    // This change only applies to the current slice window.
    SbVec2s newPos = theEvent->getPosition();
    int delta = newPos[1] - m_mousePosition[1];
    // Sometimes we get a stream of mouse move events with the same event position.
    // This is probably a glitch in the mouse and/or Windows. Ignore this case.
    if (delta != 0) {
      // Increment slice number and update display
      int viewIndex = m_viewManager->getLastEventViewIndex();
      goToNewImage( viewIndex, delta );
      ui_updateSliceNum( viewIndex );
    }
  }

  else if (m_mouseMode == MOUSE_CHANGE_WINCW) {
    //---------------------------------------------------------------
    // This change applies to all slice windows.
    updateWindowCW( theEvent->getPosition() );
  }

  // Remember new position
  m_mousePosition = theEvent->getPosition();
}

///////////////////////////////////////////////////////////////////////////////
// Handle mouse wheel events for slices
// --> Change the image number.
// For consistency with other applications:
//   - Mouse wheel forward decreases image number.
//   - Mouse wheel backward increases image number.
void onMouseWheel_Slice( void* data, SoEventCallback* node )
{
  const int MOUSE_WHEEL_DELTA = 120; // True for all known mice...

  const SoMouseWheelEvent* theEvent = (SoMouseWheelEvent*)node->getEvent();
  int delta = theEvent->getDelta() / MOUSE_WHEEL_DELTA;  // Should be -1 or +1

  // Note that RemoteViz does not set event position for wheel events.
  int viewIndex = m_viewManager->getLastEventViewIndex();
  goToNewImage( viewIndex, delta );
  ui_updateSliceNum( viewIndex );
}

///////////////////////////////////////////////////////////////////////////////
// Handle touch events for slices
// -- Affects what happens when the mouse cursor is moved.
// -- SceneExaminer automatically converts 1-finger down/up touch events to
//    mouse button down/up events. Handled in onMouseButton_Slice.

void onTouchEvent_Slice( void* data, SoEventCallback* node )
{
  const SoEvent* theEvent = node->getEvent();

  // --------- 2-Finger Drag event ----------
  // Open Inventor doesn't currently provide a 2-finger-drag gesture.
  // We could easily write a recognizer for this simple gesture, but since we
  // don't need an actual scale gesture in the slice views, we can use the
  // this event to tell us when two fingers are down.
  if (theEvent->isOfType(SoScaleGestureEvent::getClassTypeId())) {
    const SoGestureEvent* gesEvent = (SoGestureEvent*)theEvent;
    SoGestureEvent::GestureState state = gesEvent->getGestureState();
    if (state == SoGestureEvent::GestureState::BEGIN) {
      m_2FingerDrag = true;     // Two fingers are down
      m_mouseMode = MOUSE_NOOP; // Don't handle mouseMove for now
    }
    else if (state == SoGestureEvent::GestureState::END) {
      m_2FingerDrag = false;  // Fingers are up
    }
    else {
      // Drag - Same code as onMouseMove_Slice
      updateWindowCW( gesEvent->getPosition() );
    }
    // Remember new position
    m_mousePosition = theEvent->getPosition();
    node->setHandled();
  }
}

///////////////////////////////////////////////////////////////////////////////
// Handle key events
// These events variously affect all views or the current view.
//
// Number keys:
//   - Open Inventor sends keyboard number keys and keypad number keys as
//     different events.
//   - Keypad number key events are only sent when NumLock is on.
//
void onKeyPress(void *userData, SoEventCallback *node )
{
  const SoEvent* theEvent = node->getEvent();

  // ---------- 'D'ata loading ------------------------------------------------
  // We just use this in the RemoteViz configuration as as easy way for the
  // RenderAreaListener to start data loading after the initial rendering.
  // In that configuration the SoIdleSensor technique doesn't work reliably.
  if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::D )) {
    loadData();
    node->setHandled();
  }

  // ---------- 'I'mage reset -------------------------------------------------
  // Reset image numbers to initial state.
  // Only affects the slice views.
  if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::I )) {
    for (int axis = 0; axis < 3; ++axis) {
      m_sliceNodes[axis]->sliceNumber = m_volDim[axis] / 2;
    }
    node->setHandled();
  }

  // ---------- 'L'ow-res Move -------------------------------------------------
  // Toggle use of quality reduction to maintain interactive rendering speed.
  // Only affects the volume rendering view.
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::L )) {
    m_useLowRes = ! m_useLowRes;
    if (! m_useLowRes)
      m_interacting = false;
    node->setHandled();
  }

  // ---------- 'M'aximize ----------------------------------------------------
  // Maximize/Restore view.
  // Only affects the view containing the cursor.
  // Note that RemoteViz does not set event position for keyPress events.
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::M )) {
    // Maximize or restore current view
    int viewIndex = m_viewManager->getLastEventViewIndex();
    maximizeView( viewIndex );
    node->setHandled();
  }

  // ---------- 'R'eset Camera -------------------------------------------------
  // Reset camera for all views.
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::R )) {
    for (int i = 0; i < m_viewManager->getNumViews(); ++i) {
      SceneView* view = m_viewManager->getView( i );
      SceneExaminer* exam = (SceneExaminer*)view->getChild( 0 );
      exam->getCameraInteractor()->popCamera();   // Restore default view
      exam->getCameraInteractor()->pushCamera();  // Re-save default view
    }
    node->setHandled();
  }

  // ---------- 'W'indow Level/Width Reset ------------------------------------
  // Reset window center/width to original values.
  // Affects all slice views regardless of cursor position.
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::W )) {
    MedicalHelper::dicomSetWindowCenterWidth( m_sliceRange, m_resetWinCtrWidth );
    ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );
  }

  // --------------------------------------------------------------------------
  // ---------- Layout 1
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::NUMBER_1)
        || SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::PAD_1   )) {
    // 4 square layout
    viewportLayout( 1 );
    node->setHandled();
  }

  // --------------------------------------------------------------------------
  // ---------- Layout 2
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::NUMBER_2)
        || SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::PAD_2 )) {
    // 1 + column layout
    viewportLayout( 2 );
    node->setHandled();
  }

  // --------------------------------------------------------------------------
  // ---------- Layout 3
  else if (SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::NUMBER_3)
        || SoKeyboardEvent::isKeyPressEvent( theEvent, SoKeyboardEvent::Key::PAD_3 )) {
    // 1 + row layout
    viewportLayout( 3 );
    node->setHandled();
  }
}

///////////////////////////////////////////////////////////////////////////////
// Handle miscellaneous events
// I.e. events that affect the current view, but independent of view type.
//
void onOtherEvent(void *userData, SoEventCallback *node )
{
  const SoEvent* theEvent = node->getEvent();

  // --------------------------------------------------------------------------
  // ---------- Double-click / Double-tap to Maximize
  if (ALLOW_DOUBLE_CLICK
   && SoMouseButtonEvent::isButtonDoubleClickEvent(theEvent,SoMouseButtonEvent::Button::BUTTON1)) {
    // Maximize or restore current view.
    // - Double-click is disabled by default because some touch screen devices
    //   automatically send a double-click when the user does a double-tap,
     //  resulting in two events for the same user action.
    // - This event would be handled by the SceneExaminer by default,
    //   but we do not allow that behavior.
    const SbVec2f& pos = theEvent->getPositionFloat();
    int viewIndex = m_viewManager->getViewIndex( pos );
    maximizeView( viewIndex );
    node->setHandled();
  }
  else if (theEvent->isOfType(SoDoubleTapGestureEvent::getClassTypeId())) {
    // Maximize or restore current view.
    // Note this event would be handled by the SceneExaminer by default,
    // but we do not allow that behavior.
    const SbVec2f& pos = theEvent->getPositionFloat();
    int viewIndex = m_viewManager->getViewIndex( pos );
    maximizeView( viewIndex );
    node->setHandled();
  }


  // --------------------------------------------------------------------------
  // ---------- LongTap event
  else if (theEvent->isOfType(SoLongTapGestureEvent::getClassTypeId())) {
    // This event is automatically handled by the SceneExaminer and toggles
    // the SceneExaminer between viewing and selection modes.
    // Unfortunately, in some cases this event is triggered by accident, for
    // example if you briefly stop moving during a 1-finger rotate gesture.
    // That can be very confusing.  Since there is no visible indication that
    // the SceneExaminer is no longer in viewing mode, the user may conclude
    // that the camera is "frozen".
    // Therefore we currently setHandled() to block this event from reaching
    // the SceneExaminer.
    node->setHandled();
  }

  // --------------------------------------------------------------------------
  // ---------- Button or Finger press/release
  // Work around for setting SoInteractionElement to trigger low-res rendering
  // while interacting.  Note that we're just "monitoring" these events.  We
  // still want them to be handled in the usual way by SceneExaminer etc.
  //
  // This is only an "approximate" solution because we're just checking for any
  // down or up event.  It should work for the most common cases: using 1 button
  // or 1 finger to move the camera around.
  //
  // On any button or finger "down" event we activate low-res rendering.
  // On any button or finger "up" event we return to normal rendering.
  else if (m_useLowRes) {
    bool relevantEvent = false;
    int  upOrDown      = 2; // 0 = down, 1 = up, everything else is "other"
    if (theEvent->isOfType(SoMouseButtonEvent::getClassTypeId())) {
      relevantEvent = true;
      SoMouseButtonEvent::State state = ((SoMouseButtonEvent*)theEvent)->getState();
      if      (state == SoMouseButtonEvent::State::DOWN) upOrDown = 0;
      else if (state == SoMouseButtonEvent::State::UP  ) upOrDown = 1;
    }
    else if (theEvent->isOfType(SoTouchEvent::getClassTypeId())) {
      relevantEvent = true;
      SoTouchEvent::State state = ((SoTouchEvent*)theEvent)->getState();
      if      (state == SoTouchEvent::State::DOWN) upOrDown = 0;
      else if (state == SoTouchEvent::State::UP  ) upOrDown = 1;
    }
    if (relevantEvent) {
      if (upOrDown == 0) { // DOWN
        m_interacting = true;
        m_interactNode->interactiveMode = SoInteractiveComplexity::InteractiveMode::FORCE_INTERACTION;
      }
      else if (upOrDown == 1) { // UP
        // Note we should be able to set InteractiveMode to AUTO, but in RemoteViz
        // mode that doesn't reliably force rendering back to full quality (to be investigated).
        m_interacting = false;
        m_interactNode->interactiveMode = SoInteractiveComplexity::InteractiveMode::FORCE_STILL;
      }
    }
  }

#if REMOTE_VIZ
  // ---------- Shift-F12 --------------------------------
  // RemoteViz does not automatically start IvTune when Shift-F12 is pressed
  // but this can be very convenient for debugging.
  else if (SoKeyboardEvent::isKeyPressEvent(theEvent,SoKeyboardEvent::Key::F12)) {
    if (theEvent->wasShiftDown()) {
      SoNode* root = m_renderArea->getSceneManager()->getSceneGraph();
      if (root != NULL)
        SoIvTune::start( root );
      node->setHandled();
    }
  }
#endif
}

///////////////////////////////////////////////////////////////////////////////
// Should be called after the render area is established.
// This allows us to display messages to the user while we're busy.
void sensorCB( void* data, SoSensor* sensor )
{
  loadData();
}

///////////////////////////////////////////////////////////////////////////////
// Load the image stack.
// Should be called after the render area is established.
// This allows us to display messages to the user while we're busy.
void loadData()
{
  std::cout << "Loading...\n";

  // Check that we can access the data
  SbBool useShortList = SoPreferences::getBool( "DICOM_SHORT_LIST", FALSE );
  SbString filename = useShortList ? VOLUME_FILENAME2 : VOLUME_FILENAME;
  if (! SbFileHelper::isAccessible( filename )) {
    filename = VOLUME_FILENAME;
    if (! SbFileHelper::isAccessible( filename )) {
      SoMessageDialog* dialog = new SoMessageDialog( filename, "Unable to open:", SoMessageDialog::MD_ERROR );
      return;
    }
  }

  // Global initialization -- Load the volume data
  m_volData->fileName = filename;
  MedicalHelper::dicomAdjustVolume( m_volData );

  // Remember volume characteristics for user interface updates.
  // Note that volume extent min is the outer edge of the first voxel,
  // DICOM origin is the _center_ of the first voxel (difference of 1/2 voxel).
  m_volDim = m_volData->data.getSize();
  m_volExt = m_volData->extent.getValue();
  SbVec3f volSize = m_volExt.getSize();
  m_voxelSize.setValue( volSize[0]/m_volDim[0], volSize[1]/m_volDim[1], volSize[2]/m_volDim[2] );
  SbVec3f dicomOrigin = m_volExt.getMin() + (0.5f * m_voxelSize);
  double volMin, volMax;
  m_volData->getMinMax( volMin, volMax );

  // Set data range for slices from the volume's window center/width.
  // Save that initial center/width and update the UI.
  MedicalHelper::dicomAdjustDataRange( m_sliceRange, m_volData );
  m_resetWinCtrWidth = MedicalHelper::dicomGetWindowCenterWidth( m_sliceRange );
  ui_updateWinCtrWidth( m_resetWinCtrWidth[0], m_resetWinCtrWidth[1] );

  std::cout << "\nVolume dims: " << m_volDim << std::endl;
  std::cout << "       extent: " << m_volExt.getMin() << " " << m_volExt.getMax() << std::endl;
  std::cout << "  DICOMorigin: " << dicomOrigin << std::endl;
  std::cout << "     PhysSize: " << volSize << std::endl;
  std::cout << "    voxelSize: " << m_voxelSize << std::endl;
  std::cout << "  scalarRange: " << volMin << " to " << volMax << std::endl;
  std::cout << "  windowCtrWid " << m_resetWinCtrWidth[0] << " / " << m_resetWinCtrWidth[1] << std::endl;

  // Slice rendering color map.
  // Usually INTENSITY (black to white), but check for MONOCHROME1 case.
  m_sliceCmap->predefColorMap = SoTransferFunction::PredefColorMap::INTENSITY;
  MedicalHelper::dicomCheckMonochrome1( m_sliceCmap, m_volData );

  // Per-view initialization...
  for (int viewIndex = 0; viewIndex < 4; ++viewIndex) {
    // For each view: Initialize camera and save (push) as 'reset' view.
    SceneExaminer* exam = (SceneExaminer*)m_viewManager->getView(viewIndex)->getChild(0);
    MedicalHelper::Axis axis = ViewAxes[viewIndex];
    MedicalHelper::orientView( axis, exam->getCamera(), m_volData );
    exam->getCameraInteractor()->pushCamera();
    // For slice views:
    if (viewIndex > 0) {
      // Set sliceNumber initially to the middle of the volume and update UI.
      m_sliceNodes[viewIndex-1]->sliceNumber = m_volDim[axis] / 2;
      ui_updateSliceNum( viewIndex );
    }
  }

  // Finally, disable the loading message
  m_messageSwitch->whichChild = -1; // Disabled
}
