///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author      : Pascal Estrade (December 2015)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Example of Multi Planar Reconstruction viewer.
Description : Listener part.
-------------------------------------------------------------------------*/
#include "medicalRemoteMPRListener.h"

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

#include <RemoteViz/Rendering/RenderArea.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/touch/SoTouchManager.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <Inventor/nodes/SoCone.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <Inventor/actions/SoGLRenderAction.h>
#include <VolumeViz/draggers/SoOrthoSliceDragger.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/actions/SoBoxHighlightRenderAction.h>
#include <Inventor/nodes/SoTranslation.h>

///////////////////////////////////////////////////////////////////////////////
#define FILENAME "$OIVHOME/examples/source/Medical/Web/medicalRemoteMPR/medicalRemoteMPRService/simpleVolumeRender.iv"

// Data used in sensor callback.
struct callBackData {
    RenderArea *localRenderArea;
    SoSeparator *localRoot;
    SoVertexProperty *localVertexProperty;
    SceneExaminer *localSceneExaminer;
    SoPath *localOrthoPath;
    int localAxis; // 0=X(Sagittal), 1=Y(Coronal), 2=Z(Transverse)
};

///////////////////////////////////////////////////////////////////////////////
// Sensor Callback to compute new orthoslice border for each new orthoslice dragger new position.
void orthoSliceSensorCB(void *data, SoSensor *sensor) {
    callBackData *localData = (callBackData*)data;

    // Get the boundingBox
    SoGetBoundingBoxAction bboxAction(localData->localRenderArea->getSceneManager()->getViewportRegion());
    bboxAction.apply(localData->localOrthoPath);
    SbBox3f bbox = bboxAction.getBoundingBox();

    float xmin, ymin, zmin, xmax, ymax, zmax;
    bbox.getBounds(xmin, ymin, zmin, xmax, ymax, zmax);

    switch (localData->localAxis) {
    case 0: { // SAGITTAL
        localData->localVertexProperty->vertex.set1Value(0, SbVec3f(xmin, ymin, zmin));
        localData->localVertexProperty->vertex.set1Value(1, SbVec3f(xmin, ymax, zmin));
        localData->localVertexProperty->vertex.set1Value(2, SbVec3f(xmax, ymax, zmax));
        localData->localVertexProperty->vertex.set1Value(3, SbVec3f(xmax, ymin, zmax));
        localData->localVertexProperty->vertex.set1Value(4, SbVec3f(xmin, ymin, zmin));
    } break;
    case 1: { // CORONAL
        localData->localVertexProperty->vertex.set1Value(0, SbVec3f(xmin, ymin, zmin));
        localData->localVertexProperty->vertex.set1Value(1, SbVec3f(xmin, ymax, zmax));
        localData->localVertexProperty->vertex.set1Value(2, SbVec3f(xmax, ymax, zmax));
        localData->localVertexProperty->vertex.set1Value(3, SbVec3f(xmax, ymin, zmin));
        localData->localVertexProperty->vertex.set1Value(4, SbVec3f(xmin, ymin, zmin));
    } break;
    case 2: { // TRANSVERS
        localData->localVertexProperty->vertex.set1Value(0, SbVec3f(xmin, ymin, zmin));
        localData->localVertexProperty->vertex.set1Value(1, SbVec3f(xmin, ymax, zmax));
        localData->localVertexProperty->vertex.set1Value(2, SbVec3f(xmax, ymax, zmax));
        localData->localVertexProperty->vertex.set1Value(3, SbVec3f(xmax, ymin, zmin));
        localData->localVertexProperty->vertex.set1Value(4, SbVec3f(xmin, ymin, zmin));
    } break;
    }
    // Common ViewAll
    localData->localSceneExaminer->viewAll(localData->localRenderArea->getSceneManager()->getViewportRegion());
}

///////////////////////////////////////////////////////////////////////////////
// Common method to create sceneGraph 2D viewers.
void medicalRemoteMPRListener::createSceneGraph(SoGroup *initialScene, SoRef<SceneExaminer> sceneExaminer, std::shared_ptr<RenderArea> renderArea, int axis, SoSeparator *borderSep) {

    SoVolumeData* localVolumeData = (SoVolumeData*)initialScene->getByName("volumeData");
    MedicalHelper::dicomAdjustVolume( localVolumeData );

    SoDataRange* localDataRange = (SoDataRange*)initialScene->getByName("dataRange");
    MedicalHelper::dicomAdjustDataRange( localDataRange, localVolumeData );

    SoSeparator *scene = new SoSeparator();
    scene->addChild(initialScene);

    SoOrthoSlice *orthoSlice = new SoOrthoSlice();
      orthoSlice->interpolation = SoSlice::MULTISAMPLE_12;

    // BOrder Color
    SoMaterial *borderMaterial = new SoMaterial();

    switch ( axis ) {
    case 0: // Sagittal
        orthoSlice->axis.setValue(SoOrthoSlice::X);
        orthoSlice->sliceNumber = localVolumeData->data.getSize()[0] / 2;
        borderMaterial->diffuseColor.setValue(1, 0, 0);
        orthoSlice->setName("orthoSliceSagittal");

        break;
    case 1:
        orthoSlice->axis.setValue(SoOrthoSlice::Y);
        orthoSlice->sliceNumber = localVolumeData->data.getSize()[1] / 2;
        borderMaterial->diffuseColor.setValue(0, 1, 0);
        orthoSlice->setName("orthoSliceCoronal");

        break;
    case 2:
        orthoSlice->axis.setValue(SoOrthoSlice::Z);
        orthoSlice->sliceNumber = localVolumeData->data.getSize()[2] / 2;
        borderMaterial->diffuseColor.setValue(0, 0, 1);
        orthoSlice->setName("orthoSliceTransvers");

        break;
    }
    scene->addChild(orthoSlice);

    // Create path to slice node
    // Note: Can be a partial path but must include the slice node.
    SoPath* path = new SoPath(scene);
    path->ref();
    path->append(orthoSlice);

    SoOrthoSliceDragger *transversOrthoDragger = new SoOrthoSliceDragger();

    transversOrthoDragger->orthoSlicePath = path;
    scene->addChild(transversOrthoDragger);

    borderSep->fastEditing.setValue(SoSeparator::KEEP_ZBUFFER);
    borderSep->addChild(borderMaterial);
    // Lineset taht define the border.
    SoLineSet *border = new SoLineSet();
    border->numVertices = 5;
    SoVertexProperty *borderVertexProperty = new SoVertexProperty();
    border->vertexProperty = borderVertexProperty;
    borderSep->addChild(border);

    scene->addChild(borderSep);

    SoNodeSensor *planeSensor = new SoNodeSensor();
    planeSensor->attach(orthoSlice);

    callBackData *data = new callBackData();
    data->localRenderArea = renderArea;
    data->localRoot = scene;
    data->localVertexProperty = borderVertexProperty;
    data->localOrthoPath = path;
    data->localAxis = axis;
    data->localSceneExaminer = sceneExaminer.ptr();

    planeSensor->setFunction(orthoSliceSensorCB, data);

    sceneExaminer->addChild(scene);
    // Apply the sceneExaminer node as renderArea scene graph
    renderArea->getSceneManager()->setSceneGraph(sceneExaminer.ptr());
}

///////////////////////////////////////////////////////////////////////////////
void medicalRemoteMPRListener::onInstantiatedRenderArea(std::shared_ptr<RenderArea> renderArea)
{
    InventorMedical::init();

    // Instantiate a renderAreaListener class to manage the renderArea events (default behaviors).
    std::shared_ptr<RenderAreaListener> renderAreaListener(new RenderAreaListener());

    // Add the renderAreaListener instance as renderArea listener
    renderArea->addListener(renderAreaListener);
    // Add recognizers for gesture events
    renderArea->getTouchManager()->addDefaultRecognizers();

    SoInput myInput;
    if (myInput.openFile(FILENAME))
    {
        static SbBool firstRead = TRUE;
        static SoGroup *group;
        
        static SoRef<SceneExaminer> rootTL = NULL;
        static SoRef<SceneExaminer> rootTR = NULL;
        static SoRef<SceneExaminer> rootBL = NULL;
        static SoRef<SceneExaminer> rootBR = NULL;

        static SoSeparator *sagittalBorderSep = NULL;
        static SoSeparator *coronalBorderSep = NULL;
        static SoSeparator *transverseBorderSep = NULL;

        // Read common sceneGraph (iv file).
        if ( firstRead == TRUE) {
            SoSeparator *scene = SoDB::readAll(&myInput);
            group = (SoGroup*)SoNode::getByName("CommonGroup");
            firstRead = FALSE;
        }
        if (group) {
            // 3D View ===============================================================================================================
            if (renderArea->getId() == "renderArea3D_TOP_LEFT") {
                // Instantiate a sceneExaminer to interact with the camera
                rootTL = new SceneExaminer();
                rootTL->setCameraMode(SceneInteractor::ORTHOGRAPHIC);
                renderArea->getSceneManager()->getGLRenderAction()->setTransparencyType(SoGLRenderAction::SORTED_PIXEL);
                renderArea->getSceneManager()->setAntialiasing(0.9f);
                SoCameraInteractor *cam3DInteractor = rootTL->getCameraInteractor();
                cam3DInteractor->setOrientation(SbRotation(SbVec3f(1, 0, 0), (float)M_PI / 2.0f));

                // Scene graph
                SoSeparator *topLeftSG = new SoSeparator();
                topLeftSG->addChild(group);
                SoVolumeRender *volumeRender = new SoVolumeRender();
                volumeRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
                volumeRender->numSlices = -1;
                volumeRender->samplingAlignment.setValue(SoVolumeRender::BOUNDARY_ALIGNED);
                topLeftSG->addChild(volumeRender);
                rootTL->addChild(topLeftSG);

                // OIV Logo
                rootTL->addChild( MedicalHelper::exampleLogoNode() );

                // Medical Gnomon.
                Gnomon *gnomon = new Gnomon();
                gnomon->setScale(SbVec3f(0.8f, 0.8f, 1.0f));
                rootTL->addChild(gnomon);

                // Apply the sceneExaminer node as renderArea scene graph
                renderArea->getSceneManager()->setSceneGraph(rootTL.ptr());
                rootTL->viewAll(renderArea->getSceneManager()->getViewportRegion());
            }

            // SAGITTAL View ===============================================================================================================
            if (renderArea->getId() == "renderArea2D_TOP_RIGHT") {
                // Instantiate a sceneExaminer to interact with the camera
                rootTR = new SceneExaminer();
                rootTR->enableOrbit(false);
                rootTR->setInteractionMode(SceneExaminer::PICKING);
                rootTR->setCameraMode(SceneInteractor::ORTHOGRAPHIC);

                SoCameraInteractor *camSagittalInteractor = rootTR->getCameraInteractor();
                camSagittalInteractor->setOrientation(SbRotation(SbVec3f(0, 1, 0), (float)M_PI / 2.0f));
                camSagittalInteractor->setRotationAxis(SbVec3f(0, 0, 1));
                camSagittalInteractor->rotate(3.14f / 2.0);

                // Scene graph
                sagittalBorderSep = new SoSeparator();
                createSceneGraph(group, rootTR.ptr(), renderArea, 0, sagittalBorderSep);
            }
            // CORONAL View. ===============================================================================================================
            if (renderArea->getId() == "renderArea2D_BOTTOM_LEFT") {
                // Instantiate a sceneExaminer to interact with the camera
                rootBL = new SceneExaminer();
                rootBL->enableOrbit(false);
                rootBL->setInteractionMode(SceneExaminer::PICKING);
                rootBL->setCameraMode(SceneInteractor::ORTHOGRAPHIC);

                SoCameraInteractor *camCoronalInteractor = rootBL->getCameraInteractor();
                camCoronalInteractor->setOrientation(SbRotation(SbVec3f(1.0, 0.0, 0.0), (float)M_PI / 2.0f));
                
                // Scene graph
                coronalBorderSep = new SoSeparator();
                createSceneGraph(group, rootBL.ptr(), renderArea, 1, coronalBorderSep);
            }
            // TRANSVERSE View. ============================================================================================================
            if (renderArea->getId() == "renderArea2D_BOTTOM_RIGHT") {
                // Instantiate a sceneExaminer to interact with the camera
                rootBR = new SceneExaminer();
                rootBR->enableOrbit(false);
                rootBR->setInteractionMode(SceneExaminer::PICKING);
                rootBR->setCameraMode(SceneInteractor::ORTHOGRAPHIC);

                // Scene graph
                transverseBorderSep = new SoSeparator();
                createSceneGraph(group, rootBR.ptr(), renderArea, 2, transverseBorderSep);
            }

            // To add & render orthoslice borders in all sceneExaminers.
            if (rootTL.ptr() && rootTR.ptr() && rootBL.ptr() && rootBR.ptr() &&
                sagittalBorderSep && coronalBorderSep && transverseBorderSep) {

                rootTL->addChild(sagittalBorderSep);
                rootTL->addChild(coronalBorderSep);
                rootTL->addChild(transverseBorderSep);

                rootTR->addChild(coronalBorderSep);
                rootTR->addChild(transverseBorderSep);

                rootBL->addChild(sagittalBorderSep);
                rootBL->addChild(transverseBorderSep);

                rootBR->addChild(sagittalBorderSep);
                rootBR->addChild(coronalBorderSep);
            }

            // Border Init. Activate nodeSensor Callback.
            if (rootTR.ptr() && rootBL.ptr() && rootBR.ptr()) {
                SoOrthoSlice *orthoSlice = (SoOrthoSlice*)transverseBorderSep->getByName("orthoSliceTransvers");
                orthoSlice->touch();

                orthoSlice = (SoOrthoSlice*)coronalBorderSep->getByName("orthoSliceCoronal");
                orthoSlice->touch();

                orthoSlice = (SoOrthoSlice*)sagittalBorderSep->getByName("orthoSliceSagittal");
                orthoSlice->touch();
            }
        }
    }
}