///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include "simpleMedicalRemoteListener.h"

#include <RemoteViz/Rendering/RenderArea.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/nodes/SoCone.h>

#include <Inventor/SbTime.h>
#include <Inventor/fields/SoSFTime.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/touch/SoTouchManager.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

#include <Inventor/ViewerComponents/nodes/SceneExaminer.h>

///////////////////////////////////////////////////////////////////////////////
// Data Set
// This file contains the list of dicom we want to load.
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm";

// color map can be edited within Avizo.
const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

///////////////////////////////////////////////////////////////////////////////

static SoSeparator* BuildSceneGraph();

///////////////////////////////////////////////////////////////////////////////

#define FILENAME "$OIVHOME/examples/source/Medical/Web/medicalSimpleRemote/medicalSimpleRemoteService/simpleVolumeRender.iv"

///////////////////////////////////////////////////////////////////////////////
void simpleMedicalRemoteListener::onInstantiatedRenderArea(std::shared_ptr<RenderArea> renderArea)
{
    // Instantiate a renderAreaListener class to manage the renderArea events (default behaviors).
    std::shared_ptr<RenderAreaListener> renderAreaListener(new RenderAreaListener());

    // Add the renderAreaListener instance as renderArea listener
    renderArea->addListener(renderAreaListener);

    // Add default recognizers for gesture events
    renderArea->getTouchManager()->addDefaultRecognizers();

    // Create root of scene graph and assign to render area
    SoRef<SoSeparator> root = new SoSeparator();
    renderArea->getSceneManager()->setSceneGraph( root.ptr() );

    // Add SceneExaminer to allow user to control camera
    SceneExaminer* examiner = new SceneExaminer();
    root->addChild( examiner );

    // Add scene graph to be displayed
    SoSeparator* sg = BuildSceneGraph();
    examiner->addChild( sg );

    // Adjust camera based on the volume loaded
    SoVolumeData* volData = MedicalHelper::find<SoVolumeData>( root.ptr() );
    MedicalHelper::orientView( MedicalHelper::Axis::CORONAL, examiner->getCamera(), volData );
}

///////////////////////////////////////////////////////////////////////////////
// Create the volume rendering scene graph
SoSeparator* BuildSceneGraph()
{
  SoRef<SoSeparator> root = new SoSeparator();

  SoSeparator* volSep = new SoSeparator();
    root->addChild( volSep );

  // Node to hold the volume data
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName = VOLUME_FILENAME;
    MedicalHelper::dicomAdjustVolume( volData );
    volSep->addChild( volData );

  std::cout << "\nVolume dimensions: " << volData->data.getSize() << std::endl;

  // Set range of values to map into the color map.
  // In this case we have pre-selected a range for the demo data set and color map.
  SoDataRange* volRange = new SoDataRange();
    volRange->min = 176;
    volRange->max = 476;
    volSep->addChild( volRange );

  // Add specular to change rendering quality.
  SoMaterial* volMatl = new SoMaterial();
    volMatl->specularColor.setValue(0.26f, 0.26f, 0.26f);
    volSep->addChild( volMatl );

  // Load the colorMap
  SoTransferFunction* volTF = new SoTransferFunction();
    volTF->loadColormap(COLORMAP_FILENAME);
    volSep->addChild( volTF );

  // Change complexity when the camera moves.
  SoInteractiveComplexity* volInter = new SoInteractiveComplexity();
    // Decrease the number of samples
    volInter->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
    // Decrease interpolation quality
    volInter->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    volInter->refinementDelay = 0; // Don't wait before returning to full quality rendering.
    volSep->addChild(volInter);
    volSep->addChild(new SoComplexity());

  // Property node which allows SoVolumeRender to draw High Quality volumes.  
  SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
    volQual->interpolateOnMove = TRUE;
    volQual->deferredLighting = TRUE;
    volQual->preIntegrated = TRUE;
    volQual->ambientOcclusion = TRUE;
    volSep->addChild( volQual );

  // Node in charge of drawing the volume
  SoVolumeRender* volRender = new SoVolumeRender();
    volRender->numSlicesControl = SoVolumeRender::NumSlicesControl::AUTOMATIC;
    volRender->lowScreenResolutionScale = 2;
    volRender->lowResMode = SoVolumeRender::LowResMode::DECREASE_SCREEN_RESOLUTION;
    volRender->samplingAlignment = SoVolumeRender::SamplingAlignment::BOUNDARY_ALIGNED;
    volSep->addChild( volRender );

  // Medical Gnomon.
  Gnomon *gnomon = new Gnomon();
    root->addChild(gnomon);

  // Define Open Inventor logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  return root.releaseNoDelete(); // Return root with refCount = 0
}