//==============================================================================
//   File        : Furnace.C
//   From        : 3Ddata
//   Author(s)   : F. CHENAIS
//   Date        : 09/08/1996
//==============================================================================
//  MeshViz (c) Copyright 1996 G5G, all right reserved
//==============================================================================
//
// Description : Show heat spreading on a furnace surface.
//               Use group6axis3, datamapping, isosurfaces, meshes, dialogBox ..
//
//============================================================================

#include <string.h>
#include <stdio.h>

#if defined(_WIN32)
#pragma warning(disable: 4996) // Disable PoXt deprecation warning
#endif

#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/draggers/SoRotateCylindricalDragger.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoProfileCoordinate2.h>
#include <Inventor/nodes/SoLinearProfile.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoAnnoText3Property.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>


#include <MeshViz/graph/PbDomain.h>
#include <MeshViz/graph/PbIsovaluesList.h>
#include <MeshViz/graph/PbNonLinearDataMapping2.h>
#include <MeshViz/graph/PoNonLinearValueLegend3.h>
#include <MeshViz/graph/PbMiscTextAttr.h>
#include <MeshViz/graph/PoGroup6Axis3.h>
#include <MeshViz/graph/PoLinearAxis.h>

#include <MeshViz/3Ddata/PbCartesianGrid2D.h>
#include <MeshViz/3Ddata/PoMeshFilled.h>
#include <MeshViz/3Ddata/PoMeshContouring.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/helpers/SbFileHelper.h>

SbString dataFilePath = SbFileHelper::expandString("$OIVHOME/examples/data/MeshViz/");

#define NB_COLOR    5
SbString ivFileName = "Furnace.iv";

/******************************************************************************/
//: Specific examiner viewer.
//   This class differs from its inherited class SoXtExaminerViewer only by
//   its viewAll() method. Here, viewAll ignore the 2D part of the scene graph
//   (legend, courtesy title) so the position of the camera is calculated only
//   to view entirely the representation of the mesh. 
/******************************************************************************/
class PoXtExaminerViewer : public SoXtExaminerViewer {
public:
  PoXtExaminerViewer(Widget parent=NULL, const char *name=NULL,
		     SbBool buildInsideParent=TRUE, 
		     SoXtFullViewer::BuildFlag flag=BUILD_ALL,
		     SoXtViewer::Type type=BROWSER) ;
  void viewAll() ;
    // Changes the position of the camera to view entirely the different
    // nodes that represent the mesh.
} ;

/*-------------------------------------------------------------------*/
PoXtExaminerViewer::PoXtExaminerViewer(Widget parent, const char *name,
				       SbBool buildInsideParent, 
				       SoXtFullViewer::BuildFlag flag,
				       SoXtViewer::Type typeViewer) :
  SoXtExaminerViewer(parent, name, buildInsideParent, flag,
		     typeViewer)
{
}/*---------------------------------------------------------------------------*/

void
PoXtExaminerViewer::viewAll()
{
  SoCamera *my_camera = getCamera();
  my_camera->viewAll(SoNode::getByName("Scene3D"), getViewportRegion()) ;
  SbVec3f newCameraPosition = my_camera->position.getValue();
  newCameraPosition[0]*=1.2F;
  my_camera->position.setValue(newCameraPosition);
}/*---------------------------------------------------------------------------*/

SoSwitch         *meshFilledSwitch, *meshContouringSwitch, *captorsSwitch ;
PoMeshFilled     *meshFilled ;
SoSeparator      *root ;
/*-------------------------------------------------------------------*/
// Create a title, at the top of the scene
//
static SoSeparator*
myTitle(const char *title, float angle, float size)
{
  SoSeparator *textSep       = new SoSeparator;
  SoText3     *text          = new SoText3;
  SoFont      *titleFont     = new SoFont;
  SoTransform *textTransform = new SoTransform;
  SoDirectionalLight *light  = new SoDirectionalLight;
  
  light->direction.setValue(1, 1, -0.8F);
  
  textSep->addChild(titleFont);
  textSep->addChild(light);
  
  SoMaterial *myMaterial = new SoMaterial;
  SbColor colors[3];
  // diffuse
  colors[0].setHSVValue(0.15F, 0.50F, 0.91F);
  colors[1].setHSVValue(0.15F, 0.73F, 1.0F);
  colors[2].setHSVValue(0.15F, 0.73F, 1.0F);
  myMaterial->diffuseColor.setValues(0, 3, colors);
  textSep->addChild(myMaterial);

  // Specify a beveled cross-section for the text
  SoProfileCoordinate2 *myProfileCoords = new SoProfileCoordinate2;
  SbVec2f coords[4];
  coords[0].setValue(0.00, 0.00);
  coords[1].setValue(size/16, size/16);
  coords[2].setValue(size/8,  size/16);
  coords[3].setValue(3*size/16, 0.00);
  myProfileCoords->point.setValues(0, 4, coords);
  textSep->addChild(myProfileCoords);
  
  SoLinearProfile *myLinearProfile = new SoLinearProfile;
  int32_t index[4] ;
  index[0] = 0;
  index[1] = 1;
  index[2] = 2;
  index[3] = 3;
  myLinearProfile->index.setValues(0, 4, index);
  textSep->addChild(myLinearProfile);
  
  // Set the material binding to PER_PART
  SoMaterialBinding *myMaterialBinding = new SoMaterialBinding;
  myMaterialBinding->value.setValue(SoMaterialBinding::PER_PART);
  textSep->addChild(myMaterialBinding);
  
  SoSeparator *textSep2       = new SoSeparator;
  textSep->addChild(textSep2);
  
  text->parts=SoText3::ALL;
  text->string.setValue(title);
  text->justification=SoText3::CENTER;
  
  textTransform->rotation.setValue(SbVec3f(1, 0, 0), angle);
  textTransform->translation.setValue(0, 0.85F, 0.0F);
  
#ifdef _WIN32
  titleFont->name.setValue("Arial");
#else
  titleFont->name.setValue("Times-Roman");
#endif

  titleFont->size.setValue(size);
  
  textSep2->addChild(textTransform);
  textSep2->addChild(text);
  
  return textSep;
}/*-------------------------------------------------------------------*/

// Create a legend for the Temperatures on the left of the scene
//
static SoAnnotation*
myLegend(const char *fontName , PbNonLinearDataMapping2 *dataMapping, 
	 PbIsovaluesList *isoValues)
{
  PbMiscTextAttr *textAttr = new PbMiscTextAttr;
  textAttr->setFontName(fontName) ;
  
  PoNonLinearValueLegend3 *myLegend = new 
    PoNonLinearValueLegend3(SbVec2f(-0.95F,-0.95F), SbVec2f(-0.65F, 0.95F)) ;
  
  myLegend->setDataMapping(dataMapping) ;
  myLegend->setIsovaluesList(isoValues) ;
  myLegend->titleString.setValue("Temperature");
  myLegend->titleFontSize = 0.075F ;  
  myLegend->marginTop = 0.02F ;
  myLegend->marginBottom = 0.02F ;
  myLegend->marginRight = 0.1F ;
  myLegend->marginLeft = 0.1F ;
  myLegend->titleVisibility = PoNonLinearValueLegend3::VISIBILITY_ON ;
  myLegend->setMiscTextAttr(textAttr) ;
  myLegend->set("backgroundApp.material", "diffuseColor 0.3 0.3 0.3") ;
  myLegend->set("backgroundBorderApp.material", "diffuseColor 1 0 0") ;

  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;

  SoAnnotation *legendAnnot = new SoAnnotation ;
  legendAnnot->addChild(annoText3Property) ;
  
  legendAnnot->addChild(myLegend) ;
  
  return legendAnnot ;
}/*-------------------------------------------------------------------*/

static void
setAxisFont(PoGroup6Axis3 *axis, const char *fontName)
{
  PbMiscTextAttr *textFont = new PbMiscTextAttr;
  textFont->setFontName(fontName);
  
  axis->rebuild() ;
  
  PoLinearAxis *axisTmp = SO_GET_PART(axis, "xUpAxis", PoLinearAxis);
  axisTmp->setMiscTextAttr(textFont);
  axisTmp->set("bodyApp.material", "diffuseColor 1 0 0") ;
  axisTmp->gradFontSize.setValue(0.1F);
  axisTmp->titleFontSize.setValue(0.11F);
  axisTmp->multFactorDistAxis.setValue(0.25F);
  
  axisTmp = SO_GET_PART(axis, "xDownAxis", PoLinearAxis);
  axisTmp->setMiscTextAttr(textFont);
  axisTmp->set("bodyApp.material", "diffuseColor 1 0 0") ;
  axisTmp->gradFontSize.setValue(0.1F);
  axisTmp->titleFontSize.setValue(0.11F);
  axisTmp->multFactorDistAxis.setValue(0.25F);
  
  axisTmp = SO_GET_PART(axis, "yLeftAxis", PoLinearAxis);
  axisTmp->setMiscTextAttr(textFont);
  axisTmp->set("bodyApp.material", "diffuseColor 1 0 0") ;
  axisTmp->gradFontSize.setValue(0.1F);
  axisTmp->titleFontSize.setValue(0.11F);
  axisTmp->multFactorDistAxis.setValue(0.25F);
  
  axisTmp = SO_GET_PART(axis, "yRightAxis", PoLinearAxis);
  axisTmp->setMiscTextAttr(textFont);
  axisTmp->set("bodyApp.material", "diffuseColor 1 0 0") ;
  axisTmp->gradFontSize.setValue(0.1F);
  axisTmp->titleFontSize.setValue(0.11F);
  axisTmp->multFactorDistAxis.setValue(0.25F);
  
  
  axisTmp = SO_GET_PART(axis, "zUpAxis", PoLinearAxis);
  axisTmp->tickFirstGrad = 1 ;
  axisTmp->setMiscTextAttr(textFont);
  axisTmp->set("bodyApp.material", "diffuseColor 1 0 0") ;
  axisTmp->gradFontSize.setValue(0.1F);
  axisTmp->titleFontSize.setValue(0.11F);
  
  axisTmp = SO_GET_PART(axis, "zDownAxis", PoLinearAxis);
  axisTmp->setMiscTextAttr(textFont);
  axisTmp->tickFirstGrad = 1 ;
  axisTmp->set("bodyApp.material", "diffuseColor 1 0 0") ;
  axisTmp->gradFontSize.setValue(0.1F);
  axisTmp->titleFontSize.setValue(0.11F);
}/*-------------------------------------------------------------------*/

class ColoringAuditor : public SoDialogChoiceAuditor
{
public:
  void
  dialogChoice( SoDialogChoice* cpt )
  {
    switch( cpt->selectedItem.getValue() ) {
    case 0: // average
      meshFilled->coloringType = PoMesh::COLOR_AVERAGE;
      break;
    case 1: // mapping
      meshFilled->coloringType = PoMesh::COLOR_MAPPING;
      break;
    case 2: // contouring
      meshFilled->coloringType = PoMesh::COLOR_CONTOURING;
      break;
    default:
      break;
    }
  }
};

// SoDialogPushButtonAuditor
class DrawingTypeAuditor : public SoDialogCheckBoxAuditor {
public:
  void
  dialogCheckBox( SoDialogCheckBox* cpt )
  {
    if( cpt->state.getValue() ) {
      meshFilledSwitch->whichChild = SO_SWITCH_NONE;
      meshContouringSwitch->whichChild = SO_SWITCH_ALL;
    }
    else {
      meshFilledSwitch->whichChild = SO_SWITCH_ALL;
      meshContouringSwitch->whichChild = SO_SWITCH_NONE;
    }
  }
};

class CaptorsAuditor : public SoDialogCheckBoxAuditor {
public:
  void
  dialogCheckBox( SoDialogCheckBox* /*cpt*/ )
  {
    if(captorsSwitch-> whichChild.getValue() == SO_SWITCH_ALL)
      captorsSwitch->whichChild = SO_SWITCH_NONE;
    else
      captorsSwitch->whichChild = SO_SWITCH_ALL;
  }
};

class SaveAuditor : public SoDialogPushButtonAuditor
{
public:
  void
  dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    SoWriteAction myAction ;
    myAction.getOutput()->openFile(ivFileName.toLatin1()) ;
    myAction.getOutput()->setBinary(FALSE) ;
    myAction.apply(root) ;
    myAction.getOutput()->closeFile() ;
  }
};

class QuitAuditor : public SoDialogPushButtonAuditor
{
public:
  void
  dialogPushButton( SoDialogPushButton* /*cpt*/ )
  {
    exit( 0 );
  }
};

// Create a dialog Box 
//
void
buildDialogBox(SoTopLevelDialog* dialog)
{
  static const char *colorList[] = { "average","mapping","contouring"};
  SoDialogComboBox* coloring = new SoDialogComboBox();
  coloring->label.setValue( "Coloring type :" );
  coloring->items.setValues( 0, 3, (const char**) colorList );
  coloring->addAuditor( new ColoringAuditor() );
  dialog->addChild( coloring );

  SoDialogCheckBox* drawingType = new SoDialogCheckBox();
  drawingType->label.setValue( "Show :" );
  drawingType->state.setValue( FALSE );
  drawingType->onString.setValue( "Contour Lines" );
  drawingType->offString.setValue( "Mesh Filled" );
  drawingType->addAuditor( new DrawingTypeAuditor() );
  dialog->addChild( drawingType );

  SoDialogCheckBox* captors = new SoDialogCheckBox();
  captors->label.setValue( "Captors :" );
  captors->state.setValue( FALSE );
  captors->onString.setValue( "On" );
  captors->offString.setValue( "Off" );
  captors->addAuditor( new CaptorsAuditor() );
  dialog->addChild( captors );

  SoDialogPushButton* saveButton = new SoDialogPushButton();
  saveButton->buttonLabel.setValue( "Save iv File :" + ivFileName );
  saveButton->addAuditor( new SaveAuditor() );
  dialog->addChild( saveButton );

  SoDialogPushButton* quitButton = new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialog->addChild( quitButton );

}/*-------------------------------------------------------------------*/

static void readMeshGeometry(const char *filename, int &nx, float *&x, int &ny, float *&y) 
{
  SbString SbFileName= dataFilePath;
  SbFileName += filename;
  FILE *file_rep;
  file_rep = SbFileHelper::open(SbFileName, "r" );
  if (file_rep == NULL ) {
    char message[256];
    sprintf(message, "File %s not found !", SbFileName.toLatin1());
#ifdef _WIN32
    MessageBox(NULL, message, "Furnace",MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
    printf("%s\n", message) ;
#endif
    exit(1);
  }
  
  int i,j,num_x,num_y;
  fscanf (file_rep , "%d%d" , &num_x, &num_y);
  nx = num_x;
  ny = num_y;
  float *xt = new float[num_x * num_y] ;
  float *yt = new float[num_x * num_y] ;
  for (j=0; j<num_y; j++) 
    for(i=0; i<num_x; i++) {
      int ind = (i*num_y)+j;
      fscanf(file_rep,"%f%f",&xt[ind], &yt[ind]);
    }
  x = xt;
  y = yt;
  fclose(file_rep) ;
}/*---------------------------------------------------------------------------*/

static void readMeshValues(const char *filename, int num_val, float *&val, float &vmin, float &vmax)
{
  SbString SbFileName= dataFilePath;
  SbFileName += filename;
  FILE *file_rep;
  float vv;
  file_rep = SbFileHelper::open(SbFileName, "r" );
  if (file_rep == NULL ) {
    char message[256];
    sprintf(message, "File %s not found !", SbFileName.toLatin1());
#ifdef _WIN32
    MessageBox(NULL, message, "Furnace",MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
    printf("%s\n", message) ;
#endif
    exit(2);
  }
  
  vmin=1E30F;
  vmax=-1E30F;
  float *v = new float [num_val] ;
  for (int i=0; i<num_val; i++) {
    fscanf(file_rep,"%f",&vv);
    v[i] = vv;
    if (vmin > vv) vmin = vv;
    if (vmax < vv) vmax = vv;
  }
  val = v;
  fclose(file_rep);
}/*---------------------------------------------------------------------------*/

static void
buildCaptors(SoSwitch *sceneGraph, int numX, int numY, float *meshX, float *meshY, 
	     float *meshZ)
{
  SoMaterial *material = new SoMaterial;
  material->diffuseColor.setValue(1.0, 1.0, 1.0);
  sceneGraph->addChild(material);
  for(int j=0; j<numY; j+=5) {
    for(int i=0; i<numX; i+=4) {
      int ind = (i*numY)+j;
      SoSeparator* sep = new SoSeparator;	
      SoSphere* sphere = new SoSphere;
      sphere->radius.setValue (100.0);
      SoTransform* trans = new SoTransform;
      trans->translation.setValue (meshX[ind], meshY[ind], meshZ[ind]);
      sep->addChild (trans);
      sep->addChild (sphere);
      sceneGraph->addChild(sep);
    }
  }
}/*-------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int
main(int, char **argv)
{
  float   *meshXArray;                   // Arrays used to get X, Y and Z values for 
  float   *meshYArray;                   // mesh points.
  float   *meshZArray;                   // 
  float   *meshTemperatureData;		 // The set of values is used for temperature	
  
  float   meshMinElevation, meshMinTemperature;
  float   meshMaxElevation, meshMaxTemperature;
  int     nbCapt, nbGen ;
  
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]) ;
  if(myWindow == NULL) exit(1) ;

  // Initialize the new nodes class
  PoMeshViz::init();
  SoDialogViz::init();
  PoBase::setNodeWriteFormat(PoBase::UNFOLD_NODE_WRITE_FORMAT);

  // In order to keep the compatibility with MeshViz 5.0
  SoPreferences::setFloat("OIV_3DDATA_CREASE_ANGLE",0.8f);
  
  // Read the mesh geometry and values from datafile
  readMeshGeometry("FURNACE.TOPO", nbGen, meshXArray, nbCapt, meshYArray) ;
  readMeshValues("FURNACE0.DAT", nbGen * nbCapt, meshZArray, meshMinElevation, meshMaxElevation) ;
  readMeshValues("FURNACE1.DAT", nbGen * nbCapt, meshTemperatureData, meshMinTemperature, meshMaxTemperature) ;
  
  // Create scene
  // Roots
  root = new SoSeparator ;
  SoSeparator *rootMesh = new SoSeparator ;
  rootMesh->setName("Scene3D");
  SoSeparator *rootDrag = new SoSeparator ;

  SoPerspectiveCamera *camera3D = new SoPerspectiveCamera;
  camera3D->orientation.setValue(1.0F, -0.3F, -0.5F, float(PPI/2.0));

  SoOrthographicCamera *camera2D = new SoOrthographicCamera ;
  camera2D->viewportMapping = SoCamera::LEAVE_ALONE ;
  
  // Create the data mapping
  // colors 
  SbColor colors[NB_COLOR] = {SbColor(0.0,0.,1.0), SbColor(0.0,1.0,0.0), 
			      SbColor(1.0,1.,0.),  SbColor(1.0,.5,0.0), 
			      SbColor(1.0,0.0,0.0)} ;
  // Level for colors
  float val[NB_COLOR] ={ 40.0, 87.5, 130.0, 150.5, 270.0};
  
  PbNonLinearDataMapping2 dataMapping;
  dataMapping.setValues(PbNonLinearDataMapping2::LINEAR_PER_LEVEL, NB_COLOR, val, colors) ;
  
  // Create Isovalues
  PbIsovaluesList isoValues(meshMinTemperature, meshMaxTemperature, 20);
  
  // Legend
#ifdef _WIN32
  SoAnnotation *legend = myLegend("Arial", &dataMapping, &isoValues);
#else
  SoAnnotation *legend = myLegend("Courier", &dataMapping, &isoValues);
#endif
  
  // courtesy title
  SoFont *font = new SoFont;
  font->size = 12;
  SoTranslation *courtesyTrans = new SoTranslation;
  courtesyTrans->translation.setValue(0.95F, -0.95F, -1);
  SoText2 *courtesyTitle = new SoText2;
  courtesyTitle->string = "by courtesy of Sollac";
  courtesyTitle->justification = SoText2::RIGHT;
  SoSeparator *courtesySep = new SoSeparator ;
  courtesySep->addChild(courtesyTrans);
  courtesySep->addChild(courtesyTitle);
  
  //Domain
  PbDomain *domain = new PbDomain(meshMinElevation, meshMinElevation, meshMinElevation, 
				  meshMaxElevation, meshMaxElevation, meshMaxElevation);

  // Axis
  PoGroup6Axis3 *axis = new PoGroup6Axis3;
  axis->start.setValue(meshMinElevation, meshMinElevation, meshYArray[0]);
  axis->end.setValue(meshMaxElevation, meshMaxElevation,   meshYArray[nbGen*nbCapt-1]);
  axis->xTitle = "Radius (mm)";
  axis->yTitle = "Radius (mm)";
  axis->zTitle = "Height (mm)";
  axis->setDomain(domain);

#ifdef _WIN32
  setAxisFont(axis, "Arial");
#else
  setAxisFont(axis, "Courier");
#endif
  
  // Rotate the furnace around X-axis to make it parallele to Z-axis
  SoRotationXYZ *rotateMesh = new SoRotationXYZ;
  rotateMesh->axis = SoRotationXYZ::X;
  rotateMesh->angle.setValue(float(PPI/2));
  
  // Furnace
  // Create 3Ddata mesh Object
  PbCartesianGrid2D mesh;
  mesh.setGeometry(nbGen, nbCapt, meshXArray, meshYArray,meshZArray);
  mesh.addValuesSet(0, meshTemperatureData);
  
  meshFilled = new PoMeshFilled;
  meshFilled->setMesh(&mesh);
  meshFilled->setDataMapping(&dataMapping);
  meshFilled->setIsovaluesList(&isoValues);
  meshFilled->valuesIndex.setValue(0);
  meshFilled->coloringType = PoMesh::COLOR_CONTOURING;
  
  PoMeshContouring *meshContouring = new PoMeshContouring;
  meshContouring->setMesh(&mesh);
  meshContouring->setDataMapping(&dataMapping);
  meshContouring->setIsovaluesList(&isoValues);
  meshContouring->valuesIndex.setValue(0);
  meshContouring->setDomain(domain) ;
  meshContouring->set("minorContourLineApp.drawStyle", "linePattern 0xF0F0") ;
  meshContouring->annotIsVisible = TRUE ;
  meshContouring->annotFontSize.setValue(0.08F) ;
  meshContouring->annotIsContourClip = TRUE ;
  meshContouring->annotCrossStatus = PoMeshContouring::DONT_CROSS_CONTOUR ;
  meshContouring->coloringType = PoMesh::COLOR_MAPPING;
  
  // Transform for Dragger
  SoTransform *transform = new SoTransform;
  
  // Create Objects Tree in the scene
  meshFilledSwitch = new SoSwitch;
  meshFilledSwitch->addChild(meshFilled) ;
  meshFilledSwitch->whichChild = SO_SWITCH_ALL;
  
  meshContouringSwitch = new SoSwitch;
  meshContouringSwitch->addChild(meshContouring) ;
  meshContouringSwitch->whichChild = SO_SWITCH_NONE;
  
  captorsSwitch = new SoSwitch;
  captorsSwitch->whichChild = SO_SWITCH_NONE;

  SoSeparator *scene2D = new SoSeparator ;
  SoSeparator *scene3D = new SoSeparator ;

  scene2D->addChild(camera2D);
  scene2D->addChild(legend);
  scene2D->addChild(myTitle("Furnace", 0.0F, 0.19F));
  scene2D->addChild(courtesySep) ;

  scene3D->addChild(camera3D);
  scene3D->addChild(rootDrag);
  scene3D->addChild(rootMesh);
  
  root->ref();
  root->addChild(scene3D);
  root->addChild(scene2D);

  rootMesh->addChild(axis);
  rootMesh->addChild(rotateMesh);
  rootMesh->addChild(transform);
  rootMesh->addChild(meshFilledSwitch) ;
  rootMesh->addChild(meshContouringSwitch) ;
  rootMesh->addChild(captorsSwitch) ;
  buildCaptors(captorsSwitch, nbGen, nbCapt, meshXArray, meshYArray, meshZArray);

  // Dragger to rotate the furnace around z-axis
  SoRotateCylindricalDragger *dragger = new SoRotateCylindricalDragger;
  transform->rotation.connectFrom(&dragger->rotation);
  
  rootDrag->addChild(rotateMesh);
  rootDrag->addChild(dragger);
  
  SoPath *path = new SoPath(rootMesh);
  path->ref();
  
  dragger->setPartAsPath("rotator", path);
  dragger->setPartAsPath("rotatorActive", NULL);
  dragger->setPartAsPath("feedback", NULL);
  dragger->setPartAsPath("feedbackActive", NULL);

  // Dialog Box
  SoTopLevelDialog *myDialog = new SoTopLevelDialog;
  myDialog->ref();
  buildDialogBox( myDialog );
  myDialog->buildDialog( myWindow );
  myDialog->show();

  // Visualisation Window
  PoXtExaminerViewer *viewer = new PoXtExaminerViewer(myWindow);

  viewer->setSceneGraph(root);
  viewer->setTitle("Furnace");
  viewer->setBackgroundColor(SbColor(0.0F, 0.0F, 0.2F));
  viewer->viewAll();
  viewer->show();
  
  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete viewer;
  myDialog->unref();
  path->unref();
  root->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();

  return 0;
}/*-------------------------------------------------------------------*/
