
#if defined(_WIN32)
#pragma warning(disable: 4996) // Disable PoXt deprecation warning
#endif

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoAnnoText3Property.h>
#include <Inventor/SbColor.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/actions/SoWriteAction.h>

#include <MeshViz/graph/PbDomain.h>
#include <MeshViz/graph/PbDateFormatMapping.h>
#include <MeshViz/graph/PbMiscTextAttr.h>

#include <MeshViz/graph/PoBase.h>
#include <MeshViz/graph/PoGroup2Axis.h>
#include <MeshViz/graph/PoItemLegend.h>
#include <MeshViz/graph/PoLinearAxis.h>
#include <MeshViz/graph/PoTimeAxis.h>
#include <MeshViz/graph/PoCurve.h>
#include <MeshViz/graph/PoArrow.h>

#include "PoDialogColorControl.h"

#include <DialogViz/SoDialogVizAll.h>

#include <GraphEditors.h>
#include <GraphAuditors.h>

#define DOTTED_LINE_STYLE "linePattern 0xf0f0"
#define PLAIN_LINE_STYLE "linePattern 0xffff"

static const int VALUE_MAXY=600;
static const int VALUE_MINY=300;

enum searchMode
{
  X_AXIS_SEARCH=0,
  Y_AXIS_SEARCH=1
};

// search the max and min inside data tab
static void searchMinMax(SbVec2f *data, int numberItem, SbVec2f & max, 
			 SbVec2f & min, searchMode mode);

static void initDataCurve();

static SoSeparator* createAxisTime(float minX, float minY, float maxX, float maxY,
				   const char *date1, const char *date2, 
				   const char *titleX, const char *titleY,
				   PbDateFormatMapping *formatMapping, PbMiscTextAttr *textAtt, 
				   PbDomain *domain);

// put a text at position in touch with a PbDomain
static SoSeparator* putTextAtPosition(float posX, float posY, float posZ,
				      SoNode *textNode, PbDomain *domain);

// create a 3D title, center on the top
static SoSeparator* createTitle(const char *title, float yposition,
				PbDomain *domain);

// create a 2D text at position in touch with a PbDomain
static SoSeparator* createText(const char *label, float posX, float posY, float posZ,
			       SoText2::Justification justification, PbDomain *domain,
			       SbColor color);

static PoArrow* createOneArrow(int numPoints, SbVec2f *points, 
			       PbDomain *domain,
			       SbColor color, const char *pattern=PLAIN_LINE_STYLE);

static SoSeparator* createAllArrow(PbDomain *domain);

static SoAnnotation*createLegend(int numberItems, const char **nameList, 
				 SbColor *colorList, PbMiscTextAttr *textAtt);
// create all inside axis
static SoSeparator* createCurve(PbDomain *domain, SbColor *colorList);

static void upDateColorsCB(SoXtPlaneViewer *viewer, PoDialogColorControl *dialog);

static PoDialogColorControl* createDialog(const char **nameList, SoXtPlaneViewer *viewer,
					  SoSeparator *curveSep, PbDomain *axisDomain,
					  SoNode *rootToApply);

enum curveID
{
  CEGID  = 0,
  IBM    = 1,
  SLIGOS = 2
};

enum typeCurveID
{
  SMOOTH        = 0,
  POLYLINE      = 1,
  HISTO         = 2,
  NB_TYPE_CURVE = 3
};

SbString ivFileName = "axisDemo2D.iv";

const size_t NB_CURVE = SLIGOS+1;
#define NB_DATA 11

const SbString AXIS_NAME("AxisBox");
const float DOLLAR_RATE=5.2F;

static SbVec2f curveDataIbm[NB_DATA], curveDataCegid[NB_DATA]; 
static SbVec2f curveDataSligos[NB_DATA];
static SoMaterial *materialCurveList[NB_CURVE];
static SoMaterial *materialRaisePoint[NB_CURVE];
static PoItemLegend *itemLegend=NULL;

static SoRef<SoTopLevelDialog> axisEditor;

class DollarAxisAuditor : public SoDialogCheckBoxAuditor
{
  SoSeparator* m_axisBox;
  PbDomain* m_domainNewAxis;

public:
  DollarAxisAuditor( SoSeparator* axisBox, PbDomain* domainNewAxis )
    : m_axisBox( axisBox )
    , m_domainNewAxis( domainNewAxis )
  { }

  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    PoAxis* axis = (PoAxis*) m_axisBox->getChild( 0 );

    if ( cpt->state.getValue() )
    {
      PoLinearAxis* yAxis =
        new PoLinearAxis( SbVec3f( -2, VALUE_MINY / DOLLAR_RATE, 0.0 ), VALUE_MAXY / DOLLAR_RATE, PoLinearAxis::YX, 0.0 );
      yAxis->setMiscTextAttr( (PbMiscTextAttr*) axis->getMiscTextAttr() );
      yAxis->setDomain( m_domainNewAxis );
      yAxis->titleString.setValue( "$" );
      yAxis->titleVisibility = PoAxis::VISIBILITY_ON;
      m_axisBox->insertChild( yAxis, 0 );
      SoMaterial* axisMtl;

      // define color for axis
      axisMtl = SO_GET_PART( yAxis, "material", SoMaterial );
      axisMtl->diffuseColor.setHSVValue( 0.15F, 0.30F, 1.0F );

      // define color for graduate
      axisMtl = SO_GET_PART( yAxis, "mainGradTickApp.material", SoMaterial );
      axisMtl->diffuseColor.setHSVValue( 0.56F, 0.20F, 0.4F );
      yAxis->set( "bodyApp.drawStyle", "lineWidth 2" );
    }
    else
      m_axisBox->removeChild( 0 );
  }
};

class AxisEditorAuditor : public SoDialogCheckBoxAuditor
{
public:
  void
  dialogCheckBox( SoDialogCheckBox* cpt )
  {
    if ( cpt->state.getValue() )
      axisEditor->show();
    else
      axisEditor->hide();
  }
};

class CurveTypeAuditor : public SoDialogChoiceAuditor
{
  SoSeparator* m_curveSep;
public:
  CurveTypeAuditor( SoSeparator* curveSep )
    : m_curveSep( curveSep )
  {
  }

  void
  dialogChoice( SoDialogChoice* cpt )
  {
    PoCurve::CurveRep curveRepresent = PoCurve::CURVE_NONE;

    switch ( cpt->selectedItem.getValue() )
    {
    case SMOOTH:
      curveRepresent = PoCurve::CURVE_SMOOTH;
      break;
    case POLYLINE:
      curveRepresent = PoCurve::CURVE_POLYLINE;
      break;
    case HISTO:
      curveRepresent = PoCurve::CURVE_HISTO_X;
      break;
    }

    PoCurve* curve;
    for ( int i = 0; i < m_curveSep->getNumChildren(); i++ )
    {
      curve = (PoCurve*) m_curveSep->getChild( i );
      curve->curveRep.setValue( curveRepresent );
    }
  }
};

class QuitAuditor : public SoDialogPushButtonAuditor
{
  public:
  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    exit( 0 );
  }
};

class WriteAuditor : public SoDialogPushButtonAuditor
{
  SoNode* m_root;

public:
  WriteAuditor(SoNode* root) : m_root(root)
  {}

  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    SoWriteAction myAction;
    myAction.getOutput()->openFile( ivFileName.toLatin1() );
    myAction.getOutput()->setBinary( FALSE );
    myAction.apply( m_root );
    myAction.getOutput()->closeFile();
  }
};

/*-------------------------------------------------------------------*/

SoSeparator* 
createAxisTime(float minX, float minY, float maxX, float maxY,
	       const char *date1, const char *date2, 
	       const char *titleX, const char *titleY,
	       PbDateFormatMapping *formatMapping, PbMiscTextAttr *textAtt, 
	       PbDomain *domain)
{
  PoLinearAxis *yAxis;
  PoTimeAxis   *xAxis;
  SoSeparator  *group2Axis;

  group2Axis = new SoSeparator;
  group2Axis->setName(AXIS_NAME);
  xAxis = new PoTimeAxis(date1, date2, minY);
  formatMapping->setDateMapping(date1, date2, minX, maxX);
  xAxis->setDateFormatMapping(formatMapping);
  xAxis->setMiscTextAttr(textAtt);
  xAxis->timesPeriod = "2DAY";
  if(titleX)
    {
      xAxis->titleVisibility=PoAxis::VISIBILITY_ON;
      xAxis->titleString.setValue(titleX);
    }
  xAxis->setDomain(domain);
  xAxis->gridVisibility=PoAxis::VISIBILITY_ON;
  xAxis->gridLengthGradOtherSide.setValue(maxY-minY);

  yAxis = new PoLinearAxis(SbVec3f(minX, minY, 0.0), maxY, PoCartesianAxis::YX, 0.0);
  yAxis->setMiscTextAttr(textAtt);
  yAxis->setDomain(domain);
  if(titleY)
    {
      yAxis->titleVisibility=PoAxis::VISIBILITY_ON;
      yAxis->titleString.setValue(titleY);
    }
  yAxis->gridVisibility=PoAxis::VISIBILITY_ON;
  yAxis->gridLengthGradOtherSide.setValue(maxX-minX);

  group2Axis->addChild(xAxis);
  group2Axis->addChild(yAxis);

  ((LinearGraphAuditor*) axisEditor->getAuditors().begin()->auditor )->setAxis(yAxis) ;

  SoMaterial *axisMtl;
  // define color for xAxis and yAxis
  axisMtl=SO_GET_PART(xAxis, "appearance.material", SoMaterial);
  axisMtl->diffuseColor.setHSVValue(0.57F, 0.40F, 0.8F);
  axisMtl=SO_GET_PART(yAxis, "appearance.material", SoMaterial);
  axisMtl->diffuseColor.setHSVValue(0.57F, 0.40F, 0.8F);

  xAxis->set("bodyApp.drawStyle", "lineWidth 2");
  xAxis->set("mainGradGridApp.drawStyle", "style LINES");
  xAxis->set("mainGradGridApp.drawStyle", "lineWidth 0.2");
  xAxis->set("mainGradGridApp.drawStyle", DOTTED_LINE_STYLE);
  yAxis->set("bodyApp.drawStyle", "lineWidth 2");
  yAxis->set("mainGradGridApp.drawStyle", "style LINES");
  yAxis->set("mainGradGridApp.drawStyle", "lineWidth 0.2");
  yAxis->set("mainGradGridApp.drawStyle", DOTTED_LINE_STYLE);
  return group2Axis;
}/*-------------------------------------------------------------------*/

SoSeparator*
createCurve(PbDomain *domain, SbColor *colorList)
{
  PoCurve *curveCegid, *curveIbm, *curveSligos;
  SoSeparator *curveGroup=new SoSeparator;

#define MARKER_STYLE_SIZE "pointSize 5.0"
#define CURVE_STYLE_SIZE  "lineWidth 2.0"
#define MARKER_COLOR      "diffuseColor 1 0 0"   // red

  initDataCurve();

  //Cegid
  curveCegid=new PoCurve(NB_DATA, (const SbVec2f *)curveDataCegid, 
			 PoCurve::CURVE_SMOOTH);
  curveCegid->setDomain(domain);

  //Ibm
  curveIbm=new PoCurve(NB_DATA, (const SbVec2f *)curveDataIbm, 
		       PoCurve::CURVE_SMOOTH);
  curveIbm->setDomain(domain);

  //Sligos
  curveSligos=new PoCurve(NB_DATA, (const SbVec2f *)curveDataSligos, 
		       PoCurve::CURVE_SMOOTH);
  curveSligos->setDomain(domain);

  materialRaisePoint[CEGID]=SO_GET_PART(curveCegid, "raisePointApp.material", SoMaterial);
  materialRaisePoint[IBM]=SO_GET_PART(curveIbm, "raisePointApp.material", SoMaterial);
  materialRaisePoint[SLIGOS]=SO_GET_PART(curveSligos, "raisePointApp.material", SoMaterial);

  materialCurveList[CEGID]=SO_GET_PART(curveCegid, "curvePointApp.material", SoMaterial);
  materialCurveList[IBM]=SO_GET_PART(curveIbm, "curvePointApp.material", SoMaterial);
  materialCurveList[SLIGOS]=SO_GET_PART(curveSligos, "curvePointApp.material", SoMaterial);

  static float maxMinValues[3]={8.2F, 0.1F, 12};

  //Cegid
  curveCegid->markerFilterType=PoCurve::X_LIST;
  curveCegid->raiseFilterType=PoCurve::X_LIST;
  curveCegid->markerXList.setValues(0, 3, maxMinValues);
  curveCegid->raiseXList.setValues(0, 2, maxMinValues);
  curveCegid->raiseThreshold=(float)VALUE_MINY;
  curveCegid->set("markerApp.material", MARKER_COLOR);
  curveCegid->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList[CEGID]->diffuseColor.setValue(colorList[0]);
  materialRaisePoint[CEGID]->diffuseColor.setValue(colorList[0]);
  curveCegid->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  //Ibm
  maxMinValues[0]=9;
  maxMinValues[1]=11;
  maxMinValues[2]=5;
  curveIbm->markerXList.setValues(0, 3, maxMinValues);
  curveIbm->raiseXList.setValues(0, 2, maxMinValues);
  curveIbm->raiseThreshold=(float)VALUE_MINY;
  curveIbm->markerFilterType=PoCurve::X_LIST;
  curveIbm->raiseFilterType=PoCurve::X_LIST;
  
  curveIbm->set("markerApp.material", MARKER_COLOR);
  curveIbm->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList[IBM]->diffuseColor.setValue(colorList[1]);
  materialRaisePoint[IBM]->diffuseColor.setValue(colorList[1]);
  curveIbm->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  //Sligos
  maxMinValues[0]=9.9F;
  maxMinValues[1]=11.4F;
  maxMinValues[2]=5.8F;
  curveSligos->markerXList.setValues(0, 3, maxMinValues);
  curveSligos->raiseXList.setValues(0, 2, maxMinValues);
  curveSligos->raiseThreshold=(float)VALUE_MINY;
  curveSligos->raiseFilterType=PoCurve::X_LIST;
  curveSligos->markerFilterType=PoCurve::X_LIST;
  curveSligos->set("markerApp.material", MARKER_COLOR);
  curveSligos->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList[SLIGOS]->diffuseColor.setValue(colorList[2]);
  materialRaisePoint[SLIGOS]->diffuseColor.setValue(colorList[2]);
  curveSligos->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  curveGroup->addChild(curveCegid);
  curveGroup->addChild(curveIbm);
  curveGroup->addChild(curveSligos);
  return curveGroup;
}/*-------------------------------------------------------------------*/

SoSeparator*
putTextAtPosition(float posX, float posY, float posZ,
		  SoNode *textNode, PbDomain *domain)
{
  SoSeparator *textSep       = new SoSeparator;
  SoTransform *textTransform = new SoTransform;
  SoTransform *domainTransform;

  float xposition, yposition, zposition;
  SbVec3f scaleFactor;

  domainTransform=domain->getTransform();

  scaleFactor=domainTransform->scaleFactor.getValue();

  // convert with domain
  xposition=posX*scaleFactor[0];
  yposition=posY*scaleFactor[1];
  zposition=posZ*scaleFactor[2];

  textTransform->rotation.setValue(SbVec3f(1, 0, 0), -0.2F);
  textTransform->translation.setValue(xposition, yposition, zposition);

  textSep->addChild(textTransform);
  textSep->addChild(textNode);

  return textSep;
}/*-------------------------------------------------------------------*/

SoSeparator*
createTitle(const char *title, float yposition, PbDomain *domain)
{
  SoSeparator *textSep;
  SoMaterial  *titleMtl  = new SoMaterial;
  SoText3     *text      = new SoText3;
  SoFont      *titleFont = new SoFont;
  
  float xmin, ymin, zmin; 
  float xmax, ymax, zmax;
  float xposition, zposition;

  domain->getDomain(xmin, ymin, zmin, xmax, ymax, zmax);

  // convert with domain
  xposition=(xmax-xmin)/2+xmin;
  zposition=zmin;

  text->parts=SoText3::ALL;
  text->string.setValue(title);
  text->justification=SoText3::CENTER;

  textSep=putTextAtPosition(xposition, yposition, zposition, text, domain);

  titleMtl->diffuseColor.setHSVValue(0.15F, 0.50F, 1.0F);

#ifdef _WIN32
  titleFont->name.setValue("Arial");
#else
  titleFont->name.setValue("Times-Roman");
#endif

  titleFont->size.setValue(2);

  textSep->insertChild(titleMtl, 0);
  textSep->insertChild(titleFont, 0);

  return textSep;
}/*-------------------------------------------------------------------*/

SoSeparator*
createText (const char *label, float posX, float posY, float posZ,
	    SoText2::Justification justification,
	    PbDomain *domain, SbColor color)
{
  SoSeparator *textSep;
  SoText2     *text     = new SoText2;
  SoMaterial  *textMtl  = new SoMaterial;
  SoFont      *textFont = new SoFont;

  text->string.setValue(label);
  text->justification=justification;

  textSep=putTextAtPosition(posX, posY, posZ, text, domain);

  textMtl->diffuseColor.setValue(color);

#ifdef _WIN32
  textFont->name.setValue("Arial");
#else
  textFont->name.setValue("Times-Roman");
#endif

  textFont->size.setValue(15);

  textSep->insertChild(textMtl, 0);
  textSep->insertChild(textFont, 0);

  return textSep;
}/*-------------------------------------------------------------------*/


PoArrow*
createOneArrow(int numPoints, SbVec2f *points, PbDomain *domain,
	       SbColor color, const char *pattern)
{
  SoMaterial *arrowMtl;
  
  // define color for xAxis and yAxis
  PoArrow *arrow=new PoArrow(numPoints, points, 
			     PoArrow::DIRECT_TRIANGLE, 
			     PoArrow::NO_PATTERN);

  arrowMtl=SO_GET_PART(arrow, "material", SoMaterial);
  arrowMtl->diffuseColor.setValue(color);

  arrow->patternHeight=0.02F;
  arrow->patternWidth=0.04F;
  arrow->set("bodyApp.drawStyle", "lineWidth 0.2");
  arrow->set("bodyApp.drawStyle", "style LINES");
  arrow->set("bodyApp.drawStyle", pattern);
  arrow->setDomain(domain);
  return arrow;
}/*-------------------------------------------------------------------*/

SoSeparator*
createAllArrow(PbDomain *domain)
{
  SoSeparator *arrowSep=new SoSeparator;

#define MIN_COLOR_HSV_ARROW 1.0F, 1.0F, 0.85F
#define MAX_COLOR_HSV_ARROW 0.15F, 0.30F, 1.0F

  // It 's the point for arrow
  const float minPtsX1=12.5;
  const float minPtsY1=400;
  const float minPtsX2=13;
  const float minPtsY2=400;

  const float maxPtsX1=5;  
  const float maxPtsY1=650;
  const float maxPtsX2=5;  
  const float maxPtsY2=650;

  PoArrow *arrow1, *arrow2, *arrow3, *arrow4;
  SbVec2f pointsMax[3];
  SbVec2f pointsMin[3];
  SbColor colorArrow;
  SoSeparator *textMin, *textMax;

  pointsMin[1].setValue(minPtsX1, minPtsY1);
  pointsMin[2].setValue(minPtsX2, minPtsY2);

  pointsMax[1].setValue(maxPtsX1, maxPtsY1);
  pointsMax[2].setValue(maxPtsX2, maxPtsY2);

  searchMinMax(curveDataCegid, NB_DATA, 
	       pointsMax[0], pointsMin[0], Y_AXIS_SEARCH);

  colorArrow.setHSVValue(MAX_COLOR_HSV_ARROW);
  arrow1=createOneArrow(3, pointsMax, domain, colorArrow, DOTTED_LINE_STYLE);

  colorArrow.setHSVValue(MIN_COLOR_HSV_ARROW);
  arrow2=createOneArrow(3, pointsMin, domain, colorArrow, PLAIN_LINE_STYLE);

  searchMinMax(curveDataIbm, NB_DATA, 
	       pointsMax[0], pointsMin[0], Y_AXIS_SEARCH);

  arrow3=createOneArrow(3, pointsMin, domain, colorArrow, DOTTED_LINE_STYLE);

  textMin=createText("Min", minPtsX2+0.5f, minPtsY2, 0, SoText2::LEFT, 
		     domain, colorArrow);

  colorArrow.setHSVValue(MAX_COLOR_HSV_ARROW);
  arrow4=createOneArrow(3, pointsMax, domain, colorArrow, PLAIN_LINE_STYLE);

  textMax=createText("Max", maxPtsX2, maxPtsY2+10, 0.0f, SoText2::CENTER, 
		     domain, colorArrow);
  arrowSep->addChild(textMin);
  arrowSep->addChild(textMax);
  arrowSep->addChild(arrow1);
  arrowSep->addChild(arrow2);
  arrowSep->addChild(arrow3);
  arrowSep->addChild(arrow4);
  return arrowSep;
}/*-------------------------------------------------------------------*/

void
searchMinMax(SbVec2f *data, int numberItem, SbVec2f & max, SbVec2f & min,
	     searchMode mode)
{
  int i;
  if(numberItem<=0 || data==NULL)
    return;

  SbVec2f *dataPtr=data;
  min.setValue(dataPtr->getValue());
  max.setValue(dataPtr->getValue());
  dataPtr++;
  for(i=1;i<numberItem;i++,dataPtr++)
    if(max[mode]<(*dataPtr)[mode])
      max.setValue(dataPtr->getValue());
    else if(min[mode]>(*dataPtr)[mode])
      min.setValue(dataPtr->getValue());
}/*-------------------------------------------------------------------*/

SoAnnotation*
createLegend(int numberItems, const char **nameList, 
	     SbColor *colorList, PbMiscTextAttr *textAtt)
{
  if(numberItems<=0 || nameList==NULL)
    return NULL;

  SoAnnotation *annotationLegend=new SoAnnotation;

  itemLegend=new PoItemLegend(SbVec2f(-0.99F, -0.25F), SbVec2f(-0.65F, 0.25F),
  			      numberItems, nameList);

  itemLegend->titleString.setValue("Companies");
  itemLegend->titleFontSize = 1;
  itemLegend->titleVisibility = PoItemLegend::VISIBILITY_ON;
  itemLegend->boxVisibility = PoItemLegend::VISIBILITY_OFF;
  itemLegend->lineVisibility = PoItemLegend::VISIBILITY_ON;

  itemLegend->lineColor.setValues(0, numberItems, colorList);

  itemLegend->setMiscTextAttr(textAtt);
  itemLegend->set("backgroundBorderApp.material", "diffuseColor 1 0 0");

  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;
  annotationLegend->addChild(pickStyle) ;

  SoMaterial *legendMtl;
  legendMtl=SO_GET_PART(itemLegend, "backgroundApp.material", SoMaterial);
  legendMtl->diffuseColor.setHSVValue(0.45F, 0.39F, 0.30F);

  annotationLegend->addChild(itemLegend);
  return annotationLegend;
}/*-------------------------------------------------------------------*/

//Create Dialog to control parameters
PoDialogColorControl*
createDialog( const char** nameList, SoXtPlaneViewer* viewer, SoSeparator* curveSep, PbDomain* domainNewAxis, SoNode* rootToApply )
{
  SbColor colList[NB_CURVE+2];

  for ( size_t i = 0; i < NB_CURVE; i++ )
    colList[i]=*(materialCurveList[i]->diffuseColor.getValues(0));

  colList[NB_CURVE]=viewer->getBackgroundColor();

  SoMaterial *colorLegendMtl=SO_GET_PART(itemLegend, "backgroundApp.material", SoMaterial);    
  colList[NB_CURVE+1]=*(colorLegendMtl->diffuseColor.getValues(0));

  PoDialogColorControl* dialogControl =
    new PoDialogColorControl( NULL, "Tools", NB_CURVE + 2, nameList, 0, (SbColor*) colList );

  const char* nameCurve[NB_TYPE_CURVE] = { nameCurve[SMOOTH] = strdup( "Smooth" ),
                                           nameCurve[POLYLINE] = strdup( "Polyline" ),
                                           nameCurve[HISTO] = strdup( "Histogram" ) };
  SoDialogComboBox* curveType = new SoDialogComboBox();
  curveType->label.setValue( "Curve type :" );
  curveType->items.setValues( 0, NB_TYPE_CURVE, nameCurve );
  curveType->addAuditor( new CurveTypeAuditor( curveSep ) );
  dialogControl->addChild( curveType );

  SoDialogCheckBox* dollarAxis = new SoDialogCheckBox();
  dollarAxis->label.setValue( "Dollar Axis :" );
  dollarAxis->state.setValue( FALSE );
  dollarAxis->onString.setValue( "ON" );
  dollarAxis->offString.setValue( "OFF" );
  dollarAxis->addAuditor( new DollarAxisAuditor((SoSeparator *)PoBase::getByName(AXIS_NAME), domainNewAxis ) );
  dialogControl->addChild( dollarAxis );

  SoDialogCheckBox* axisEditor = new SoDialogCheckBox();
  axisEditor->label.setValue( "Axis editor :" );
  axisEditor->state.setValue( FALSE );
  axisEditor->onString.setValue( "ON" );
  axisEditor->offString.setValue( "OFF" );
  axisEditor->addAuditor( new AxisEditorAuditor );
  dialogControl->addChild( axisEditor );

  SoDialogPushButton* buttonIvFile = new SoDialogPushButton();
  buttonIvFile->label.setValue( "save iv File" );
  buttonIvFile->buttonLabel.setValue( ivFileName );
  buttonIvFile->addAuditor( new WriteAuditor( rootToApply ) );
  dialogControl->addChild( buttonIvFile  );

  SoDialogPushButton* quitButton= new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialogControl->addChild( quitButton  );

  dialogControl->updateDialogBuild( TRUE );

  dialogControl->addColorChangedCallback( (PoDialogColorControl::DialogColorControlCB*) upDateColorsCB, viewer );

  for ( int i = 0; i < NB_TYPE_CURVE; i++ )
    delete nameCurve[i];

  return dialogControl;
}/*-------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int main (int, char **argv)

{
  // Initialize Inventor and Xt
  Widget rootWindow = SoXt::init(argv[0]) ;

  if(rootWindow == NULL) exit(1) ;

  SoSeparator *root; 
  SoSeparator *group2Axis;
  SoXtPlaneViewer *viewer;
  PbMiscTextAttr textAxis, textLegend;
  PbDateFormatMapping formatMapping; // for using axisTime
  SbColor colorList[3] = { SbColor(1.0, 0.0, 1.0), 
			   SbColor(0.0, 0.0, 1.0), 
			   SbColor(1.0, 1.0, 0.0)};
  static const char *nameList[] = {"Cegid", "Ibm", "Sligos", "diagram backGround", 
				   "legend backGround"};

  PbDomain domain(0.0, 100.0, 12.0, 600);
  PbDomain axisDollarDomain(0, 100/DOLLAR_RATE, 12, 600/DOLLAR_RATE);

  // Initialize the new nodes class
  PoMeshViz::init();
  SoDialogViz::init();
   
  PoBase::setNodeWriteFormat(PoBase::UNFOLD_NODE_WRITE_FORMAT);

  axisEditor = GraphEditors::createLinearAxisEditor();

  LinearGraphAuditor* myAuditor = new LinearGraphAuditor( NULL );

  axisEditor->addAuditor(myAuditor);
  axisEditor->setSynchronizable(false);
  axisEditor->buildDialog( rootWindow, FALSE );

#ifdef _WIN32
  textAxis.setFontName("Arial");
  textLegend.setFontName("Arial");
#else
  textAxis.setFontName("Courier");
  textLegend.setFontName("Times-Italic");
#endif

  textAxis.setLineLength('/', 1);

  group2Axis=createAxisTime(0.0f, (float)VALUE_MINY, 12.0f, (float)VALUE_MAXY,
  			    "01-04-95", "01-04-96", NULL, "Change/F",
			    &formatMapping, &textAxis, &domain);

  SoSeparator *curveGroup=createCurve(&domain, colorList);
  SoSeparator *arrowGroup=createAllArrow(&domain);
  SoSeparator *titleSep=createTitle("Stock Exchange", 150, &domain);
  SoAnnotation *legend = createLegend(NB_CURVE, nameList, colorList,
				       &textLegend);

  SoSeparator *axisElements= new SoSeparator;
  axisElements->addChild(group2Axis);
  axisElements->addChild(curveGroup);
  axisElements->addChild(arrowGroup);

  SoPerspectiveCamera *cameraScene = new SoPerspectiveCamera;
  SoOrthographicCamera *cameraLegend = new SoOrthographicCamera ;
  cameraLegend->viewportMapping = SoCamera::LEAVE_ALONE ;

  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;

  root=new SoSeparator;
  root->ref();

  root->addChild(cameraScene);
  root->addChild(annoText3Property) ;
  root->addChild(titleSep);
  root->addChild(axisElements);
  root->addChild(cameraLegend) ;
  root->addChild(legend);


  viewer = new SoXtPlaneViewer(rootWindow);
  // default Background
  SbColor backgroung;
  backgroung.setHSVValue(0.62F, 0.82F, 0.24F);
  viewer->setBackgroundColor(backgroung);

  PoDialogColorControl *dialogControl=createDialog(nameList, viewer, curveGroup,
						   &axisDollarDomain, root);
  dialogControl->ref();

  SbViewportRegion myRegion(viewer->getSize());
  cameraScene->viewAll(axisElements, myRegion);
  cameraScene->position = SbVec3f(cameraScene->position.getValue()[0] - 1.2f, 
				  cameraScene->position.getValue()[1],
				  cameraScene->position.getValue()[2]) ;
  
  viewer->setSceneGraph(root);
  viewer->setTitle("Axis Demo");
  viewer->show();
  dialogControl->show();
  viewer->setCamera(cameraScene);
  SoXt::show(rootWindow);
  SoXt::mainLoop();

  delete viewer;
  axisEditor = NULL;
  delete myAuditor;

  dialogControl->unref();
  root->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();
  return 0;
}/*-------------------------------------------------------------------*/

void
initDataCurve()
{
   // CEGID data
  curveDataCegid[0].setValue(0,   487);
  curveDataCegid[1].setValue(1,   450);
  curveDataCegid[2].setValue(2,   421);
  curveDataCegid[3].setValue(5,   410);
  curveDataCegid[4].setValue(6,   381);
  curveDataCegid[5].setValue(7,   374);
  curveDataCegid[6].setValue(8,   365);
  curveDataCegid[7].setValue(9,   370);
  curveDataCegid[8].setValue(10,  403);
  curveDataCegid[9].setValue(11,  464);
  curveDataCegid[10].setValue(12, 485);

  // IBM data
  curveDataIbm[0].setValue(0,   458);
  curveDataIbm[1].setValue(1,   470);
  curveDataIbm[2].setValue(2,   463);
  curveDataIbm[3].setValue(5,   526);
  curveDataIbm[4].setValue(6,   468);
  curveDataIbm[5].setValue(7,   475);
  curveDataIbm[6].setValue(8,   480);
  curveDataIbm[7].setValue(9,   447);
  curveDataIbm[8].setValue(10,  548);
  curveDataIbm[9].setValue(11,  594);
  curveDataIbm[10].setValue(12, 564);

  // Sligos data
  curveDataSligos[0].setValue(0,   413);
  curveDataSligos[1].setValue(1,   461);
  curveDataSligos[2].setValue(2,   421);
  curveDataSligos[3].setValue(5,   455);
  curveDataSligos[4].setValue(6,   464);
  curveDataSligos[5].setValue(7,   426);
  curveDataSligos[6].setValue(8,   400);
  curveDataSligos[7].setValue(9,   407);
  curveDataSligos[8].setValue(10,  395);
  curveDataSligos[9].setValue(11,  471);
  curveDataSligos[10].setValue(12, 460);
}/*-------------------------------------------------------------------*/

// update color from slider changed
void
upDateColorsCB( SoXtPlaneViewer* viewer, PoDialogColorControl* dialog )
{
  int itemCurrent = dialog->getCurrentItem();
  SbColor color = dialog->getCurrentColor();

  switch ( itemCurrent )
  {
  case NB_CURVE:
    viewer->setBackgroundColor( color );
    break;
  case NB_CURVE + 1:
  {
    SoMaterial* colorLegendMtl = SO_GET_PART( itemLegend, "backgroundApp.material", SoMaterial );
    colorLegendMtl->diffuseColor.setValue( color );
    break;
  }
  default:
    itemLegend->lineColor.set1Value( itemCurrent, color.getValue() );
    materialRaisePoint[itemCurrent]->diffuseColor.setValue( color );
    materialCurveList[itemCurrent]->diffuseColor.setValue( color );
  }
}/*-------------------------------------------------------------------*/
