#if defined(_WIN32)
#pragma warning(disable: 4996) // Disable PoXt deprecation warning
#endif

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMatrixTransform.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/nodes/SoLinearProfile.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoProfileCoordinate2.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

#include <Inventor/sensors/SoTimerSensor.h>

#include <MeshViz/graph/PbDomain.h>
#include <MeshViz/graph/PbMiscTextAttr.h>

#include <MeshViz/graph/PoBase.h>
#include <MeshViz/graph/PoLinearAxis.h>
#include <MeshViz/graph/PoLogAxis.h>
#include <MeshViz/graph/PoArrow3.h>

#include <DialogViz/SoDialogVizAll.h>

#include "PoDialogColorControl.h"

static const float MINRADIUS=1;
static const float MAXRADIUS=4.7F;
static const float MINHEIGHT=4.5F;
static const float MAXHEIGHT=8;
static const int   SPIRALNUMBER=6;

#define NPTMAX 1000 
#define CURVE_ORIGIN_MTL_NAME "OriginMtl"
#define CURVE_X_MTL_NAME      "XMtl"
#define CURVE_Z_MTL_NAME      "ZMtl"
#define ARROW_NAME            "arrowName"
#define POINT_ARROW           52

enum colorID
{
  CURVE_ORIGIN      = 0,
  CURVE_X           = 1,
  CURVE_Z           = 2,
  NB_COLOR_CURVE    = 3,
  VIEWER_BACKGROUND = 3,
  NB_ITEMS_CHOICE   = 4
};

const SbString IV_FILE_NAME( "axisLogDemo.iv" );

// compute height of point for spiral
static float height(float minH, float maxH, int pointNumber, int currentStep);
static float radius(float minR, float maxR, int pointNumber, int currentStep);
static float angle(int spiralNumber, int pointNumber, int currentStep);

static SoSeparator *createSpiral(const char *nameControlPointsNode,
				 const SbColor *colorList, int spiralNumber,
				 float minHeight, float maxHeight, 
				 float minR, float maxR, int pointNumber=NPTMAX);

static void initAxis(PoCartesianAxis *axis, const char *title,
		     PbMiscTextAttr *textAtt, PbDomain *domain);

static SoSeparator *createAxisBox(PbMiscTextAttr *textAtt, PbDomain *domain);

static SoSeparator *createTitle(const char *title, float angle, float size);


static PoDialogColorControl *createDialog(const char **nameList, 
					  const SbColor *color,
					  SoXtExaminerViewer *viewer,
					  SoNode *rootToApply);

//Callback from slider Color
static void upDateColorsCB( SoXtExaminerViewer* viewer, PoDialogColorControl* dialog );
// Callback from dialog Box

class QuitAuditor : public SoDialogPushButtonAuditor
{
  public:
  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    exit( 0 );
  }
};

class WriteAuditor : public SoDialogPushButtonAuditor
{
  SoNode* m_root;

public:
  WriteAuditor(SoNode* root) : m_root(root)
  {}

  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    SoWriteAction myAction;

    myAction.getOutput()->openFile( IV_FILE_NAME );
    myAction.getOutput()->setBinary( FALSE );
    myAction.apply( m_root );
    myAction.getOutput()->closeFile();
  }
};

class AnimateAuditor : public SoDialogCheckBoxAuditor
{
  SoTimerSensor* m_timer;

public:
  AnimateAuditor( SoTimerSensor* timer )
    : m_timer( timer )
  { }

  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    if( cpt->state.getValue() )
      m_timer->schedule();
    else
      m_timer->unschedule();
  }
};

/*-------------------------------------------------------------------*/

float 
height(float minH, float maxH, int pointNumber, int currentStep)
{
  float delta = fabs(maxH-minH)/(float)(pointNumber-1);
  delta = sqrt(minH + (float)currentStep*delta);
  return delta; 
}/*-------------------------------------------------------------------*/

float 
radius(float minR, float maxR, int pointNumber, int currentStep)
{
  float delta = fabs(maxR-minR)/(float)(pointNumber-1);
  delta = minR + (float)currentStep*delta;
  return delta;
}/*-------------------------------------------------------------------*/

float 
angle(int spiralNumber, int pointNumber, int currentStep)
{
  return (float)((currentStep*spiralNumber*2.0f) * 3.1415927) / ((float)(pointNumber-1)) ;
}/*-------------------------------------------------------------------*/

SoSeparator*
createOneArrow(int numPoints, SbVec3f *points, SbColor color, 
	       const char *label, PbDomain *domain)
{
  SoSeparator *arrowSep=new SoSeparator;
  SoMaterial *arrowMtl;
  
  // define color for xAxis and yAxis
  PoArrow3 *arrow=new PoArrow3(numPoints, points, 
			       PoArrow3::DIRECT_TRIANGLE, 
			       PoArrow3::NO_PATTERN);

  arrowMtl=SO_GET_PART(arrow, "material", SoMaterial);
  arrowMtl->diffuseColor.setValue(color);

  arrow->patternHeight=0.05F;
  arrow->patternWidth=0.1F;
  arrow->set("bodyApp.drawStyle", "style LINES");
  arrow->set("bodyApp.drawStyle", "lineWidth 2");
  arrow->setDomain(domain);
  arrow->setName(ARROW_NAME);
  arrowSep->addChild(arrow);

  SoTranslation *textTranslate= new SoTranslation;
  SoText2     *text     = new SoText2;
  SoMaterial  *textMtl  = new SoMaterial;
  SoFont      *textFont = new SoFont;

  text->string.setValue(label);
  textTranslate->translation.setValue(points[numPoints-1][0], 
				      points[numPoints-1][1]+0.5f,
				      points[numPoints-1][2]);
  textMtl->diffuseColor.setValue(color);

#ifdef _WIN32
  textFont->name.setValue("Arial");
#else
  textFont->name.setValue("Times-Roman");
#endif
  textFont->size.setValue(15);

  arrowSep->addChild(domain->getTransform());
  arrowSep->addChild(textTranslate);
  arrowSep->addChild(textFont);
  arrowSep->addChild(textMtl);
  arrowSep->addChild(text);

  return arrowSep;
}/*-------------------------------------------------------------------*/

SoSeparator *
createSpiral(const char *nameControlPointsNode, const SbColor *colorList,
	      int spiralNumber, float minHeight, float maxHeight, 
	      float minR, float maxR, int pointNumber)
{
  int i;
  float r, t;
  SbVec3f *points;
  
  SoSeparator   *curveSep = new SoSeparator;
  SoSeparator   *curveSepOrigin  = new SoSeparator;
  SoSeparator   *curveSepX       = new SoSeparator;
  SoSeparator   *curveSepZ       = new SoSeparator;
  SoMaterial    *curveMtl        = new SoMaterial;
  SoMaterial    *curveXMtl       = new SoMaterial;
  SoMaterial    *curveZMtl       = new SoMaterial;
  SoCoordinate3 *controlPtsCurve = new SoCoordinate3;
  SoLineSet     *curve           = new SoLineSet;
  SoDrawStyle   *draw            = new SoDrawStyle;

  curveMtl->setName(CURVE_ORIGIN_MTL_NAME);
  curveXMtl->setName(CURVE_X_MTL_NAME);
  curveZMtl->setName(CURVE_Z_MTL_NAME);
  controlPtsCurve->setName(nameControlPointsNode);

  draw->lineWidth=3;
  draw->style=SoDrawStyle::LINES;

  pointNumber =(NPTMAX < pointNumber ? NPTMAX : pointNumber);
  points=new SbVec3f[pointNumber];

  for(i=0;i<pointNumber;i++) 
    {
      r = radius(minR, maxR, pointNumber, i);
      t = angle(spiralNumber, pointNumber, i);
      points[i][0] = r * cos(t) + 5.f;
      points[i][1] = r * sin(t) + 5.f;
      points[i][2] = height(minHeight, maxHeight, pointNumber, i);
    } 

  // Spiral
  controlPtsCurve->point.setValues(0, pointNumber, (const SbVec3f *)points);
  curve->numVertices = pointNumber;
  curveMtl->diffuseColor.setValue(colorList[CURVE_ORIGIN].getValue());
  curveSepOrigin->addChild(draw);
  curveSepOrigin->addChild(curveMtl);
  curveSepOrigin->addChild(controlPtsCurve);
  curveSepOrigin->addChild(curve);


  // x axis projection
  SoMatrixTransform *transformX=new SoMatrixTransform;
  transformX->matrix.setValue(1, 0, 0, 0,
			      0, 0, 0, 0,
			      0, 0, 1, 0,
			      0, 0, 0, 1);

  curveXMtl->diffuseColor.setValue(colorList[CURVE_X].getValue());
  curveSepX->addChild(draw);
  curveSepX->addChild(curveXMtl);
  curveSepX->addChild(transformX);
  curveSepX->addChild(controlPtsCurve);
  curveSepX->addChild(curve);

  // z axis projection
  SoMatrixTransform *transformZ=new SoMatrixTransform;
  transformZ->matrix.setValue(1, 0, 0, 0,
			      0, 1, 0, 0,
			      0, 0, 0, 0,
			      0, 0, 0, 1);

  curveZMtl->diffuseColor.setValue(colorList[CURVE_Z].getValue());
  curveSepZ->addChild(draw);
  curveSepZ->addChild(curveZMtl);
  curveSepZ->addChild(transformZ);
  curveSepZ->addChild(controlPtsCurve);
  curveSepZ->addChild(curve);

  curveSep->addChild(curveSepOrigin);
  curveSep->addChild(curveSepX);
  curveSep->addChild(curveSepZ);

  delete []points;
  return curveSep;
}/*-------------------------------------------------------------------*/

void
initAxis(PoCartesianAxis *axis, const char *title,
	 PbMiscTextAttr *textAtt, PbDomain *domain)
{
  SoMaterial *axisMtl;

  axis->setMiscTextAttr(textAtt);
  axis->setDomain(domain);
  axis->titleVisibility = PoAxis::VISIBILITY_ON;

  axis->gridVisibility=PoAxis::VISIBILITY_ON;
  axis->arrowVisibility=PoAxis::VISIBILITY_OFF;
  axis->gradFontSize = 0.06F;
  axis->set("mainGradGridApp.drawStyle", "style LINES");
  axis->set("mainGradGridApp.drawStyle", "lineWidth 0.2");

  if(axis->type.getValue()!=PoCartesianAxis::ZX && 
      axis->type.getValue()!=PoCartesianAxis::ZY)
    {
      axis->titlePosition = PoAxis::TITLE_MIDDLE;
      axis->gradPath = PoAxis::PATH_RIGHT;
      axis->titlePath = PoAxis::PATH_RIGHT;
    }
  else
    {
      axis->titlePosition = PoAxis::TITLE_END;
      axis->titlePath = PoAxis::PATH_DOWN;
      if(axis->start.getValue()==SbVec3f(0.0, 0.0, 0.0))
	axis->gradPosition = PoAxis::GRAD_BELOW;
    }

  axis->titleString.setValue(title);
  axis->titleFontSize = 0.11F;

  // grid
  // color define
  axis->set("appearance.material", "diffuseColor 1 1 1");
  axis->set("bodyApp.drawStyle", "style LINES");
  axis->set("bodyApp.drawStyle", "lineWidth 2.0");
  axisMtl=SO_GET_PART(axis, "bodyApp.material", SoMaterial);
  axisMtl->diffuseColor.setHSVValue(0.56F, 0.25F, 1);
}/*-------------------------------------------------------------------*/

SoSeparator*
createAxisBox(PbMiscTextAttr *textAtt, PbDomain *domain)
{
  PoLinearAxis *xAxis1;
  PoLinearAxis *xAxis2;
  PoLinearAxis *yAxis1;
  PoLinearAxis *yAxis2;
  PoLogAxis    *zAxis1;
  PoLogAxis    *zAxis2;
  SoSeparator  *groupAxis=new SoSeparator;

  float xMin, yMin, zMin, xMax, yMax, zMax;

  domain->getDomain(xMin, yMin, zMin, xMax, yMax, zMax);

  xAxis1 = new PoLinearAxis(SbVec3f(xMin, yMin, zMin), xMax, PoLinearAxis::XZ);
  initAxis(xAxis1, "X", textAtt, domain);
  xAxis1->gridLengthGradOtherSide.setValue(zMax);
  xAxis1->gradVisibility = PoAxis::VISIBILITY_OFF;
  xAxis1->titleVisibility = PoAxis::VISIBILITY_OFF;
  xAxis1->gradPosition = PoAxis::GRAD_BELOW;

  xAxis2 = new PoLinearAxis(SbVec3f(xMin, yMax, zMin), xMax, PoLinearAxis::XY);
  xAxis2->gridLengthGradOtherSide.setValue(yMax);
  initAxis(xAxis2, "X", textAtt, domain);
  xAxis2->gradPosition = PoAxis::GRAD_ABOVE;
  xAxis2->gradPath = PoAxis::PATH_LEFT;
  xAxis2->titlePath = PoAxis::PATH_LEFT;

  yAxis1 = new PoLinearAxis(SbVec3f(xMin, yMin, zMin), yMax, PoLinearAxis::YX);
  initAxis(yAxis1, "Y",textAtt, domain);
  yAxis1->gridVisibility=PoAxis::VISIBILITY_OFF;

  yAxis2 = new PoLinearAxis(SbVec3f(xMax, yMin, zMin), yMax, PoLinearAxis::YX);
  initAxis(yAxis2, "Y",textAtt, domain);
  yAxis2->gridLengthGradOtherSide.setValue(yMax);
  yAxis2->gradPosition = PoAxis::GRAD_BELOW;

  zAxis1 = new PoLogAxis(SbVec3f(xMin, yMin, zMin), zMax, PoLogAxis::ZX);
  zAxis1->gridLengthGradOtherSide.setValue(yMax);
  initAxis(zAxis1, "Z", textAtt, domain);

  zAxis2 = new PoLogAxis(SbVec3f(xMax, yMin, zMin), zMax, PoLogAxis::ZX);
  initAxis(zAxis2, "Z", textAtt, domain);
  zAxis2->gridLengthGradOtherSide.setValue(xMax);

  groupAxis->addChild(xAxis1);
  groupAxis->addChild(yAxis1);
  groupAxis->addChild(zAxis1);
  groupAxis->addChild(xAxis2);
  groupAxis->addChild(yAxis2);
  groupAxis->addChild(zAxis2);
  return groupAxis;
}/*-------------------------------------------------------------------*/

SoSeparator *
createTitle(const char *title, float angle, float size)
{
  SoSeparator  *textSep       = new SoSeparator;
  SoText3      *text          = new SoText3;
  SoFont       *titleFont     = new SoFont;
  SoTransform  *textTransform = new SoTransform;
  SoDirectionalLight *light   = new SoDirectionalLight;

  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;
  textSep->addChild(pickStyle) ;

  light->direction.setValue(1, 1, -0.8F);
 
  textSep->addChild(titleFont);
  textSep->addChild(light);

  SoMaterial *myMaterial = new SoMaterial;
  SbColor colors[3];
  // diffuse
  colors[0].setHSVValue(0.15F, 0.50F, 0.91F);
  colors[1].setHSVValue(0.15F, 0.73F, 1.0F);
  colors[2].setHSVValue(0.15F, 0.73F, 1.0F);
  myMaterial->diffuseColor.setValues(0, 3, colors);
  textSep->addChild(myMaterial);

  // Specify a beveled cross-section for the text
  SoProfileCoordinate2 *myProfileCoords = 
    new SoProfileCoordinate2;
  SbVec2f coords[4];
  coords[0].setValue(0.00, 0.00);
  coords[1].setValue(size/16, size/16);
  coords[2].setValue(size/8,  size/16);
  coords[3].setValue(3*size/16, 0.00);
  myProfileCoords->point.setValues(0, 4, coords);
  textSep->addChild(myProfileCoords);

  SoLinearProfile *myLinearProfile = new SoLinearProfile;
  int32_t index[4];
  index[0] = 0;
  index[1] = 1;
  index[2] = 2;
  index[3] = 3;
  myLinearProfile->index.setValues(0, 4, index);
  textSep->addChild(myLinearProfile);

  // Set the material binding to PER_PART
  SoMaterialBinding *myMaterialBinding = new SoMaterialBinding;
  myMaterialBinding->value.setValue(SoMaterialBinding::PER_PART);
  textSep->addChild(myMaterialBinding);

  SoSeparator *textSep2       = new SoSeparator;
  textSep->addChild(textSep2);

  text->parts=SoText3::ALL;
  text->string.setValue(title);
  text->justification=SoText3::CENTER;

  textTransform->rotation.setValue(SbVec3f(1, 0, 0), angle);
  textTransform->translation.setValue(0, 0.85F, 0.0F);

#ifdef _WIN32
  titleFont->name.setValue("Arial");
#else
  titleFont->name.setValue("Times-Roman");
#endif
  titleFont->size.setValue(size);

  textSep2->addChild(textTransform);
  textSep2->addChild(text);

  return textSep;
}/*-------------------------------------------------------------------*/

void
timerCallBack(SoCoordinate3 *data, SoTimerSensor *)
{
  if(!data) return;
  static int   spiralNumber=8;
  static float minHeight=MINHEIGHT;
  static int   step=-1;

  spiralNumber+=step;

  if(spiralNumber<2 || spiralNumber>8)
    {
      spiralNumber-=step*2;
      step=0-step;
    }

  minHeight+=step*0.75f;
  SbVec3f *points=(SbVec3f *)data->point.getValues(0);
  int pointNumber=data->point.getNum();
  int i;	
  for(i=0;i<pointNumber;i++) 
    points[i][2] = height(minHeight, MAXHEIGHT, pointNumber, i);

  data->point.setValues(0, pointNumber, (const SbVec3f *)points);

  // update arrow position
  PoArrow3 *arrow=(PoArrow3 *)SoNode::getByName(ARROW_NAME);
  if(!arrow) return;

  SbVec3f *pointsArrow=(SbVec3f *)arrow->point.getValues(0);
  pointsArrow[0].setValue(points[POINT_ARROW].getValue());
  arrow->point.setValues(0, arrow->point.getNum(), 
			 (const SbVec3f *)pointsArrow);
}/*-------------------------------------------------------------------*/

PoDialogColorControl*
createDialog(const char **nameListCurves, const SbColor *colorCurves,
	     SoXtExaminerViewer *viewer, SoNode *rootToApply)
{
  SbColor colList[NB_ITEMS_CHOICE];
  char *nameList[NB_ITEMS_CHOICE];
  int i;
  for(i=0; i<NB_COLOR_CURVE;i++)
    {
      nameList[i]=(char *)nameListCurves[i];
      colList[i]=colorCurves[i];
    }
  nameList[VIEWER_BACKGROUND]=strdup("Background");
  colList[VIEWER_BACKGROUND]=viewer->getBackgroundColor();

  PoDialogColorControl* dialogControl =
    new PoDialogColorControl( NULL, "Tools", NB_ITEMS_CHOICE, (const char**) nameList, 0, (SbColor*) colList );

  delete nameList[VIEWER_BACKGROUND];

  SoTimerSensor* timer = new SoTimerSensor( (SoSensorCB*) timerCallBack, SoNode::getByName( "ControlPoints" ) );
  timer->setInterval( 0.2 );

  SoDialogCheckBox* animate = new SoDialogCheckBox();
  animate->label.setValue( "Animate :" );
  animate->state.setValue( FALSE );
  animate->onString.setValue( "ON" );
  animate->offString.setValue( "OFF" );
  animate->addAuditor( new AnimateAuditor( timer ) );
  dialogControl->addChild( animate );

  SoDialogPushButton* buttonIvFile = new SoDialogPushButton();
  buttonIvFile->label.setValue( "save iv File" );
  buttonIvFile->buttonLabel.setValue( IV_FILE_NAME );
  buttonIvFile->addAuditor( new WriteAuditor( rootToApply ) );
  dialogControl->addChild( buttonIvFile  );

  SoDialogPushButton* quitButton= new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialogControl->addChild( quitButton  );

  dialogControl->updateDialogBuild( TRUE );

  return dialogControl;
}/*-------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int main (int, char **argv)
{  
  // Initialize Inventor and Xt
  static const char *nameList[NB_COLOR_CURVE] = {"Origin Curve",
						  "Curve Projection X",
						  "Curve Projection Z"};

  const static SbColor colorsCurves[NB_COLOR_CURVE]={ SbColor(1, 0, 0), 
						      SbColor(0, 1, 1), 
						      SbColor(0, 1, 0)};
  Widget rootWindow = SoXt::init(argv[0]) ;
  if(rootWindow == NULL) exit(1) ;

  SoXtExaminerViewer *viewer;
  PbMiscTextAttr textAxis;
#ifdef _WIN32
  textAxis.setFontName("Arial");
#else
  textAxis.setFontName("Courier");
#endif
  textAxis.setLineLength('/', 1);
  viewer = new SoXtExaminerViewer(rootWindow);

  SbColor backgroung;
  backgroung.setHSVValue(0.62F, 0.35F, 0.24F);
  viewer->setBackgroundColor(backgroung);

  PbDomain domain(0.0, 0.0, 0.0, 10.0, 10.0, 3);

  // Initialize the new nodes class
  PoMeshViz::init();
  SoDialogViz::init();
   
  PoBase::setNodeWriteFormat(PoBase::UNFOLD_NODE_WRITE_FORMAT);

  // create 7 axis box
  SoSeparator *axisBox;
  axisBox=createAxisBox(&textAxis, &domain);

  // createScene graph
  SoPerspectiveCamera *camera3D = new SoPerspectiveCamera;
  SoOrthographicCamera *camera2D = new SoOrthographicCamera ;
  camera2D->viewportMapping = SoCamera::LEAVE_ALONE ;

  SoSeparator *scene2D = new SoSeparator ;
  SoSeparator *scene3D = new SoSeparator ;

  scene2D->addChild(camera2D) ;
  scene2D->addChild(createTitle("Logarithmic Axis", 0.0F, 0.19F));
  
  SoSeparator *root=new SoSeparator;
  root->ref();
  root->addChild(scene3D);
  root->addChild(scene2D) ;

  scene3D->addChild(camera3D) ;
  int pointNumber=100;
  float r = radius(MINRADIUS, MAXRADIUS, pointNumber, POINT_ARROW);
  float t = angle(SPIRALNUMBER, pointNumber, POINT_ARROW);

  SbVec3f pointsArrow[3];
  pointsArrow[0][0] = r * cos(t) + 5.f;
  pointsArrow[0][1] = r * sin(t) + 5.f;
  pointsArrow[0][2] = height(MINHEIGHT, MAXHEIGHT, pointNumber, POINT_ARROW);
  pointsArrow[1].setValue(pointsArrow[0][0], 10, 1);
  pointsArrow[2].setValue(pointsArrow[0][0], 15, 1);

  scene3D->addChild(createOneArrow(3, pointsArrow, 
				   SbColor(0, 0.5F, 0.9F), "Spiral",
				   &domain));
  
  SoTransformSeparator *spiralsSep = new SoTransformSeparator ;
  scene3D->addChild(spiralsSep) ;
  spiralsSep->addChild((SoNode *)domain.getTransform());
  spiralsSep->addChild(createSpiral("ControlPoints", colorsCurves,
				    SPIRALNUMBER,  MINHEIGHT,  MAXHEIGHT, 
				    MINRADIUS, MAXRADIUS, pointNumber));
  scene3D->addChild(axisBox); 

  //dialog handle
  PoDialogColorControl* dialog = createDialog(nameList, colorsCurves, viewer, root);
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsCB, viewer);

  // viewer config
  SbViewportRegion myRegion(viewer->getSize());
  camera3D->orientation.setValue(1.3F, 2.2F, 3.2F, 1.5F);
  camera3D->viewAll(axisBox, myRegion);
  viewer->setSceneGraph(root);
  viewer->setTitle("3D Axis logarithmic Demo");

  dialog->show();
  viewer->show();
  viewer->setCamera(camera3D);

  SoXt::show(rootWindow);
  SoXt::mainLoop();

  delete viewer;
  root->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();

  return 0;
}/*-------------------------------------------------------------------*/

// update color from slider changed
void
upDateColorsCB( SoXtExaminerViewer* viewer, PoDialogColorControl* dialog )
{
  int itemCurrent = dialog->getCurrentItem();
  SbColor color = dialog->getCurrentColor();

  SoMaterial *mtl=NULL;
  switch(itemCurrent)
    {
    case CURVE_ORIGIN:
      mtl=(SoMaterial *)SoNode::getByName(CURVE_ORIGIN_MTL_NAME);
      break;
    case CURVE_X:
      mtl=(SoMaterial *)SoNode::getByName(CURVE_X_MTL_NAME);
      break;
    case CURVE_Z:
      mtl=(SoMaterial *)SoNode::getByName(CURVE_Z_MTL_NAME);
      break;
    case VIEWER_BACKGROUND:
      viewer->setBackgroundColor(color);
      return;
    }
  mtl->diffuseColor.setValue(color.getValue());
}/*-------------------------------------------------------------------*/
