// **************************************************************************
//  Original source code :
//    3DMSJava362\demos\advanced\graph\Grapher.java
//
//  Ported to C++ by Nicolas Daguise, 04/12/2002
//
//  Converted with DialogViz 26 mar 2003
// **************************************************************************

#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/engines/SoCalculator.h> 
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h> 
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoAnnoText3Property.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <DialogViz/SoDialogVizAll.h>

#include <MeshViz/PoMeshViz.h>
#include <MeshViz/graph/PoAxis.h>
#include <MeshViz/graph/PoCartesianAxis.h> 
#include <MeshViz/graph/PoCurve.h>
#include <MeshViz/graph/PoLinearAxis.h>
#include <MeshViz/nodes/PoDomain.h>
#include <MeshViz/view/PoSceneView.h>
#include <Inventor/helpers/SbFileHelper.h>

#if !defined(_WIN32) && !defined(__APPLE__)
#  include <values.h>  
#endif

#define TEXT_AMIN_ID 0
#define TEXT_AMAX_ID 1
#define TEXT_XFA_ID 2
#define TEXT_YFA_ID 3
#define TEXT_POINTS_ID 4
#define PUSH_GO_ID 5

static SoXtPlaneViewer *viewer;
static PoDomain *domain;
static PoDomain *domain_axis;
static PoSceneView *axisView;
static PoSceneView *graphView;
static SoOrthographicCamera *axisCamera;
static SoOrthographicCamera *graphCamera;
static SbString formula1;
static SbString formula2;
static SbVec2f *points;
static float xmin, xmax, ymin, ymax;
static PoLinearAxis *xAxis;
static PoLinearAxis *xAxisUp;
static PoLinearAxis *yAxis;
static PoLinearAxis *yAxisUp;

void 
initData(float minVal, float maxVal, int numVal)
{
  SbString formula;
  SoCalculator *calc = new SoCalculator();
  formula = "oa=";
  formula += formula1;

  calc->expression.set1Value(0, formula);
  SoMFFloat *outputVal1 = (SoMFFloat*)
    SoDB::createGlobalField("outputVal1", SoMFFloat::getClassTypeId());
  outputVal1->connectFrom(&(calc->oa));
  SoMFFloat *outputVal2 = NULL;
  if (formula2.getLength() != 0) {
    outputVal2 = (SoMFFloat*)
      SoDB::createGlobalField("outputVal2", SoMFFloat::getClassTypeId());
    formula = "ob=";
    formula += formula2;
    calc->expression.set1Value(1, formula);
    outputVal2->connectFrom(&(calc->ob));
  }
  points = new SbVec2f[numVal];
  float x, y;
  float step = (maxVal-minVal)/(numVal-1);
  float a = minVal;
  xmin = FLT_MAX;
  xmax = FLT_MIN;
  ymin = FLT_MAX;
  ymax = FLT_MIN;
  for (int i = 0; i < numVal; i++, a += step) {
    calc->a.setValue(a);
    if (formula2.getLength() != 0) {
      calc->b.setValue(a);
      x = outputVal1->getValues(0)[0];
      y = outputVal2->getValues(0)[0];
    }
    else {
      x = a;
      y = outputVal1->getValues(0)[0];
    }
    xmin = (xmin < x ? xmin : x);
    xmax = (xmax > x ? xmax : x);
    ymin = (ymin < y ? ymin : y);
    ymax = (ymax > y ? ymax : y);
    points[i] = SbVec2f(x, y);
  }
  //domain = new PoDomain();
  domain->min.setValue(SbVec3f(xmin, ymin, 0));
  domain->max.setValue(SbVec3f(xmax, ymax, 0));
  domain->transformType.setValue(PoDomain::TRANSFORM_01);
  domain_axis->min.setValue(SbVec3f(xmin, ymin, 0));
  domain_axis->max.setValue(SbVec3f(xmax, ymax, 0));
  domain_axis->transformType.setValue(PoDomain::TRANSFORM_01);
}

SoAnnotation *
buildAxis()
{
  SoAnnoText3Property *textProp = new SoAnnoText3Property();
  textProp->renderPrintType.setValue(SoAnnoText3Property::RENDER2D_PRINT_RASTER);

  SoTransform *trans = new SoTransform();
  trans->scaleFactor.setValue(0.8f, 0.8f, 1.0f);
  trans->translation.setValue(0.1f, 0.1f, 0.0f);

  xAxis = new PoLinearAxis();
  xAxis->start.setValue(SbVec3f(xmin, ymin, 0));
  xAxis->end.setValue(xmax);
  xAxis->type.setValue(PoCartesianAxis::XY);
  xAxis->arrowVisibility.setValue(PoAxis::VISIBILITY_OFF);
  xAxis->gridVisibility.setValue(PoAxis::VISIBILITY_ON);
  xAxis->gridLengthGradOtherSide.setValue(ymax-ymin);
  xAxis->tickSubDef.setValue(PoAxis::NUM_SUB_TICK);
  xAxis->tickNumOrPeriod.setValue((short)8);
  xAxis->set((char *)"appearance.material", (char *)"diffuseColor 1 1 1");
  xAxis->set((char *)"subGradGridApp.drawStyle", (char *)"style INVISIBLE");
  xAxis->set((char *)"mainGradGridApp.material", (char *)"diffuseColor .3 .3 .3");

  xAxisUp = new PoLinearAxis();
  xAxisUp->start.setValue(SbVec3f(xmin, ymax, 0));
  xAxisUp->end.setValue(xmax);
  xAxisUp->type.setValue(PoCartesianAxis::XY);
  xAxisUp->arrowVisibility.setValue(PoAxis::VISIBILITY_OFF);
  xAxisUp->gridVisibility.setValue(PoAxis::VISIBILITY_OFF);
  xAxisUp->gradVisibility.setValue(PoAxis::VISIBILITY_OFF);
  xAxisUp->tickVisibility.setValue(PoAxis::VISIBILITY_OFF);
  xAxisUp->set((char *)"appearance.material", (char *)"diffuseColor 1 1 1");

  yAxis = new PoLinearAxis();
  yAxis->start.setValue(SbVec3f(xmin, ymin, 0));
  yAxis->end.setValue(ymax);
  yAxis->type.setValue(PoCartesianAxis::YX);
  yAxis->arrowVisibility.setValue(PoAxis::VISIBILITY_OFF);
  yAxis->gridVisibility.setValue(PoAxis::VISIBILITY_ON);
  yAxis->gridLengthGradOtherSide.setValue(xmax-xmin);
  yAxis->tickSubDef.setValue(PoAxis::NUM_SUB_TICK);
  yAxis->tickNumOrPeriod.setValue((short)8);
  yAxis->set((char *)"appearance.material", (char *)"diffuseColor 1 1 1");
  yAxis->set((char *)"subGradGridApp.drawStyle", (char *)"style INVISIBLE");
  yAxis->set((char *)"mainGradGridApp.material", (char *)"diffuseColor .3 .3 .3");

  yAxisUp = new PoLinearAxis();
  yAxisUp->start.setValue(SbVec3f(xmax, ymin, 0));
  yAxisUp->end.setValue(ymax);
  yAxisUp->type.setValue(PoCartesianAxis::YX);
  yAxisUp->arrowVisibility.setValue(PoAxis::VISIBILITY_OFF);
  yAxisUp->gridVisibility.setValue(PoAxis::VISIBILITY_OFF);
  yAxisUp->gradVisibility.setValue(PoAxis::VISIBILITY_OFF);
  yAxisUp->tickVisibility.setValue(PoAxis::VISIBILITY_OFF);
  yAxisUp->set((char *)"appearance.material", (char *)"diffuseColor 1 1 1");

  SoTranslation *transTitle = new SoTranslation();
  transTitle->translation.setValue(1, .97f, 0);

  SoFont *fontTitle = new SoFont();
  fontTitle->size.setValue(14);

  SoText2 *title = new SoText2();
  title->justification.setValue(SoText2::RIGHT);

  SbString formula = "x = ";
  if (formula2.getLength() != 0) {
    formula += formula1;
    title->string.set1Value(0, formula);
    formula = "y = ";
    formula += formula2;
    title->string.set1Value(1, formula);
  }
  else {
    formula = "y=";
    formula += formula1;
    title->string.set1Value(0, formula);
  }

  SoTransformSeparator *titleSep = new SoTransformSeparator();
  titleSep->addChild(transTitle);
  titleSep->addChild(fontTitle);
  titleSep->addChild(title);

  SoAnnotation *root = new SoAnnotation();
  root->addChild(textProp);
  root->addChild(titleSep);
  root->addChild(trans);
  root->addChild(domain_axis);
  root->addChild(xAxis);
  root->addChild(yAxis);
  root->addChild(xAxisUp);
  root->addChild(yAxisUp);

  axisCamera->position.setValue(0.5f, 0.5f, 1.0f);
  axisCamera->pointAt(SbVec3f(0.5f, 0.5f, 0.0f));
  axisCamera->height.setValue(1);

  return root;
}

SoAnnotation *
buildScene(int numPts)
{
  SoAnnotation *root = new SoAnnotation();

  PoCurve *curve = new PoCurve();
  curve->point.setValues(0, numPts, points);
  curve->set((char *)"curvePointApp.material", (char *)"diffuseColor [1 0 0]");
  curve->set((char *)"curvePointApp.drawStyle", (char *)"lineWidth 2");

  root->addChild(domain);
  root->addChild(curve);

  graphCamera->position.setValue(.5f, .5f, 1);
  graphCamera->pointAt(SbVec3f(.5f, .5f, 0));
  graphCamera->height.setValue(1);

  return root;
}

void
updateView(int numPts)
{
  axisView->setPart("scene", buildAxis());
  graphView->setPart("scene", buildScene(numPts));
}

void
nodeSensorCB(void *, SoSensor *)
{
  float pos[3];
  graphCamera->position.getValue().getValue(pos[0], pos[1], pos[2]);  
  float height = graphCamera->height.getValue();

  float newXmin = (xmax - xmin) * (pos[0] - height / 2.0f) + xmin;
  float newXmax = (xmax - xmin) * (pos[0] + height / 2.0f) + xmin;
  float newYmin = (ymax - ymin) * (pos[1] - height / 2.0f) + ymin;
  float newYmax = (ymax - ymin) * (pos[1] + height / 2.0f) + ymin;
 
  domain_axis->min.setValue(SbVec3f(newXmin, newYmin, 0));
  domain_axis->max.setValue(SbVec3f(newXmax, newYmax, 0));
        
  xAxis->start.setValue(SbVec3f(newXmin, newYmin, 0));
  xAxis->end.setValue(newXmax);
  xAxis->gridLengthGradOtherSide.setValue(newYmax-newYmin);
        
  xAxisUp->start.setValue(SbVec3f(newXmin, newYmax, 0));
  xAxisUp->end.setValue(newXmax);

  yAxis->start.setValue(SbVec3f(newXmin, newYmin, 0));
  yAxis->end.setValue(newYmax);
  yAxis->gridLengthGradOtherSide.setValue(newXmax-newXmin);

  yAxisUp->start.setValue(SbVec3f(newXmax, newYmin, 0));
  yAxisUp->end.setValue(newYmax);  
}

SoNode*
initScene()
{
  SoSeparator *root = new SoSeparator();
  root->ref();

  domain = new PoDomain();
  domain_axis = new PoDomain();

  // Build the axis view
  axisView = new PoSceneView();
  axisView->viewportOrigin.setValue(0.0f, 0.0f);
  axisView->viewportSize.setValue(1.0f, 1.0f);
  axisCamera = new SoOrthographicCamera();
  axisCamera->viewportMapping.setValue(SoCamera::LEAVE_ALONE);
  axisView->setPart("cameraKit.camera", axisCamera);

  // Build the curve view
  graphView = new PoSceneView();
  graphView->viewportOrigin.setValue(0.1f, 0.1f);
  graphView->viewportSize.setValue(0.8f, 0.8f);
  graphCamera = new SoOrthographicCamera();
  graphCamera->viewportMapping.setValue(SoCamera::LEAVE_ALONE);
  graphView->setPart("cameraKit.camera", graphCamera);

  SoDirectionalLight *dirLight = new SoDirectionalLight();
  dirLight->direction.setValue(0.2f, -0.2f, -1.0f);
  root->addChild(dirLight);
  root->addChild(axisView);
  root->addChild(graphView);
  //root->addChild(new SoCube);

  SoEventCallback *myEvent = new SoEventCallback;
  root->addChild(myEvent);
        
  viewer->setSceneGraph(root);
  viewer->setCamera(graphCamera);
  viewer->setTransparencyType (SoGLRenderAction::SORTED_OBJECT); 
  viewer->viewAll();
  viewer->show();

  SoNodeSensor *nodeSensor = new SoNodeSensor;
  nodeSensor->setFunction(nodeSensorCB);
  nodeSensor->attach((SoNode *)graphCamera);

  return root;
}

class MyWindowAuditor : public SoDialogAuditor
{
public:
  void dialogEditText(SoDialogEditText *et);
};

void 
MyWindowAuditor::dialogEditText(SoDialogEditText *et)
{
  static float text_amin_float = 0;
  static float text_amax_float = (float)6.28;
  static SbString text_xfa_str = "cos(4*a)*cos(a)";
  static SbString text_yfa_str = "cos(4*a)*sin(a)";
  static int text_points_int = 250;
  
  if (et->auditorID.getValue() == SbString("TEXT_AMIN_ID"))
    text_amin_float = (float)atof(et->editText.getValue().getString());
  else if (et->auditorID.getValue() == SbString("TEXT_AMAX_ID"))
    text_amax_float = (float)atof(et->editText.getValue().getString());
  else if (et->auditorID.getValue() == SbString("TEXT_XFA_ID"))
    text_xfa_str = et->editText.getValue().getString();
  else if (et->auditorID.getValue() == SbString("TEXT_YFA_ID"))
    text_yfa_str = et->editText.getValue().getString();
  else if (et->auditorID .getValue() == SbString("TEXT_POINTS_ID"))
    text_points_int = atoi(et->editText.getValue().getString());
  
  formula1 = (text_xfa_str != formula1) ? text_xfa_str : formula1;
  formula2 = (text_yfa_str != formula2) ? text_yfa_str : formula2;
  
  initData(text_amin_float, text_amax_float, text_points_int);
  updateView(text_points_int);
}

#include <Inventor/SoWinApp.h>

int
main(int, char **)
{
  // Initialize Inventor and Xt
  Widget topLevelWidget = SoXt::init("Grapher");

  // Create the root of the scene graph 
  PoMeshViz::init();
  SoDialogViz::init();
  SbString FileName= SbFileHelper::expandString( "$OIVHOME/examples/source/MeshViz/graph/grapherDemoGUI.iv" ) ;
  SoTopLevelDialog *myWindow = (SoTopLevelDialog *)
  SoDialogViz::loadFromFile(FileName);
  myWindow->addAuditor(new MyWindowAuditor());
  myWindow->buildDialog(topLevelWidget, TRUE);
  myWindow->show();

  SoDialogCustom *custom = (SoDialogCustom *)
    myWindow->searchForAuditorId("planeViewer");
  viewer = new SoXtPlaneViewer(custom->getWidget(), "", TRUE);
  viewer->setHeadlight(FALSE);
  viewer->setDecoration(FALSE);
  viewer->setBackgroundColor(SbColor(0, 0, 0));

  formula1 = "cos(4*a)*cos(a)";
  formula2 = "cos(4*a)*sin(a)";

  SoNode* root = initScene();
  
  initData(0, 6.28f, 250);

  updateView(250);

  SoXt::show(topLevelWidget);
  SoXt::mainLoop();

  delete viewer;
  root->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();

  return 0;
}


