#ifndef _WIN32
#include <Xm/Form.h>
#include <X11/Intrinsic.h>
#endif

#if defined(_WIN32)
#pragma warning(disable: 4996) // Disable PoXt deprecation warning
#endif

#include <Inventor/SbColor.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoAnnoText3Property.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoEvents.h>
#include <Inventor/actions/SoWriteAction.h>

#include <MeshViz/graph/PbDomain.h>
#include <MeshViz/graph/PbMiscTextAttr.h>
#include <MeshViz/graph/PbLinearDataMapping.h>

#include <MeshViz/graph/PoBase.h>
#include <MeshViz/graph/PoCurve.h>
#include <MeshViz/graph/PoMultipleHistogram.h>
#include <MeshViz/graph/PoLinearAxis.h>
#include <MeshViz/graph/PoGenAxis.h>
#include <MeshViz/graph/PoItemLegend.h>

#include "PoDialogColorControl.h"

#include <DialogViz/SoDialogVizAll.h>

#define DOTTED_LINE_STYLE "linePattern 0xf0f0"
#define PLAIN_LINE_STYLE "linePattern 0xffff"

enum COMPANY
{
  CEGID       = 0,
  DASSAULT    = 1,
  IBM         = 2,
  SLIGOS      = 3,
  NB_COMPANY = 4
};

enum MONTH
{
  APRIL95     = 0,
  SEPTEMBER95 = 1,
  APRIL96     = 2,
  JANUARY96   = 3,
  NB_MONTH    = 4
};

enum BACKGROUND
{
  AREAHISTO_BACKGROUND = NB_COMPANY+NB_MONTH,
  AREACURVES_BACKGROUND   = AREAHISTO_BACKGROUND+1,
  LEGEND_BACKGROUND = AREACURVES_BACKGROUND+1,
  NB_ITEMS_CHOICE   = LEGEND_BACKGROUND+1
};

#define CURVE_SWITCH_NAME "CurveSwitchName"
#define ROOT_HISTO_NAME  "RootHistoName"
#define ROOT_CURVE_NAME  "RootCurveName"

static PoMultipleHistogram* createHisto(float minX, float minY, float maxX,
					const float *data, const char **companyName, 
					const char **monthList, const SbColor *colorsList,
					PbDomain *domain, PbMiscTextAttr *textAtt,
					PbLinearDataMapping *mapping=NULL);

static SoSeparator* createCurve(const float *data, const SbColor *colorList, PbDomain *domain);

static PoLinearAxis* createAxis(float minX, float minY, float maxY,PbMiscTextAttr *textAtt, 
				PbDomain *domain);

static SoSeparator* createTitle(const char *title, float yposition, float size,
				float angle, PbDomain *domain);

// put a text at position in touch with a PbDomain
static SoSeparator* putTextAtPosition(float posX, float posY, float posZ,
				      float angle, SoNode *textNode, PbDomain *domain);

static PoDialogColorControl* createDialog( Widget parent,
                                           const char** nameListCompany,
                                           SbColor* colorCompany,
                                           const char** nameListMonth,
                                           SbColor* colorMonth,
                                           SoXtRenderArea* areaHisto,
                                           SoXtRenderArea* areaCurves,
                                           PoItemLegend* legend,
                                           SoNode* rootToApply,
                                           PoMultipleHistogram* histoToSwitch );

static PoItemLegend* createLegend(int numberItems, const char **nameList, 
				  const SbColor *colorList, PbMiscTextAttr *textAtt) ;
				  
// callback function for coloring

// update color from slider changed background viewer
static void upDateColorsAreaHistoCB(SoXtRenderArea *areaHisto,
				    PoDialogColorControl *dialog);

// update color from slider changed background areaCurves
static void upDateColorsAreaCurvesCB(SoXtRenderArea *areaCurves,
				     PoDialogColorControl *dialog);

// update color from slider changed for curve
static void upDateColorsCurveCB(SoSeparator *sepCurves,
				PoDialogColorControl *dialog);

// update color from slider changed for curve
static void upDateColorsHistoCB(PoMultipleHistogram *histo,
				PoDialogColorControl *dialog);

// update color from slider changed for legend color
static void upDateColorsLegendCB( PoItemLegend* legend, PoDialogColorControl* dialog );

class QuitAuditor : public SoDialogPushButtonAuditor
{
  public:
  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    exit( 0 );
  }
};

class AutoViewingAuditor : public SoDialogCheckBoxAuditor
{
  PoMultipleHistogram* m_histo;
  SoSwitch* m_switchCurvesHisto;
public:
  AutoViewingAuditor( PoMultipleHistogram* histo, SoSwitch* switchCurvesHisto )
    : m_histo( histo )
    , m_switchCurvesHisto( switchCurvesHisto )
  {
  }

  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    if ( cpt->state.getValue() )
    {
      m_switchCurvesHisto->whichChild.setValue( SO_SWITCH_NONE );
      m_histo->valuePath.setValue( PoMultipleHistogram::PATH_RIGHT );
      m_histo->representation.setValue( PoMultipleHistogram::PERCENTAGE );
    }
    else
    {
      m_switchCurvesHisto->whichChild.setValue( SO_SWITCH_ALL );
      m_histo->valuePath.setValue( PoMultipleHistogram::PATH_UP );
      m_histo->representation.setValue( PoMultipleHistogram::JUXTAPOSED );
    }
  }
};

/*-------------------------------------------------------------------*/

SoSeparator*
putTextAtPosition(float posX, float posY, float posZ, float angle,
		  SoNode *textNode, PbDomain *domain)
{
  SoSeparator *textSep       = new SoSeparator;
  SoTransform *textTransform = new SoTransform;
  SoTransform *domainTransform;
  SoDirectionalLight *light  = new SoDirectionalLight;

  float xposition, yposition, zposition;
  SbVec3f scaleFactor;

  domainTransform=domain->getTransform();

  scaleFactor=domainTransform->scaleFactor.getValue();

  // convert with domain
  xposition=posX*scaleFactor[0];
  yposition=posY*scaleFactor[1];
  zposition=posZ*scaleFactor[2];

  textTransform->rotation.setValue(SbVec3f(1, 0, 0), angle);
  textTransform->translation.setValue(xposition, yposition, zposition);

  light->direction.setValue( xposition,  yposition-5,  zposition-10);

  textSep->addChild(light);
  textSep->addChild(textTransform);
  textSep->addChild(textNode);

  return textSep;
}/*-------------------------------------------------------------------*/

SoSeparator*
createTitle(const char *title, float yposition, float size,
	    float angle, PbDomain *domain)
{
  SoSeparator *textSep;
  SoMaterial  *titleMtl  = new SoMaterial;
  SoText3     *text      = new SoText3;
  SoFont      *titleFont = new SoFont;
 
  float xmin, ymin, zmin; 
  float xmax, ymax, zmax;
  float xposition, zposition;

  domain->getDomain(xmin, ymin, zmin, xmax, ymax, zmax);

  // convert with domain
  xposition=(xmax-xmin)/2+xmin;
  zposition=zmin;

  text->parts=SoText3::ALL;
  text->string.setValue(title);
  text->justification=SoText3::CENTER;

  textSep=putTextAtPosition(xposition, yposition, zposition, angle,
			    text, domain);

  titleMtl->diffuseColor.setHSVValue(0.15F, 0.50F, 1.0F);

#ifdef _WIN32
  titleFont->name.setValue("Arial");
#else
  titleFont->name.setValue("Times-Roman");
#endif
  titleFont->size.setValue(size);

  textSep->insertChild(titleMtl, 0);
  textSep->insertChild(titleFont, 0);
  return textSep;
}/*-------------------------------------------------------------------*/

PoLinearAxis*
createAxis(float minX, float minY, float maxY,
	   PbMiscTextAttr *textAtt, PbDomain *domain)
{
  // create axis
  PoLinearAxis *yAxis = new 
    PoLinearAxis(SbVec3f(minX, minY, 0.0), maxY, PoCartesianAxis::YX, 0.0);

  yAxis->setMiscTextAttr(textAtt);
  yAxis->setDomain(domain);
  yAxis->gradFontSize = 0.05F;
  yAxis->titleString.setValue("Rate/Exchange");
  yAxis->titleVisibility = PoLinearAxis::VISIBILITY_ON;
  yAxis->arrowVisibility = PoLinearAxis::VISIBILITY_ON;
  yAxis->titleFontSize = 0.06F;
  yAxis->set("appearance.material", "diffuseColor 1 1 1") ;
  yAxis->set("bodyApp.drawStyle", "lineWidth 2.0") ;
  SoMaterial *axisMtl;
  axisMtl=SO_GET_PART(yAxis, "bodyApp.material", SoMaterial);
  axisMtl->diffuseColor.setHSVValue(0.56F, 0.25F, 1);
  return yAxis;
}/*-------------------------------------------------------------------*/

PoGenAxis*
createGenAxis(float minX, float minY, float maxX, const char **gradList,
	      PbMiscTextAttr *textAtt, PbDomain *domain)
{
  // create axis
  PoGenAxis *xAxis = new 
    PoGenAxis(SbVec3f(minX, minY, 0.0), maxX, PoGenAxis::XY, gradList, NB_MONTH);

  xAxis->setMiscTextAttr(textAtt);
  xAxis->setDomain(domain);
  xAxis->gradAddStringVisibility = TRUE;
  xAxis->gradAddString.setValue(".");
  xAxis->titlePath = PoGenAxis::PATH_DOWN;
  xAxis->titlePosition = PoGenAxis::TITLE_END;
  xAxis->gradFontSize = 0.04F;
  xAxis->titleString.setValue("Month");
  xAxis->titleVisibility = PoLinearAxis::VISIBILITY_ON;
  xAxis->arrowVisibility = PoLinearAxis::VISIBILITY_OFF;
  xAxis->titleFontSize = 0.06F;
  xAxis->set("appearance.material", "diffuseColor 1 1 1") ;
  xAxis->set("bodyApp.drawStyle", "lineWidth 2.0") ;
  SoMaterial *axisMtl;
  axisMtl=SO_GET_PART(xAxis, "bodyApp.material", SoMaterial);
  axisMtl->diffuseColor.setHSVValue(0.56F, 0.25F, 1);
  return xAxis;
}/*-------------------------------------------------------------------*/

PoMultipleHistogram*
createHisto(float minX, float minY, float maxX,
	    const float *data, const char **companyName, 
	    const char **monthList, const SbColor *colorsList,
	    PbDomain *domain,PbMiscTextAttr *textAtt,
	    PbLinearDataMapping *mapping)
{
  PoMultipleHistogram *histo; 
  histo = new 
    PoMultipleHistogram(SbVec2f(minX, minY), maxX, PoMultipleHistogram::X, 
			NB_COMPANY, NB_MONTH, data, companyName, 
			monthList);
  histo->setDomain(domain);

  histo->set("barApp.material", "diffuseColor 0. 1. 0.") ;
  histo->set("valueTextApp.material", "diffuseColor 1 1 1");
  histo->set("nameTextApp.material", "diffuseColor 1 1 1") ;
  histo->set("sumTextApp.material", "diffuseColor 0.8 0 0");
  if(mapping) {
    histo->setDataMapping(mapping);
    histo->coloring = PoMultipleHistogram::DATA_MAPPING;
  }

  histo->setMiscTextAttr(textAtt);
  histo->barHeight = 500.0;
  histo->valuePath = PoMultipleHistogram::PATH_UP;

  histo->color.setValues(0, NB_MONTH, colorsList);

  histo->valueFontSize = 0.05F;
  histo->nameFontSize = 0.045F;
  histo->sumFontSize = 0.06F;

  return histo;
}/*-------------------------------------------------------------------*/

SoSeparator*
createCurve(const float *data, const SbColor *colorList, PbDomain *domain)
{
  PoCurve *curveCegid, *curveDassault, *curveIbm, *curveSligos;

  SoSeparator *curveGroup=new SoSeparator;

#define MARKER_STYLE_SIZE "pointSize 5.0"
#define CURVE_STYLE_SIZE  "lineWidth 2.0"
#define MARKER_COLOR      "diffuseColor 1 0 0"   // red

  SbVec2f curveData[NB_COMPANY][NB_MONTH];
  SoMaterial *materialCurveList;
  float xpos=2;
  int i;
  for(i=0;i<NB_COMPANY;i++,xpos+=2)
    for(int j=0;j<NB_MONTH;j++,xpos+=2)
      curveData[i][j].setValue(xpos, data[i*NB_MONTH+j]);

  curveCegid=new PoCurve(NB_MONTH, (const SbVec2f *)curveData[CEGID]);
  curveCegid->setDomain(domain);
  curveDassault=new PoCurve(NB_MONTH, (const SbVec2f *)curveData[DASSAULT]);
  curveDassault->setDomain(domain);
  curveIbm=new PoCurve(NB_MONTH, (const SbVec2f *)curveData[IBM]);
  curveIbm->setDomain(domain);
  curveSligos=new PoCurve(NB_MONTH, (const SbVec2f *)curveData[SLIGOS]);
  curveSligos->setDomain(domain);

  //Cegid
  materialCurveList=SO_GET_PART(curveCegid, "curvePointApp.material", SoMaterial);
  curveCegid->markerFilterType=PoCurve::ALL_POINTS;
  curveCegid->set("markerApp.material", MARKER_COLOR);
  curveCegid->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList->diffuseColor.setValue(colorList[CEGID]);
  curveCegid->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  //Dassault
  materialCurveList=SO_GET_PART(curveDassault, "curvePointApp.material", SoMaterial);
  curveDassault->markerFilterType=PoCurve::ALL_POINTS;
  curveDassault->set("markerApp.material", MARKER_COLOR);
  curveDassault->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList->diffuseColor.setValue(colorList[DASSAULT]);
  curveDassault->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  //IBM
  materialCurveList=SO_GET_PART(curveIbm, "curvePointApp.material", SoMaterial);
  curveIbm->markerFilterType=PoCurve::ALL_POINTS;
  curveIbm->set("markerApp.material", MARKER_COLOR);
  curveIbm->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList->diffuseColor.setValue(colorList[IBM]);
  curveIbm->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  //Sligos
  materialCurveList=SO_GET_PART(curveSligos, "curvePointApp.material", SoMaterial);
  curveSligos->markerFilterType=PoCurve::ALL_POINTS;
  curveSligos->set("markerApp.material", MARKER_COLOR);
  curveSligos->set("markerApp.drawStyle", MARKER_STYLE_SIZE);
  materialCurveList->diffuseColor.setValue(colorList[SLIGOS]);
  curveSligos->set("curvePointApp.drawStyle", CURVE_STYLE_SIZE);

  curveGroup->addChild(curveCegid);
  curveGroup->addChild(curveDassault);
  curveGroup->addChild(curveIbm);
  curveGroup->addChild(curveSligos);
  return curveGroup;
}/*-------------------------------------------------------------------*/

SoSeparator*
createCurveWithoutHisto(const float *data, const SbColor *colorList, 
			PbDomain *domain)
{
  int i;
  SoSeparator *sepCurves=createCurve(data, colorList, domain);
  int childNumber=sepCurves->getNumChildren();
  PoCurve *curve;
  float *value;
  for(i=0;i<childNumber;i++) {
    curve= (PoCurve *)sepCurves->getChild(i);
    curve->set("raisePointApp.material", "diffuseColor 0 1 0");
    curve->set("raisePointApp.drawStyle", DOTTED_LINE_STYLE);
    curve->raiseFilterType=PoCurve::ALL_POINTS;
    for(int j=0;j<curve->point.getNum();j++) {
      value=(float *)curve->point.getValues(j)->getValue();
      value[0]=j*2+1.0f;
    }
  }
  return sepCurves;
}/*-------------------------------------------------------------------*/

PoItemLegend*
createLegend(int numberItems, const char **nameList, 
	     const SbColor *colorList, PbMiscTextAttr *textAtt)
{
  if(numberItems<=0 || nameList==NULL)
    return NULL;

  PoItemLegend *itemLegend;
  itemLegend=new PoItemLegend(SbVec2f(-0.99F, -0.25F), SbVec2f(-0.65F, 0.25F),
  			      numberItems, nameList);

  itemLegend->titleString.setValue("Month");
  itemLegend->titleFontSize = 0.07F;
  itemLegend->titleVisibility = PoItemLegend::VISIBILITY_ON;
  itemLegend->boxVisibility = PoItemLegend::VISIBILITY_ON;
  itemLegend->lineVisibility = PoItemLegend::VISIBILITY_OFF;
  itemLegend->boxColor.setValues(0, numberItems, colorList);

  itemLegend->boxRatio=1.2F;

  itemLegend->setMiscTextAttr(textAtt);
  itemLegend->set("backgroundBorderApp.material", "diffuseColor 0.9 0.5 0.5");
  itemLegend->set("titleApp.drawStyle", "style FILLED");

  SoMaterial *legendMtl;
  legendMtl=SO_GET_PART(itemLegend, "backgroundApp.material", SoMaterial);
  legendMtl->diffuseColor.setHSVValue(0.40F, 0.29F, 0.25F);

  return itemLegend;
}/*-------------------------------------------------------------------*/

PoDialogColorControl*
createDialog( Widget parent,
              const char** nameListCompany,
              SbColor* colorCompany,
              const char** nameListMonth,
              SbColor* colorMonth,
              SoXtRenderArea* areaHisto,
              SoXtRenderArea* areaCurves,
              PoItemLegend* legend,
              SoNode* /*rootToApply*/,
              PoMultipleHistogram* histoToSwitch )
{
  SbColor colList[NB_ITEMS_CHOICE];
  char *nameList[NB_ITEMS_CHOICE];
  int i;

  for(i=0; i<NB_COMPANY;i++)
    {
      nameList[i]=(char *)nameListCompany[i];
      colList[i]=colorCompany[i];
    }
  for(; i<NB_COMPANY+NB_MONTH;i++)
    {
      nameList[i]=(char *)nameListMonth[i-NB_COMPANY];
      colList[i]=colorMonth[i-NB_COMPANY];
    }

  nameList[AREAHISTO_BACKGROUND]=strdup("Histogram Background");
  colList[AREAHISTO_BACKGROUND]=areaHisto->getBackgroundColor();

  nameList[AREACURVES_BACKGROUND]=strdup("Curves Background");
  colList[AREACURVES_BACKGROUND]=areaCurves->getBackgroundColor();

  SoMaterial *colorLegendMtl=SO_GET_PART(legend, "backgroundApp.material", SoMaterial);    

  nameList[LEGEND_BACKGROUND]=strdup("Legend Background");
  colList[LEGEND_BACKGROUND]=*(colorLegendMtl->diffuseColor.getValues(0));

  PoDialogColorControl* dialogControl = new PoDialogColorControl(
    parent, "Tools", NB_ITEMS_CHOICE, (const char**) nameList, 0, (SbColor*) colList );

  delete nameList[AREAHISTO_BACKGROUND];
  delete nameList[AREACURVES_BACKGROUND];
  delete nameList[LEGEND_BACKGROUND];

  SoSwitch* switchCurvesHisto = (SoSwitch*) SoNode::getByName( CURVE_SWITCH_NAME );

  SoDialogCheckBox* representation = new SoDialogCheckBox();
  representation->label.setValue( "Representation :" );
  representation->state.setValue( FALSE );
  representation->onString.setValue( "Percentage" );
  representation->offString.setValue( "Juxtaposed" );
  representation->addAuditor( new AutoViewingAuditor( histoToSwitch, switchCurvesHisto ) );
  dialogControl->addChild( representation );

  SoDialogPushButton* quitButton= new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialogControl->addChild( quitButton  );

  dialogControl->updateDialogBuild( TRUE );

  return dialogControl;
}/*-------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int main (int, char **argv)
{
  // Initialize Inventor and Xt
  Widget rootWindow = SoXt::init(argv[0]) ;
  SoDialogViz::init();

  if(rootWindow == NULL) 
    exit(1) ;

  SoXtRenderArea *areaHisto;
  SoXtRenderArea     *areaCurves;
  PbMiscTextAttr textHisto, textLegend;
  static const char *companyList[NB_COMPANY] = {"Cegid", "Dassault",
						  "Ibm", "Sligos"};

  static const char *monthList[NB_MONTH] = {"April 95", "September 95",
					    "January 95", "April 96"};

  const static SbColor colorsHisto[NB_MONTH]={ SbColor(0.0F, 0.0F, 0.51F), 
					       SbColor(0.0F, 0.68F, 0.92F), 
					       SbColor(0.22F, 0.98F, 0.59F), 
					       SbColor(0.85F, 0.83F, 0.14F)};
  const static SbColor colorsCurve[NB_COMPANY]={SbColor(1, 0, 0), 
						 SbColor(0, 1, 0),
						 SbColor(0, 0, 1), 
						 SbColor(0, 1, 1)};
  static float data[NB_COMPANY][NB_MONTH];
  PbLinearDataMapping mapping ;
  mapping.setValues(200, SbColor(1.0, 1.0, 1.0), 564, SbColor(0.0, 1.0, 0.0));
  Widget container=NULL;

  data[CEGID][APRIL95]=487.0;
  data[CEGID][SEPTEMBER95]=410.0;
  data[CEGID][JANUARY96]=370.0;
  data[CEGID][APRIL96]=485.0;

  data[DASSAULT][APRIL95]=340.9F;
  data[DASSAULT][SEPTEMBER95]=206.5F;
  data[DASSAULT][JANUARY96]=161.0F;
  data[DASSAULT][APRIL96]=285.7F;

  data[IBM][APRIL95]=458.6F;
  data[IBM][SEPTEMBER95]=526.0F;
  data[IBM][JANUARY96]=447.1F;
  data[IBM][APRIL96]=564.0F;

  data[SLIGOS][APRIL95]=413.0;
  data[SLIGOS][SEPTEMBER95]=455.0;
  data[SLIGOS][JANUARY96]=407.0;
  data[SLIGOS][APRIL96]=460.0;

#ifndef _WIN32
  container=XtVaCreateManagedWidget("HistoDemo", xmFormWidgetClass,
				    rootWindow,
				    XmNfractionBase, 100,
				    NULL);
#else
  container = rootWindow;
#endif

  areaHisto = new SoXtRenderArea(container);

#ifndef _WIN32
  areaCurves = new SoXtRenderArea(container);
#else  
  // Under _WIN32, creating an Inventor component (a viewer in this
  // case) with no parent is not supported.  Under UNIX/X/Motif a
  // top level shell would be automatically created to be the parent,
  // but here we need to create a simple window ourselves.
  HWND hwnd;
  RECT rect;
  GetWindowRect( container, &rect );
  hwnd = CreateWindow( "button",
                       "",
		       WS_OVERLAPPEDWINDOW | BS_OWNERDRAW,
		       rect.right + 1,         // X position
		       rect.top,               // Y position
		       rect.right - rect.left, // Width
		       rect.bottom - rect.top, // Height
		       NULL,                   // Parent
		       NULL,                   // Menu
		       SoWin::getInstance(),   // App instance handle
		       NULL );                 // Window creation data
  
  areaCurves = new SoXtRenderArea( hwnd );
  ShowWindow( hwnd, SW_SHOW );
#endif

  SbColor backgroung;
  backgroung.setHSVValue(0.62F, 0.35F, 0.24F);
  areaHisto->setBackgroundColor(backgroung);

  backgroung.setHSVValue(0.61F, 0.23F, 0.38F);
  areaCurves->setBackgroundColor(backgroung);


  PbDomain domain(0.0, 100.0, (NB_MONTH+1)*2*NB_COMPANY, 600.0);
  PbDomain domainCurveArea(0.0, -300.0, NB_MONTH*2-1, 600.0);

  // Initialize the new nodes class
  PoMeshViz::init();
   
  PoBase::setNodeWriteFormat(PoBase::UNFOLD_NODE_WRITE_FORMAT);

#ifdef _WIN32
  textHisto.setFontName("Arial");
  textLegend.setFontName("Arial");
#else
  textHisto.setFontName("Courier");
  textLegend.setFontName("Times-Roman");
#endif
  
  textHisto.setLineLength('/', 1);
  
  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;

  // createScene graph
  SoPerspectiveCamera *cameraAreaCurves = new SoPerspectiveCamera;
  SoPerspectiveCamera *cameraScene = new SoPerspectiveCamera;

  SoSeparator *curvesAreaCurves=
    createCurveWithoutHisto((const float *)data, colorsCurve, &domainCurveArea);
  SoSeparator *rootCurve=new SoSeparator;
  rootCurve->ref();
  rootCurve->addChild(annoText3Property) ;
  rootCurve->setName(ROOT_CURVE_NAME);

  rootCurve->addChild(cameraAreaCurves);
  rootCurve->addChild(createGenAxis(1, 0, 7, monthList,
				    &textHisto, &domainCurveArea));

  rootCurve->addChild(createAxis(0, 0, 600, &textHisto, &domainCurveArea));
  rootCurve->addChild(curvesAreaCurves);
  rootCurve->addChild(createTitle("Rate evolution", 700, 0.6F, 0.2F, &domainCurveArea));

  // Create Histogram areaHisto
  PoMultipleHistogram *histo=
    createHisto(0, 0, (NB_MONTH+1)*2*NB_COMPANY, (const float *)data,
		(const char **)companyList, (const char **)monthList, 
		(const SbColor *) colorsHisto, &domain, &textHisto);
  
  SoSeparator *curvesAreaHisto=createCurve((const float *)data, colorsCurve, &domain);

  SoSeparator *rootHisto=new SoSeparator;
  rootHisto->ref();
  rootHisto->addChild(annoText3Property) ;
  rootHisto->setName(ROOT_HISTO_NAME);

  PoItemLegend *legend=createLegend(NB_MONTH, (const char **)monthList, colorsHisto, &textLegend);
  SoAnnotation *legendAno=new SoAnnotation;
  SoSwitch *switchCurve=new SoSwitch;
  switchCurve->whichChild.setValue(SO_SWITCH_ALL);
  switchCurve->setName(CURVE_SWITCH_NAME);
  
  switchCurve->addChild(curvesAreaHisto);
  // create axis
  switchCurve->addChild(createAxis(0, 0, 600, 
				 &textHisto, &domain));
  legendAno->addChild(legend);
  rootHisto->addChild(legendAno);
  rootHisto->addChild(cameraScene);
 
  rootHisto->addChild(histo);
  rootHisto->addChild(switchCurve);
  rootHisto->addChild(createTitle("Stock Exchange", -100, 4, -0.5, &domain));

  PoDialogColorControl* dialog = createDialog( container,
                                               (const char**) companyList,
                                               (SbColor*) colorsCurve,
                                               (const char**) monthList,
                                               (SbColor*) colorsHisto,
                                               areaHisto,
                                               areaCurves,
                                               legend,
                                               rootHisto,
                                               histo );

  //callback to change colors
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsLegendCB,
				  legend);
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsHistoCB,
				  histo);
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsCurveCB,
				  curvesAreaHisto);
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsCurveCB,
				  curvesAreaCurves);
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsAreaCurvesCB,
				  areaCurves);
  dialog->addColorChangedCallback((PoDialogColorControl::DialogColorControlCB *) upDateColorsAreaHistoCB,
				  areaHisto);

#ifndef _WIN32
  Widget areaHistoWidget=areaHisto->getWidget();

  XtVaSetValues(areaHistoWidget,
  		XmNtopAttachment,    XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNleftAttachment,   XmATTACH_FORM,
  		XmNrightAttachment,  XmATTACH_POSITION,
  		XmNrightPosition,    60,
  		XmNtraversalOn, False,
  		NULL);

  Widget areaCurvesWidget=areaCurves->getWidget();
  XtVaSetValues(areaCurvesWidget,
  		XmNtopAttachment,    XmATTACH_POSITION,
		XmNtopPosition,      50,
  		XmNbottomAttachment, XmATTACH_FORM,
  		XmNleftAttachment,   XmATTACH_POSITION,
  		XmNleftPosition,     61,
  		XmNrightAttachment,  XmATTACH_POSITION,
  		XmNrightPosition,    99,
  		NULL);

  XtVaSetValues(rootWindow,
  		XmNwidth, 850,
  		XmNheight, 600,
  		NULL);
#endif

  cameraAreaCurves->viewAll(rootCurve, areaCurves->getViewportRegion()) ;
  cameraAreaCurves->scaleHeight(0.85F) ;

  cameraScene->viewAll(rootHisto, areaHisto->getViewportRegion());
  cameraScene->position.setValue(cameraScene->position.getValue()[0] - 
				 cameraScene->position.getValue()[0] * 0.2f,
				 cameraScene->position.getValue()[1],
				 cameraScene->position.getValue()[2]);
  cameraScene->scaleHeight(0.85F) ;

  areaHisto->setSceneGraph(rootHisto);
  areaCurves->setSceneGraph(rootCurve);
  areaHisto->setTitle("Histogram Demo");
  dialog->show();
  areaHisto->show();
  areaCurves->show();

  SoXt::show(rootWindow);
  SoXt::mainLoop();

  delete areaCurves;
  delete areaHisto;
  rootHisto->unref();
  rootCurve->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();

  return 0;
}/*-------------------------------------------------------------------*/

// update color from slider changed background areaHisto
void
upDateColorsAreaHistoCB( SoXtRenderArea* areaHisto, PoDialogColorControl* dialog )
{
  int itemCurrent = dialog->getCurrentItem();

  if(itemCurrent!=AREAHISTO_BACKGROUND)
    return;
  SbColor color = dialog->getCurrentColor();
  areaHisto->setBackgroundColor(color);
}/*-------------------------------------------------------------------*/

// update color from slider changed background areaCurves
void 
upDateColorsAreaCurvesCB(SoXtRenderArea *areaCurves, PoDialogColorControl *dialog)
{
  int itemCurrent = dialog->getCurrentItem();

  if(itemCurrent!=AREACURVES_BACKGROUND)
    return;
  SbColor color = dialog->getCurrentColor();
  areaCurves->setBackgroundColor(color);
}/*-------------------------------------------------------------------*/

// update color from slider changed for curve
void 
upDateColorsCurveCB(SoSeparator *sepCurves, PoDialogColorControl *dialog)
{
  int itemCurrent = dialog->getCurrentItem();

  if(itemCurrent>=NB_COMPANY)
    return;

  SbColor color = dialog->getCurrentColor();
  PoCurve *curve = (PoCurve *)sepCurves->getChild(itemCurrent);
  SoMaterial *colorCurveMtl=SO_GET_PART(curve, "curvePointApp.material", SoMaterial);
  colorCurveMtl->diffuseColor.setValue(color);
}/*-------------------------------------------------------------------*/

// update color from slider changed for legend color
void 
upDateColorsLegendCB(PoItemLegend *legend, PoDialogColorControl* dialog)
{
  int itemCurrent = dialog->getCurrentItem();

  if ( ( itemCurrent < NB_COMPANY || itemCurrent >= NB_MONTH + NB_COMPANY ) && itemCurrent != LEGEND_BACKGROUND )
    return;

  SbColor color = dialog->getCurrentColor();
  if(itemCurrent==LEGEND_BACKGROUND)
    {
      SoMaterial *colorLegendMtl=SO_GET_PART(legend, "backgroundApp.material", SoMaterial);    
      colorLegendMtl->diffuseColor.setValue(color);
      return;
    }
  itemCurrent-=NB_COMPANY;
  legend->boxColor.set1Value(itemCurrent,color.getValue());
}/*-------------------------------------------------------------------*/


// update color from slider changed for curve
void 
upDateColorsHistoCB(PoMultipleHistogram *histo, PoDialogColorControl *dialog)
{
  int itemCurrent = dialog->getCurrentItem();

  if(itemCurrent<NB_COMPANY || itemCurrent>=NB_MONTH+NB_COMPANY)
    return;

  itemCurrent-=NB_COMPANY;
  SbColor color = dialog->getCurrentColor();

  histo->color.set1Value(itemCurrent,color.getValue());
}/*-------------------------------------------------------------------*/
