#if defined(_WIN32)
#pragma warning(disable: 4996) // Disable PoXt deprecation warning
#endif

#include <Inventor/SbColor.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoLinearProfile.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoProfileCoordinate2.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoAnnoText3Property.h>
#include <Inventor/actions/SoWriteAction.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/sensors/SoTimerSensor.h>

#include <MeshViz/graph/PbDomain.h>
#include <MeshViz/graph/PbNonLinearDataMapping2.h>
#include <MeshViz/graph/PbMiscTextAttr.h>
#include <MeshViz/graph/PbIsovaluesList.h>

#include <MeshViz/graph/PoBase.h>
#include <MeshViz/graph/PoMultipleHistogram.h>
#include <MeshViz/graph/PoLinearAxis.h>
#include <MeshViz/graph/PoGenAxis.h>
#include <MeshViz/graph/PoTimeAxis.h>
#include <MeshViz/graph/PoNonLinearValueLegend2.h>
#include <MeshViz/graph/PoPointsFieldBars.h>
#include <MeshViz/graph/PoAutoCubeAxis.h>

#include <DialogViz/SoDialogVizAll.h>

#define DOTTED_LINE_STYLE "linePattern 0xf0f0"
#define PLAIN_LINE_STYLE "linePattern 0xffff"

#define IBM         0
#define SLIGOS      1
#define CEGID       2
#define DASSAULT    3
#define NB_COMPAGNY 4

#define APRIL95     0
#define SEPTEMBER95 1
#define APRIL96     2
#define JANUARY96   3
#define NB_MONTH    4

SbString ivFileName =  "histoDemo3D.iv";

#define AUTO_CUBE_AXIS_NAME  "AutoCubeAxis"
#define AXIS_BOX_SWITCH_NAME "AxisBoxSwitchName"
#define Z1_AXIS_NAME         "AxisZ1"
#define Z2_AXIS_NAME         "AxisZ2"
#define Z3_AXIS_NAME         "AxisZ3"
#define FIELD_BAR_NAME       "FieldBar"

static void initAxis(PoCartesianAxis *axis, const char *title,
		     PbMiscTextAttr *textAtt, PbDomain *domain);

static void initAutoAxis(PoCartesianAxis *axis, PbMiscTextAttr *textAtt, SbBool isXYAxis) ;

static SoSeparator* createAxisBox(const char **compagnyList, const char **monthList, 
				  PbMiscTextAttr *textAtt, PbDomain *domain);

static SoSeparator* createAutoAxisBox(const char **compagnyList, const char **monthList, 
				      PbMiscTextAttr *textAtt, PbDomain *domain);

static PoNonLinearValueLegend2* createLegend(PbMiscTextAttr *textAtt, PbNonLinearDataMapping2 *dataMapping,
					     PbIsovaluesList *isoList) ;

static SoTopLevelDialog* createDialog(SoNode *rootToApply, PbDomain *domain) ;

static SoSeparator* createTitle(const char *title, float angle, float size);

class QuitAuditor : public SoDialogPushButtonAuditor
{
  public:
  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    exit( 0 );
  }
};

class AutoViewingAuditor : public SoDialogCheckBoxAuditor
{
  SoSwitch* m_axisBox;
public:
  AutoViewingAuditor( SoSwitch* axisBox )
    : m_axisBox( axisBox )
  {
  }

  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    if ( cpt->state.getValue() )
      m_axisBox->whichChild.setValue(1) ;
    else
      m_axisBox->whichChild.setValue(0) ;

  }
};

class MonthEvolutionAuditor : public SoDialogCheckBoxAuditor
{
  SoTimerSensor* m_timer;

public:
  MonthEvolutionAuditor( SoTimerSensor* timer )
    : m_timer( timer )
  { }

  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    if( cpt->state.getValue() )
      m_timer->schedule();
    else
      m_timer->unschedule();
  }
};

class WriteAuditor : public SoDialogPushButtonAuditor
{
  SoNode* v_Root;

public:
  WriteAuditor(SoNode* root) : v_Root(root)
  {}

  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    SoWriteAction myAction;
    myAction.getOutput()->openFile( ivFileName.toLatin1() );
    myAction.getOutput()->setBinary( FALSE );
    myAction.apply( v_Root );
    myAction.getOutput()->closeFile();
  }
};

/*-------------------------------------------------------------------*/

static void
initAutoAxis(PoCartesianAxis *axis, PbMiscTextAttr *textAtt, SbBool isXYAxis)
{

  if(isXYAxis) {
    axis->marginType.setValue(PoCartesianAxis::FIXED_MARGIN) ;
    axis->marginStart.setValue(0.5) ;
    axis->marginEnd.setValue(0.5) ;
    axis->tickSubDef.setValue(PoCartesianAxis::NUM_SUB_TICK);
    axis->tickNumOrPeriod.setValue(1);
  }
  axis->titleFontSize = 0.10F ;
  axis->setMiscTextAttr(textAtt) ;
  axis->gradFontSize = 0.06F ;
  SoMaterial *axisMtl = SO_GET_PART(axis, "bodyApp.material", SoMaterial) ;
  axisMtl->diffuseColor.setHSVValue(0.56F, 0.25F, 1) ;
  axis->set("bodyApp.drawStyle", "lineWidth 3.0");

}/*-------------------------------------------------------------------*/

static void
initAxis(PoCartesianAxis *axis, const char *title,
	 PbMiscTextAttr *textAtt, PbDomain *domain)
{
  SoMaterial *axisMtl;

  axis->setMiscTextAttr(textAtt);
  axis->setDomain(domain);
  axis->titleVisibility = PoAxis::VISIBILITY_ON;
  if(axis->type.getValue()!=PoCartesianAxis::ZX && 
      axis->type.getValue()!=PoCartesianAxis::ZY)
    {
      axis->gradFontSize = 0.06F;
      axis->marginType.setValue(PoCartesianAxis::FIXED_MARGIN);
      axis->marginStart.setValue(0.5);
      axis->marginEnd.setValue(0.5);
      axis->titlePosition = PoAxis::TITLE_MIDDLE;
      axis->tickVisibility.setValue(PoCartesianAxis::VISIBILITY_ON);
      axis->tickPosition.setValue(PoCartesianAxis::TICK_GRAD_OTHER_SIDE);
      axis->tickSubDef.setValue(PoCartesianAxis::NUM_SUB_TICK);
      axis->tickNumOrPeriod.setValue(1);
      axis->tickSubLength.setValue(1);
      if(axis->start.getValue()!=SbVec3f(0.0, 0.0, 0.0))
	{ 
	  axis->gradPath = PoAxis::PATH_RIGHT;
	  axis->titlePath = PoAxis::PATH_RIGHT;
	}
      else
	{// origin axis
	  axis->gradVisibility = PoAxis::VISIBILITY_OFF;
	  axis->titleVisibility = PoAxis::VISIBILITY_OFF;
	  axis->gradPosition = PoAxis::GRAD_BELOW;
	  axis->set("subGradTickApp.drawStyle", DOTTED_LINE_STYLE);
	}
      axis->arrowVisibility = PoAxis::VISIBILITY_OFF;
    }
  else
    {
      if(axis->start.getValue()==SbVec3f(0.0, 0.0, 0.0))
	axis->titleVisibility = PoAxis::VISIBILITY_OFF;
      else
	{
	  axis->gridVisibility=PoAxis::VISIBILITY_ON;
	  axis->titlePosition = PoAxis::TITLE_END;
	  axis->titlePath = PoAxis::PATH_DOWN;
	  axis->set("mainGradGridApp.drawStyle", "style LINES");
	  axis->set("mainGradGridApp.drawStyle", "lineWidth 0.2");
	  axis->set("mainGradGridApp.drawStyle", DOTTED_LINE_STYLE);
	}
      axis->gradFontSize = 0.08F;
    }
  axis->titleString.setValue(title);
  axis->titleFontSize = 0.11F;

  // grid
  // color define
  axis->set("appearance.material", "diffuseColor 1 1 1");
  axis->set("bodyApp.drawStyle", "style LINES");
  axis->set("bodyApp.drawStyle", "lineWidth 2.0");
  axisMtl=SO_GET_PART(axis, "bodyApp.material", SoMaterial);
  axisMtl->diffuseColor.setHSVValue(0.56F, 0.25F, 1);
}/*-------------------------------------------------------------------*/

static SoSeparator*
createAutoAxisBox(const char **compagnyList, const char **monthList,
		  PbMiscTextAttr *textAtt, PbDomain *domain)
{
  int i ;

  PoAutoCubeAxis *autoCubeAxis = new 
    PoAutoCubeAxis(SbVec3f(0., 0., 0.), SbVec3f(NB_MONTH, NB_COMPAGNY, 600.), PoAutoCubeAxis::GENERALIZED,
		   PoAutoCubeAxis::GENERALIZED, PoAutoCubeAxis::LINEAR, "Month", "Companies", "Rate") ;
  autoCubeAxis->set("appearance.material", "diffuseColor 1 1 1") ;
  autoCubeAxis->set("mainGradGridApp.drawStyle", "lineWidth 0.2") ;
  autoCubeAxis->set("mainGradGridApp.drawStyle", "linePattern 0xF0F0") ;
  autoCubeAxis->set("subGradGridApp.drawStyle", "style INVISIBLE") ;
  autoCubeAxis->setName(AUTO_CUBE_AXIS_NAME) ;
  autoCubeAxis->isGridLinesXVisible = TRUE ;
  autoCubeAxis->isGridLinesYVisible = TRUE ;
  autoCubeAxis->isGridLinesZVisible = TRUE ;
  autoCubeAxis->isIntersectingGradsVisible = TRUE ;
  autoCubeAxis->setDomain(domain) ;
  
  PoGenAxis *xAxis[4] ;
  xAxis[0] = SO_GET_PART(autoCubeAxis, "xAxis03", PoGenAxis) ;
  xAxis[1] = SO_GET_PART(autoCubeAxis, "xAxis12", PoGenAxis) ;
  xAxis[2] = SO_GET_PART(autoCubeAxis, "xAxis65", PoGenAxis) ;
  xAxis[3] = SO_GET_PART(autoCubeAxis, "xAxis74", PoGenAxis) ;
  for(i=0; i < 4; i++) {
    initAutoAxis(xAxis[i], textAtt, TRUE) ;
    xAxis[i]->gradList.setValues(0, NB_MONTH, (const char**)monthList) ;
  }

  PoGenAxis *yAxis[4] ;
  yAxis[0] = SO_GET_PART(autoCubeAxis, "yAxis01", PoGenAxis) ;
  yAxis[1] = SO_GET_PART(autoCubeAxis, "yAxis76", PoGenAxis) ;
  yAxis[2] = SO_GET_PART(autoCubeAxis, "yAxis45", PoGenAxis) ;
  yAxis[3] = SO_GET_PART(autoCubeAxis, "yAxis32", PoGenAxis) ;
  for(i=0; i < 4; i++) {
    initAutoAxis(yAxis[i], textAtt, TRUE) ;
    yAxis[i]->gradList.setValues(0, NB_COMPAGNY, (const char**)compagnyList) ;      
  }
  
  PoLinearAxis *zAxis[4] ;
  zAxis[0] = SO_GET_PART(autoCubeAxis, "zAxis07", PoLinearAxis) ;
  zAxis[1] = SO_GET_PART(autoCubeAxis, "zAxis34", PoLinearAxis) ;
  zAxis[2] = SO_GET_PART(autoCubeAxis, "zAxis25", PoLinearAxis) ;
  zAxis[3] = SO_GET_PART(autoCubeAxis, "zAxis16", PoLinearAxis) ;
  for(i=0; i < 4; i++) {
    initAutoAxis(zAxis[i], textAtt, FALSE) ;
    zAxis[i]->multFactorPosition = PoLinearAxis::MULT_FACTOR_GRAD ;
  }
  SoSeparator *groupAxis = new SoSeparator ;
  groupAxis->addChild(autoCubeAxis) ;
  return groupAxis ;

}/*-------------------------------------------------------------------*/

// do a 7 axis box
static SoSeparator*
createAxisBox(const char **compagnyList, const char **monthList,
	      PbMiscTextAttr *textAtt, PbDomain *domain)
{
  PoGenAxis    *xAxis1, *xAxis2, *yAxis1, *yAxis2;
  PoLinearAxis *zAxis1, *zAxis2, *zAxis3;
  SoSeparator  *group3Axis=new SoSeparator;

  xAxis1 = new PoGenAxis(SbVec3f(0.0, 0.0, 0.0), NB_MONTH,
			 PoGenAxis::XZ, monthList, NB_COMPAGNY);
  initAxis(xAxis1, "Month", textAtt, domain);

  xAxis2 = new PoGenAxis(SbVec3f(0.0, NB_COMPAGNY, 0.0), NB_MONTH,
			 PoGenAxis::XY, monthList, NB_COMPAGNY);
  initAxis(xAxis2, "Month", textAtt, domain);
  xAxis2->gradPosition = PoAxis::GRAD_ABOVE;
  xAxis2->gradPath = PoAxis::PATH_LEFT;
  xAxis2->titlePath = PoAxis::PATH_LEFT;


  yAxis1 = new PoGenAxis(SbVec3f(0.0, 0.0, 0.0), NB_COMPAGNY,
			 PoGenAxis::YZ, compagnyList, NB_COMPAGNY);
  initAxis(yAxis1, "Companies",textAtt, domain);

  yAxis2 = new PoGenAxis(SbVec3f(NB_MONTH, 0.0, 0.0), NB_COMPAGNY,
			 PoGenAxis::YX, compagnyList, NB_COMPAGNY);
  initAxis(yAxis2, "Companies",textAtt, domain);
  yAxis2->gradPosition = PoAxis::GRAD_BELOW;

  zAxis1 = new PoLinearAxis(SbVec3f(0.0, 0.0, 0.0), 600, PoLinearAxis::ZY);
  zAxis1->setName(Z1_AXIS_NAME);
  initAxis(zAxis1, "Rate", textAtt, domain);

  zAxis2 = new PoLinearAxis(SbVec3f(0.0, NB_COMPAGNY, 0.0), 600, PoLinearAxis::ZY);
  zAxis2->setName(Z2_AXIS_NAME);
  zAxis2->gridLengthGradOtherSide.setValue(NB_MONTH);
  initAxis(zAxis2, "Rate", textAtt, domain);
  zAxis2->gradPosition = PoAxis::GRAD_BELOW;

  zAxis3 = new PoLinearAxis(SbVec3f(NB_MONTH, 0.0, 0.0), 600, PoLinearAxis::ZX);
  zAxis3->setName(Z3_AXIS_NAME);
  zAxis3->gridLengthGradOtherSide.setValue(NB_COMPAGNY);
  initAxis(zAxis3, "Rate", textAtt, domain);
  group3Axis->addChild(xAxis1);
  group3Axis->addChild(yAxis1);
  group3Axis->addChild(zAxis1);
  group3Axis->addChild(xAxis2);
  group3Axis->addChild(yAxis2);
  group3Axis->addChild(zAxis2);
  group3Axis->addChild(zAxis3);
  return group3Axis;
}/*-------------------------------------------------------------------*/

static PoNonLinearValueLegend2*
createLegend(PbMiscTextAttr *textAtt, PbNonLinearDataMapping2 *dataMapping,
	     PbIsovaluesList *isoList)
{
  PoNonLinearValueLegend2 *legend;
  legend=new PoNonLinearValueLegend2(SbVec2f(-0.99F, -0.50F), SbVec2f(-0.75F, 0.50F));

  legend->setDataMapping(dataMapping);
  legend->setIsovaluesList(isoList);
  legend->titleString.setValue("Rate");
  legend->titleFontSize = 0.075F;
  legend->titleVisibility = PoNonLinearValueLegend2::VISIBILITY_ON;
  legend->boxRatio = 2.0F;
  legend->incrementType = PoNonLinearValueLegend2::VERTICAL_INCREMENT;
  legend->numColumns.setValue(1);

  legend->setMiscTextAttr(textAtt);

  legend->boxRatio=1.5;

  legend->set("backgroundBorderApp.material", "diffuseColor 0.9 0.5 0.5");

  SoMaterial *legendMtl;
  legendMtl=SO_GET_PART(legend, "backgroundApp.material", SoMaterial);
  legendMtl->diffuseColor.setHSVValue(0.40F, 0.29F, 0.25F);

  return legend;
}/*-------------------------------------------------------------------*/

static SoSeparator*
createTitle(const char *title, float angle, float size)
{
  SoSeparator *textSep      = new SoSeparator ;
  SoText3     *text          = new SoText3;
  SoFont      *titleFont     = new SoFont;
  SoTransform *textTransform = new SoTransform;
  SoDirectionalLight *light  = new SoDirectionalLight;

  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;
  textSep->addChild(pickStyle) ;

  light->direction.setValue(1, 1, -0.8F);
 
  textSep->addChild(titleFont);
  textSep->addChild(light);

  SoMaterial *myMaterial = new SoMaterial;
  SbColor colors[3];
  // diffuse
  colors[0].setHSVValue(0.15F, 0.50F, 0.91F);
  colors[1].setHSVValue(0.15F, 0.73F, 1.0F);
  colors[2].setHSVValue(0.15F, 0.73F, 1.0F);
  myMaterial->diffuseColor.setValues(0, 3, colors);
  textSep->addChild(myMaterial);

  // Specify a beveled cross-section for the text
  SoProfileCoordinate2 *myProfileCoords = new SoProfileCoordinate2;
  SbVec2f coords[4];
  coords[0].setValue(0.00, 0.00);
  coords[1].setValue(size/16, size/16);
  coords[2].setValue(size/8,  size/16);
  coords[3].setValue(3*size/16, 0.00);
  myProfileCoords->point.setValues(0, 4, coords);
  textSep->addChild(myProfileCoords);

  SoLinearProfile *myLinearProfile = new SoLinearProfile;
  int32_t	index[4] ;
  index[0] = 0;
  index[1] = 1;
  index[2] = 2;
  index[3] = 3;
  myLinearProfile->index.setValues(0, 4, index);
  textSep->addChild(myLinearProfile);

  // Set the material binding to PER_PART
  SoMaterialBinding *myMaterialBinding = new SoMaterialBinding;
  myMaterialBinding->value.setValue(SoMaterialBinding::PER_PART);
  textSep->addChild(myMaterialBinding);

  SoSeparator *textSep2       = new SoSeparator;
  textSep->addChild(textSep2);

  text->parts=SoText3::ALL;
  text->string.setValue(title);
  text->justification=SoText3::CENTER;

  textTransform->rotation.setValue(SbVec3f(1, 0, 0), angle);
  textTransform->translation.setValue(0, 0.85F, 0.0F);

#ifdef _WIN32
  titleFont->name.setValue("Arial");
#else
  titleFont->name.setValue("Times-Roman");
#endif

  titleFont->size.setValue(size);

  textSep2->addChild(textTransform);
  textSep2->addChild(text);

  return textSep;
}/*-------------------------------------------------------------------*/

static const SbVec3f*
initData(float *value)
{
  SbVec3f *data=new SbVec3f[NB_COMPAGNY*NB_MONTH];
  int i;
#define OFFSET 0.5

  i=CEGID*NB_MONTH;
  value[i+APRIL95]=487.0;
  value[i+SEPTEMBER95]=410.0;
  value[i+JANUARY96]=370.0;
  value[i+APRIL96]=485.0;
  data[i+APRIL95].setValue(APRIL95+OFFSET, CEGID+OFFSET, value[i+APRIL95]);
  data[i+SEPTEMBER95].setValue(SEPTEMBER95+OFFSET, CEGID+OFFSET, value[i+SEPTEMBER95]);
  data[i+JANUARY96].setValue(JANUARY96+OFFSET, CEGID+OFFSET, value[i+JANUARY96]);
  data[i+APRIL96].setValue(APRIL96+OFFSET, CEGID+OFFSET, value[i+APRIL96]);

  i=DASSAULT*NB_MONTH;
  value[i+APRIL95]=340.9F;
  value[i+SEPTEMBER95]=206.5F;
  value[i+JANUARY96]=161.0F;
  value[i+APRIL96]=285.7F;
  data[i+APRIL95].setValue(APRIL95+OFFSET, DASSAULT+OFFSET, value[i+APRIL95]);
  data[i+SEPTEMBER95].setValue(SEPTEMBER95+OFFSET, DASSAULT+OFFSET, value[i+SEPTEMBER95]);
  data[i+JANUARY96].setValue(JANUARY96+OFFSET, DASSAULT+OFFSET, value[i+JANUARY96]);
  data[i+APRIL96].setValue(APRIL96+OFFSET, DASSAULT+OFFSET, value[i+APRIL96]);


  i=IBM*NB_MONTH;
  value[i+APRIL95]=458.6F;
  value[i+SEPTEMBER95]=526.0F;
  value[i+JANUARY96]=447.1F;
  value[i+APRIL96]=564.0F;
  data[i+APRIL95].setValue(APRIL95+OFFSET, IBM+OFFSET, value[i+APRIL95]);
  data[i+SEPTEMBER95].setValue(SEPTEMBER95+OFFSET, IBM+OFFSET, value[i+SEPTEMBER95]);
  data[i+JANUARY96].setValue(JANUARY96+OFFSET, IBM+OFFSET, value[i+JANUARY96]);
  data[i+APRIL96].setValue(APRIL96+OFFSET, IBM+OFFSET, value[i+APRIL96]);


  i=SLIGOS*NB_MONTH;
  value[i+APRIL95]=413.0;
  value[i+SEPTEMBER95]=455.0;
  value[i+JANUARY96]=407.0;
  value[i+APRIL96]=460.0;
  data[i+APRIL95].setValue(APRIL95+OFFSET, SLIGOS+OFFSET, value[i+APRIL95]);
  data[i+SEPTEMBER95].setValue(SEPTEMBER95+OFFSET, SLIGOS+OFFSET, value[i+SEPTEMBER95]);
  data[i+JANUARY96].setValue(JANUARY96+OFFSET, SLIGOS+OFFSET, value[i+JANUARY96]);
  data[i+APRIL96].setValue(APRIL96+OFFSET, SLIGOS+OFFSET, value[i+APRIL96]);

  return (const SbVec3f *) data;
}/*-------------------------------------------------------------------*/

// callback for change the value of field bar and axis box..
static void
timerCallBack(PbDomain *domain, SoTimerSensor *)
{
  if(!domain)
    return;
  static int currentPos=0;
  static int step=-1;

  static PoPointsFieldBars *fieldBar= (PoPointsFieldBars *) 
    SoNode::getByName(FIELD_BAR_NAME);
  static PoCartesianAxis *z1Axis= (PoCartesianAxis *) 
    SoNode::getByName(Z1_AXIS_NAME);
  static PoCartesianAxis *z2Axis= (PoCartesianAxis *) 
    SoNode::getByName(Z2_AXIS_NAME);
  static PoCartesianAxis *z3Axis= (PoCartesianAxis *) 
    SoNode::getByName(Z3_AXIS_NAME);
  static PoAutoCubeAxis *autoCubeAxis= (PoAutoCubeAxis *) 
    SoNode::getByName(AUTO_CUBE_AXIS_NAME);
  SbVec3f *points=(SbVec3f *)fieldBar->point.getValues(0);
  int pointNumber=fieldBar->point.getNum();

  if(currentPos>7 || currentPos<-7)
    step=(-step);

  currentPos+=step;
  float coef=0.9F;
  float max=0;
  if(step<0)
    coef=1/coef;
  // update fieldBars
  for(int i=0;i<NB_MONTH;i++)
    {
      coef=1/coef;
      for(int j=0;j<NB_COMPAGNY;j++)
	{
	  float pointsValue=points[i*NB_COMPAGNY+j][2];
	  if(j%2==0)
	    points[i*NB_COMPAGNY+j][2] = pointsValue*coef;
	  else
	    points[i*NB_COMPAGNY+j][2] = pointsValue*1/coef;
	  fieldBar->value.set1Value(i*NB_COMPAGNY+j, pointsValue);
	  if(points[i*NB_COMPAGNY+j][2]>max)
	    max=points[i*NB_COMPAGNY+j][2];
	} 
    }
  max+=10;
  fieldBar->point.setValues(0, pointNumber, (const SbVec3f *)points);

  // update Axis box

  float xMin, yMin, zMin, xMax, yMax, zMax; 
  domain->getDomain(xMin, yMin, zMin, xMax, yMax, zMax);
  z1Axis->end.setValue(max);
  z2Axis->end.setValue(max);
  z3Axis->end.setValue(max);
  autoCubeAxis->end.setValue(autoCubeAxis->end.getValue()[0],
			     autoCubeAxis->end.getValue()[1],
			     max) ;
  domain->setDomain(xMin, yMin, zMin, xMax, yMax, max);

}/*-------------------------------------------------------------------*/

// create dialog box to control print box and animate
SoTopLevelDialog*
createDialog(SoNode *rootToApply, PbDomain *domain)
{
  SoTopLevelDialog* dialog = new SoTopLevelDialog;

  SoTimerSensor *timer = new SoTimerSensor((SoSensorCB *)timerCallBack,
					    domain);
  timer->setInterval(0.5);

  SoSwitch* axisBox = (SoSwitch*) SoNode::getByName( AXIS_BOX_SWITCH_NAME );

  SoDialogCheckBox* autoViewing = new SoDialogCheckBox();
  autoViewing->label.setValue( "Auto viewing axes :" );
  autoViewing->state.setValue( FALSE );
  autoViewing->onString.setValue( "On" );
  autoViewing->offString.setValue( "Off" );
  autoViewing->addAuditor( new AutoViewingAuditor( axisBox ) );
  dialog->addChild( autoViewing );

  SoDialogCheckBox* monthEvolution = new SoDialogCheckBox();
  monthEvolution->label.setValue( "Month/Evolution :" );
  monthEvolution->state.setValue( FALSE );
  monthEvolution->onString.setValue( "Animate" );
  monthEvolution->offString.setValue( "Fixed" );
  monthEvolution->addAuditor( new MonthEvolutionAuditor( timer ) );
  dialog->addChild( monthEvolution );

  SoDialogPushButton* buttonIvFile = new SoDialogPushButton();
  buttonIvFile->label.setValue( "Save iv file" );
  buttonIvFile->buttonLabel.setValue( ivFileName );
  buttonIvFile->addAuditor( new WriteAuditor( rootToApply ) );
  dialog->addChild( buttonIvFile  );

  SoDialogPushButton* quitButton= new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialog->addChild( quitButton  );
  
  return dialog;
}/*-------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int main (int, char **argv)
{
#define NB_COLOR 5
  // Initialize Inventor and Xt
  Widget rootWindow = SoXt::init(argv[0]) ;

  if(rootWindow == NULL) 
    exit(1) ;

  SoXtExaminerViewer *viewer;
  
  PbMiscTextAttr textHisto, textLegend;
  static const char *compagnyList[NB_COMPAGNY] = {"Ibm", "Sligos", "Cegid", "Dassault"};
  static const char *monthList[NB_MONTH] = {"Apr 95", "Sep 95", "Jan 96", "Apr 96"};

  const SbVec3f *data;
  float value[NB_COMPAGNY*NB_MONTH];
  float thresholdColors[NB_COLOR] = {120, 300, 400, 500, 600};
  PbIsovaluesList isoList(120, 600, 8);
  SbColor colorsMapping[NB_COLOR] = {SbColor(1.0, 0.0, 0.0),   
				     SbColor(1.0, 1.0, 0.0), 
				     SbColor(0.0, 1.0, 0.0),   
				     SbColor(0.0, 1.0, 1.0), 
				     SbColor(0.0, 0.0, 1.0)};

  PbNonLinearDataMapping2 dataMapping;
  dataMapping.setValues(PbNonLinearDataMapping2::LINEAR_PER_LEVEL, 
			NB_COLOR, thresholdColors, colorsMapping);

  data=initData(value);

  viewer = new SoXtExaminerViewer(rootWindow);

  SbColor backgroung;
  backgroung.setHSVValue(0.62F, 0.35F, 0.24F);
  viewer->setBackgroundColor(backgroung);

  PbDomain domain(0.0, 0.0, 0.0, NB_MONTH, NB_COMPAGNY, 600.0);
  domain.setTransformType(PbDomain::TRANSFORM_01);

  // Initialize the new nodes class
  SoDialogViz::init();
  PoMeshViz::init();
   
  PoBase::setNodeWriteFormat(PoBase::UNFOLD_NODE_WRITE_FORMAT);

#ifdef _WIN32
  textHisto.setFontName("Arial");
  textLegend.setFontName("Arial");
#else
  textHisto.setFontName("Courier");
  textLegend.setFontName("Times-Roman");
#endif

  // create points field bars
  PoPointsFieldBars *pointsBar;
  pointsBar=new PoPointsFieldBars(NB_COMPAGNY*NB_MONTH, data, 
  				  (const float *)value, 0.5, 0.5);
  pointsBar->setName(FIELD_BAR_NAME);

  pointsBar->setDomain(&domain);
  pointsBar->setDataMapping(&dataMapping);

  // create 7 axis box
  SoSwitch *axisBox = new SoSwitch ;
  axisBox->setName(AXIS_BOX_SWITCH_NAME) ;
  axisBox->addChild(createAxisBox(compagnyList, monthList, &textHisto, &domain)) ;
  axisBox->addChild(createAutoAxisBox(compagnyList, monthList, &textHisto, &domain)) ;
  axisBox->whichChild.setValue(0) ;

  // createScene graph
  SoPerspectiveCamera *camera3D = new SoPerspectiveCamera;
  SoOrthographicCamera *camera2D = new SoOrthographicCamera ;
  camera2D->viewportMapping = SoCamera::LEAVE_ALONE ;
  SoSeparator *scene2D = new SoSeparator ;
  SoSeparator *scene3D = new SoSeparator ;

  SoSeparator *root=new SoSeparator;
  root->ref();
  root->addChild(scene3D) ;
  root->addChild(scene2D) ;

  SoAnnotation *legendAnn=new SoAnnotation;
  PoNonLinearValueLegend2 *legend;
  legend=createLegend(&textLegend, &dataMapping, &isoList) ;

  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;

  legendAnn->addChild(pickStyle) ;  
  legendAnn->addChild(legend);

  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;

  scene2D->addChild(annoText3Property) ;
  scene2D->addChild(camera2D) ;
  scene2D->addChild(legendAnn);
  scene2D->addChild(createTitle("Rate Exchange", 0.0, 0.19F));

  SoSeparator *sepDiagram=new SoSeparator;
  sepDiagram->addChild(axisBox);
  sepDiagram->addChild(pointsBar);

  scene3D->addChild(camera3D);
  scene3D->addChild(sepDiagram);

  // create dialog box
  SoTopLevelDialog *dialog=createDialog(root, &domain);
  dialog->buildDialog(rootWindow);
  dialog->show();

  SbViewportRegion myRegion(viewer->getSize());
  camera3D->orientation.setValue(1.3F, 2.2F, 3.2F, 1.5F);
  camera3D->viewAll(sepDiagram, myRegion);
  viewer->setSceneGraph(root);
  viewer->setTitle("3D Histogram Demo");
  viewer->show();
  viewer->setCamera(camera3D);
  SoXt::show(rootWindow);
  SoXt::mainLoop();
  delete viewer;
  root->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();

  return 0;
}/*-------------------------------------------------------------------*/
