//
// Pie Chart Demo
//////////////////////////////

#if defined(_WIN32)
#pragma warning(disable: 4996) // Disable PoXt deprecation warning
#endif

#include <Inventor/Xt/SoXt.h>

#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoProfileCoordinate2.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoLinearProfile.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoAnnoText3Property.h>

#include <Inventor/sensors/SoTimerSensor.h>

#include <MeshViz/graph/PoPieChart2D.h>
#include <MeshViz/graph/PoPieChart3D.h>
#include <MeshViz/graph/PbDomain.h>
#include <MeshViz/graph/PbMiscTextAttr.h>

#include <DialogViz/SoDialogVizAll.h>

#define NUM_SLICES           5
#define NUM_COLORS           5

#define YEAR_95              0
#define YEAR_96              1

#define TRANSLATE_VALUE2D    0.1F
#define TRANSLATE_VALUE3D    0.1F
#define NUM_SLICE_95_TRANS   0
#define NUM_SLICE_96_TRANS   1

#define NUM_XT_ELTS_INSERTED 2  

#define PIDIV4               0.785F

static const char *sliceTexts[NUM_SLICES] = {
  "OPGS", "Maintenance$Services",  "Component$Tools", 
  "OIV/GL", "Other$Products"
} ;

static const char *sliceInfo[NUM_SLICES] = {
  "OPGS : GPHIGS, GPHIGS GUI and PHIGURE (C/FORTRAN)",
  "Maintenance/Services : Specific development and maintenance",
  "Component Tools : Active X,...",
  "OIV/GL : Open Inventor and Open GL",
  "Other Products : GKS, CGM, ..."
} ;

static float sliceValues[2][NUM_SLICES] = {
  {11.96F, 8.84F,   0.F, 3.12F, 2.08F},
  { 9.24F, 9.52F, 0.28F, 7.28F, 1.68F}
} ;

static SbColor sliceColors[NUM_COLORS] = {
  SbColor(0.2F, 0.1F, 1.0F), 
  SbColor(1.0F, 0.1F, 1.0F),
  SbColor(0.7F, 0.F, 0.4F), 
  SbColor(0.1F, 0.8F, 0.1F),
  SbColor(0.F, 0.6F, 0.F)
} ;

static PoPieChart2D *pieChart2D95, *pieChart2D96 ;
static PoPieChart3D *pieChart3D95, *pieChart3D96 ;

static PbDomain               domain ;
static PbMiscTextAttr         textAttr ;
static PbNumericDisplayFormat numeriFormat(PbNumericDisplayFormat::INTEGER_FORMAT, 0) ;
	    

static SoSwitch       *pieChart2DSwitch, *pieChart3DSwitch, *pieChart3DInfoSwitch ;

static SoText2        *infoNodeText ;

static SbBool         isAnimation = FALSE, is2DRepresentation = TRUE ;
static int            currentYear = YEAR_96 ;

static SoOrthographicCamera *camera2D ;
static SoPerspectiveCamera *camera3D ;

static SoSeparator *root ;
static SoXtPlaneViewer *viewer ;

/*---------------------------------------------------------------------------*/

void selectionPathCB(void *, SoPath *path) 
{
  int slicePicked ;
  SbBool is95, is96 ;
  if ( ( is95 = pieChart3D95->getPickedSliceNumber( path, slicePicked ) ) )
  {
    pieChart3D95->sliceToTranslateNumber.deleteValues(0, -1) ;
    pieChart3D95->sliceToTranslateValue.deleteValues(0, -1) ;
    pieChart3D95->sliceToTranslateNumber.set1Value(0, slicePicked) ;
    pieChart3D95->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE3D) ;
  }
  if ( ( is96 = pieChart3D96->getPickedSliceNumber( path, slicePicked ) ) )
  {
    pieChart3D96->sliceToTranslateNumber.deleteValues(0, -1) ;
    pieChart3D96->sliceToTranslateValue.deleteValues(0, -1) ;
    pieChart3D96->sliceToTranslateNumber.set1Value(0, slicePicked) ;
    pieChart3D96->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE3D) ;
  }
  if(is95 || is96)
    infoNodeText->string.setValue(sliceInfo[slicePicked]) ;
    
}/*---------------------------------------------------------------------------*/

void buildPieChart2D(SoGroup *group)
{
  // Pie Chart 95
  pieChart2D95 = new PoPieChart2D(SbVec2f(0.0, 2.0), 1.0, NUM_SLICES, (const char**)sliceTexts, 
				  sliceValues[YEAR_95], NUM_COLORS, sliceColors) ;
  pieChart2D95->setDomain(&domain) ;
  pieChart2D95->setFormat(&numeriFormat) ;
  pieChart2D95->setMiscTextAttr(&textAttr) ;
  pieChart2D95->sliceToTranslateNumber.set1Value(0, NUM_SLICE_95_TRANS) ;
  pieChart2D95->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE2D) ;
  pieChart2D95->extAnnotPosition = PoPieChart::VERT_ALIGNED_EXT_POS ;
  pieChart2D95->extAnnotDistance2.setValue(0.2F) ;
  pieChart2D95->extAnnotDistance3.setValue(0.05F) ;
  pieChart2D95->intAnnotValueVisibility = FALSE ;
  pieChart2D95->intAnnotPercentVisibility = TRUE ;
  pieChart2D95->set("sliceBorderApp.drawStyle", "lineWidth 2") ;

  group->addChild(pieChart2D95) ;

  // Pie Chart 96
  pieChart2D96 = new PoPieChart2D(SbVec2f(0.0, 5.5), 1.0, NUM_SLICES, (const char**)sliceTexts, 
				  sliceValues[YEAR_96], NUM_COLORS, sliceColors) ;
  pieChart2D96->setDomain(&domain) ;
  pieChart2D96->setFormat(&numeriFormat) ;
  pieChart2D96->setMiscTextAttr(&textAttr) ;
  pieChart2D96->sliceToTranslateNumber.set1Value(0, NUM_SLICE_96_TRANS) ;
  pieChart2D96->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE2D) ;
  pieChart2D96->extAnnotDistance2.setValue(0.2F) ;
  pieChart2D96->extAnnotDistance3.setValue(0.05F) ;
  pieChart2D96->extAnnotPosition = PoPieChart::VERT_ALIGNED_EXT_POS ;
  pieChart2D96->intAnnotValueVisibility = FALSE ;
  pieChart2D96->intAnnotPercentVisibility = TRUE ;
  pieChart2D96->set("sliceBorderApp.drawStyle", "lineWidth 2") ;

  group->addChild(pieChart2D96) ;

}/*---------------------------------------------------------------------------*/

SoText2* buildText2(SoGroup *group, const SbVec2f &position, char *string, float size) 
{

  SoSeparator *textSep       = new SoSeparator ;
  SoTransform *textTransform = new SoTransform ;
  SoText2     *text          = new SoText2;
  SoFont      *font          = new SoFont;
  SoMaterial *mat = new SoMaterial ;

  mat->diffuseColor.setValue(1., 0., 0.) ;
#ifdef _WIN32
  font->name.setValue("Arial") ;
#else
  font->name.setValue("Times-Roman") ;
#endif

  font->size.setValue(size) ;

  text->string.setValue(string);
  text->justification=SoText2::CENTER;

  textTransform->translation.setValue(position[0], position[1], 0.0);

  textSep->addChild(mat) ;
  textSep->addChild(font) ;
  textSep->addChild(textTransform) ;
  textSep->addChild(text) ;
  group->addChild(textSep) ;
  return text ;

}/*---------------------------------------------------------------------------*/

void buildText(SoGroup *group, const SbVec2f &position, char *string,
	       float angle, float size) 
{
  SoSeparator *textSep       = new SoSeparator;
  SoText3     *text          = new SoText3;
  SoFont      *font          = new SoFont;
  SoTransform *textTransform = new SoTransform;
  SoDirectionalLight *light  = new SoDirectionalLight;

  SoPickStyle *pickStyle = new SoPickStyle ;
  pickStyle->style = SoPickStyle::UNPICKABLE ;

  light->direction.setValue(1, 1, 0.8F);
 
  textSep->addChild(pickStyle) ;
  textSep->addChild(font);
  textSep->addChild(light);

  SoMaterial *myMaterial = new SoMaterial;
  SbColor colors[3];
  // diffuse
  colors[0].setHSVValue(0.15F, 0.50F, 0.91F);
  colors[1].setHSVValue(0.15F, 0.73F, 1.0F);
  colors[2].setHSVValue(0.15F, 0.73F, 1.0F);
  myMaterial->diffuseColor.setValues(0, 3, colors);
  textSep->addChild(myMaterial);

  // Specify a beveled cross-section for the text
  SoProfileCoordinate2 *myProfileCoords = new SoProfileCoordinate2;
  SbVec2f coords[4];
  coords[0].setValue(0.00, 0.00);
  coords[1].setValue(size/16, size/16);
  coords[2].setValue(size/8,  size/16);
  coords[3].setValue(3*size/16, 0.00);
  myProfileCoords->point.setValues(0, 4, coords);
  textSep->addChild(myProfileCoords);

  SoLinearProfile *myLinearProfile = new SoLinearProfile;
  int32_t	index[4] ;
  index[0] = 0;
  index[1] = 1;
  index[2] = 2;
  index[3] = 3;
  myLinearProfile->index.setValues(0, 4, index);
  textSep->addChild(myLinearProfile);

  // Set the material binding to PER_PART
  SoMaterialBinding *myMaterialBinding = new SoMaterialBinding;
  myMaterialBinding->value.setValue(SoMaterialBinding::PER_PART);
  textSep->addChild(myMaterialBinding);

  SoSeparator *textSep2       = new SoSeparator;
  textSep->addChild(textSep2);

  text->parts=SoText3::ALL;
  text->string.setValue(string);
  text->justification=SoText3::CENTER;

  textTransform->rotation.setValue(SbVec3f(1, 0, 0), angle);
  textTransform->translation.setValue(position[0], position[1], 0.0);

#ifdef _WIN32
  font->name.setValue("Arial");
#else
  font->name.setValue("Times-Roman");
#endif

  font->size.setValue(size);

  textSep2->addChild(textTransform);
  textSep2->addChild(text);
  
  group->addChild(textSep) ;

}/*---------------------------------------------------------------------------*/

void buildPieChart3D(SoGroup *group)
{
  SoRotationXYZ *rotation = new SoRotationXYZ ;
  rotation->axis = SoRotationXYZ::X ;
  rotation->angle.setValue(-PIDIV4) ;

  // Pie Chart 95
  pieChart3D95 = new PoPieChart3D(SbVec3f(0.0, 0.0, 0.), 0.05F, 1.0, NUM_SLICES, 
				  (const char**)sliceTexts, 
				  sliceValues[YEAR_95], NUM_COLORS, sliceColors) ;
  pieChart3D95->setDomain(&domain) ;
  pieChart3D95->setFormat(&numeriFormat) ;
  pieChart3D95->setMiscTextAttr(&textAttr) ;
  pieChart3D95->sliceToTranslateNumber.set1Value(0, NUM_SLICE_95_TRANS) ;
  pieChart3D95->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE3D) ;
  pieChart3D95->intAnnotTextFontSize.setValue(0.015F) ;
  pieChart3D95->intAnnotValueVisibility = FALSE ;
  pieChart3D95->intAnnotTextVisibility = TRUE ;
  pieChart3D95->intAnnotPercentVisibility = TRUE ;
  pieChart3D95->extAnnotTextVisibility = FALSE ;
  pieChart3D95->set("intAnnotBoxBorderApp.drawStyle", "style INVISIBLE") ;
  pieChart3D95->set("intAnnotBoxApp.drawStyle", "style INVISIBLE") ;

  SoTransformSeparator *transSep95 = new SoTransformSeparator ;
  group->addChild(transSep95) ;
  transSep95->addChild(rotation) ;
  transSep95->addChild(pieChart3D95) ;

  // Pie Chart 96
  pieChart3D96 = new PoPieChart3D(SbVec3f(0.0, 0.0, 0.), 0.05F, 1.0, NUM_SLICES, 
				  (const char**)sliceTexts, 
				  sliceValues[YEAR_96], NUM_COLORS, sliceColors) ;
  pieChart3D96->setDomain(&domain) ;
  pieChart3D96->setFormat(&numeriFormat) ;
  pieChart3D96->setMiscTextAttr(&textAttr) ;
  pieChart3D96->sliceToTranslateNumber.set1Value(0, NUM_SLICE_96_TRANS) ;
  pieChart3D96->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE3D) ;
  pieChart3D96->intAnnotTextFontSize.setValue(0.015F) ;
  pieChart3D96->intAnnotValueVisibility = FALSE ;
  pieChart3D96->intAnnotTextVisibility = TRUE ;
  pieChart3D96->intAnnotPercentVisibility = TRUE ;
  pieChart3D96->extAnnotTextVisibility = FALSE ;
  pieChart3D96->set("intAnnotBoxBorderApp.drawStyle", "style INVISIBLE") ;
  pieChart3D96->set("intAnnotBoxApp.drawStyle", "style INVISIBLE") ;

  SoTransformSeparator *transSep96 = new SoTransformSeparator ;
  group->addChild(transSep96) ;
  transSep96->addChild(rotation) ;
  transSep96->addChild(pieChart3D96) ;

}/*---------------------------------------------------------------------------*/

class QuitAuditor : public SoDialogPushButtonAuditor
{
  public:
  void dialogPushButton(SoDialogPushButton* /*cpt*/)
  {
    exit( 0 );
  }
};

class YearAuditor : public SoDialogChoiceAuditor
{
public:
  void dialogChoice( SoDialogChoice* cpt )
  {
    currentYear = cpt->selectedItem.getValue();
    pieChart3DSwitch->whichChild.setValue( currentYear );
    if ( currentYear == YEAR_95 )
      infoNodeText->string.setValue( sliceInfo[pieChart3D95->sliceToTranslateNumber[0]] );
    else // YEAR_96
      infoNodeText->string.setValue( sliceInfo[pieChart3D96->sliceToTranslateNumber[0]] );
  }
};

class AnimateAuditor : public SoDialogCheckBoxAuditor
{
public:
  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    if ( cpt->state.getValue() )
      isAnimation = TRUE;
    else
      isAnimation = FALSE;
  }
};

class RepresentationAuditor : public SoDialogCheckBoxAuditor
{
  SoTopLevelDialog* m_dialog;
public:
  RepresentationAuditor( SoTopLevelDialog *dialog )
    : m_dialog( dialog )
  { }

  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    if ( cpt->state.getValue() )
    {
      // 3D
      is2DRepresentation = FALSE;

      ( (SoTopComponent*) m_dialog->getChild( 2 ) )->enable = TRUE;
      ( (SoTopComponent*) m_dialog->getChild( 3 ) )->enable = TRUE;

      currentYear = YEAR_96;
      root->replaceChild( 1, camera3D );
      viewer->setCamera( camera3D );
      pieChart2DSwitch->whichChild.setValue( SO_SWITCH_NONE );
      pieChart3DSwitch->whichChild.setValue( currentYear );
      pieChart3DInfoSwitch->whichChild.setValue( SO_SWITCH_ALL );
      infoNodeText->string.setValue( sliceInfo[pieChart3D96->sliceToTranslateNumber[0]] );
    }
    else
    {
      // 2D
      root->replaceChild( 1, camera2D );
      viewer->setCamera( camera2D );
      is2DRepresentation = TRUE;

      ( (SoTopComponent*) m_dialog->getChild( 2 ) )->enable = FALSE;
      ( (SoTopComponent*) m_dialog->getChild( 3 ) )->enable = FALSE;

      pieChart3DInfoSwitch->whichChild.setValue( SO_SWITCH_NONE );
      pieChart2DSwitch->whichChild.setValue( SO_SWITCH_ALL );
      pieChart3DSwitch->whichChild.setValue( SO_SWITCH_NONE );
    }
  }
};

void timerCallback(void *, SoSensor *sensor)
{
  if(isAnimation) {
    int sliceTransNum ;
    if(is2DRepresentation) {
      sliceTransNum = pieChart2D95->sliceToTranslateNumber[0] ;
      sliceTransNum = (sliceTransNum + 1) % NUM_SLICES ;
      if(sliceTransNum == 2) sliceTransNum++ ;
      pieChart2D95->sliceToTranslateNumber.deleteValues(0, -1) ;
      pieChart2D95->sliceToTranslateValue.deleteValues(0, -1) ;
      pieChart2D95->sliceToTranslateNumber.set1Value(0, sliceTransNum) ;
      pieChart2D95->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE2D) ;

      sliceTransNum = pieChart2D96->sliceToTranslateNumber[0] ;
      sliceTransNum = (sliceTransNum + 1) % NUM_SLICES ;
      pieChart2D96->sliceToTranslateNumber.deleteValues(0, -1) ;
      pieChart2D96->sliceToTranslateValue.deleteValues(0, -1) ;
      pieChart2D96->sliceToTranslateNumber.set1Value(0, sliceTransNum) ;
      pieChart2D96->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE2D) ;
    }
    else {
      if(currentYear == YEAR_95) {
	sliceTransNum = pieChart3D95->sliceToTranslateNumber[0] ;
	sliceTransNum = (sliceTransNum + 1) % NUM_SLICES ;
	if(sliceTransNum == 2) sliceTransNum++ ;
	pieChart3D95->sliceToTranslateNumber.deleteValues(0, -1) ;
	pieChart3D95->sliceToTranslateValue.deleteValues(0, -1) ;
	pieChart3D95->sliceToTranslateNumber.set1Value(0, sliceTransNum) ;
	pieChart3D95->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE3D) ;
      }
      else { // YEAR_96
	sliceTransNum = pieChart3D96->sliceToTranslateNumber[0] ;
	sliceTransNum = (sliceTransNum + 1) % NUM_SLICES ;
	pieChart3D96->sliceToTranslateNumber.deleteValues(0, -1) ;
	pieChart3D96->sliceToTranslateValue.deleteValues(0, -1) ;
	pieChart3D96->sliceToTranslateNumber.set1Value(0, sliceTransNum) ;
	pieChart3D96->sliceToTranslateValue.set1Value(0, TRANSLATE_VALUE3D) ;
      }
      infoNodeText->string.setValue(sliceInfo[sliceTransNum]) ;
    }
  }
  sensor->schedule() ;

}/*---------------------------------------------------------------------------*/

SoTopLevelDialog* buildDialogBox(Widget myWindow)
{
  SoTopLevelDialog *dialog = new SoTopLevelDialog;
  dialog->label = "Pie Chart Attributes";

  SoDialogCheckBox* representation = new SoDialogCheckBox();
  representation->label.setValue( "Representation :" );
  representation->state.setValue( FALSE );
  representation->onString.setValue( "3D" );
  representation->offString.setValue( "2D" );
  representation->addAuditor( new RepresentationAuditor( dialog ) );
  dialog->addChild( representation );

  SoDialogCheckBox* animate = new SoDialogCheckBox();
  animate->label.setValue( "Animation :" );
  animate->state.setValue( FALSE );
  animate->onString.setValue( "ON" );
  animate->offString.setValue( "OFF" );
  animate->addAuditor( new AnimateAuditor() );
  dialog->addChild( animate );

  static const char *yearList[] = {"1995", "1996"} ;
  SoDialogComboBox* yearCombo = new SoDialogComboBox();
  yearCombo->label.setValue( "Year :" );
  yearCombo->items.setValues( 0, 2, yearList );
  yearCombo->selectedItem = YEAR_96;
  yearCombo->addAuditor( new YearAuditor() );
  dialog->addChild( yearCombo );

  SoDialogLabel* label = new SoDialogLabel;
  label->label = "Click on a slice for more information !";
  dialog->addChild( label );

  SoDialogPushButton* quitButton= new SoDialogPushButton();
  quitButton->buttonLabel.setValue( "Quit" );
  quitButton->addAuditor( new QuitAuditor() );
  dialog->addChild( quitButton );

  dialog->buildDialog( myWindow );

  yearCombo->enable = FALSE;

  return dialog;

}/*---------------------------------------------------------------------------*/

#include <Inventor/SoWinApp.h>

int main (int, char **argv)	
{  
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]) ;
  if (myWindow == NULL) exit(1) ;

  // Initialize the new nodes class
  PoMeshViz::init() ;
  SoDialogViz::init();
  
  domain.setDomain(0, 0, 7., 7.) ;
#ifdef _WIN32
  textAttr.setFontName("Arial") ;
#else
  textAttr.setFontName("Courier") ;
#endif

  textAttr.setLineLength('$', 10) ;

  root = new SoSeparator ;
  root->ref() ; 

  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;

  buildText(root, SbVec2f(0., 0.85F), (char*)"REVENUE BY PRODUCT", 0.05F, 0.10F) ;

  // Camera
  camera2D = new SoOrthographicCamera ;
  camera3D = new SoPerspectiveCamera ;
  camera2D->ref() ;
  camera3D->ref() ;
  root->addChild(camera2D) ;
  root->addChild(annoText3Property) ;

  // 2D Pie Chart
  pieChart2DSwitch = new SoSwitch ;
  pieChart2DSwitch->whichChild.setValue(SO_SWITCH_ALL) ;
  root->addChild(pieChart2DSwitch) ;
  buildPieChart2D(pieChart2DSwitch) ;
  buildText(pieChart2DSwitch, SbVec2f(0., 3.75F), (char*)"1996", 0.05F, 0.5F) ;
  buildText(pieChart2DSwitch, SbVec2f(0., 0.25F), (char*)"1995", 0.05F, 0.5F) ;

  // 3D info
  pieChart3DInfoSwitch = new SoSwitch ;
  pieChart3DInfoSwitch->whichChild.setValue(SO_SWITCH_NONE) ;
  root->addChild(pieChart3DInfoSwitch) ;
  infoNodeText = buildText2(pieChart3DInfoSwitch, SbVec2f(0., -1.2F), (char*)sliceInfo[1], 14) ;

  // 3D Pie Chart
  SoSelection *pieChart3DSelection = new SoSelection ;
  pieChart3DSelection->addSelectionCallback(selectionPathCB) ;
  root->addChild(pieChart3DSelection) ;

  pieChart3DSwitch = new SoSwitch ;
  pieChart3DSwitch->whichChild.setValue(0) ;
  pieChart3DSelection->addChild(pieChart3DSwitch) ;
  buildPieChart3D(pieChart3DSwitch) ;

  // Dialog box
  SoTopLevelDialog* dialog = buildDialogBox( myWindow );
  dialog->show();

  // Plane viewer
  viewer = new SoXtPlaneViewer(myWindow);
  camera3D->viewAll(pieChart3DSwitch, viewer->getViewportRegion()) ;
  pieChart3DSwitch->whichChild.setValue(SO_SWITCH_NONE) ;
  camera2D->viewAll(pieChart2DSwitch, viewer->getViewportRegion()) ;

  viewer->setSceneGraph(root);
  viewer->setTitle("Pie Chart Demo");
  viewer->setCamera(camera2D) ;
  viewer->setSize(viewer->getSize()*1.3) ;

  viewer->show();

  // Time sensor
  SoTimerSensor timerSensor(timerCallback, NULL) ;
  timerSensor.schedule() ;
  timerSensor.setInterval(SbTime(1.0)) ;

  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete viewer;
  root->unref();
  SoDialogViz::finish();
  PoMeshViz::finish();
  SoXt::finish();

  return 0;
}/*---------------------------------------------------------------------------*/
