
#include "gui/GuiDialogViz.h"
#include "scenegraph/OivSceneGraph.h"
#include "gui/AuditorAnimation.h"

#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <DialogViz/auditors/SoDialogAuditor.h> 
#include <DialogViz/dialog/SoDialogIntegerSlider.h>
#include <DialogViz/dialog/SoDialogRealSlider.h>
#include <DialogViz/dialog/SoDialogPushButton.h>
#include <DialogViz/dialog/SoDialogCheckBox.h>

#include <Inventor/sensors/SoAlarmSensor.h>
#include <Inventor/sensors/SoTimerSensor.h>


//---------------------------------------------------------------------------

static void
animationWrapperCB(void *cb_arg, SoSensor*)
{
  AuditorAnimation* animation_object;

  animation_object = (AuditorAnimation*)cb_arg;

  animation_object->animationStepCB();
}


//---------------------------------------------------------------------------

AuditorAnimation::AuditorAnimation(GuiDialogViz* gui)
  : m_gui(gui), m_playing(TRUE)
{
  SoDialogCheckBox* loop_toggle = (SoDialogCheckBox*)m_gui->m_topLevelDialog->searchForAuditorId("animationlooptoggle");
  SoDialogIntegerSlider* frame_slider = (SoDialogIntegerSlider*)m_gui->m_topLevelDialog->searchForAuditorId("animationslider");
  SoDialogIntegerSlider* speed_slider = (SoDialogIntegerSlider*)m_gui->m_topLevelDialog->searchForAuditorId("animationspeedslider");


  m_loop = loop_toggle->state.getValue();

  // don't call setFrame because it would try to refresh the scene graph which hasn't been initialized completely yet
  m_frame = computeFrameFromSliderPosition(frame_slider->value.getValue());

  m_timer = new SoTimerSensor(animationWrapperCB, this);
  setAnimationDelay(speed_slider->value.getValue());
}

AuditorAnimation::~AuditorAnimation()
{
  delete m_timer;
}

//---------------------------------------------------------------------------

int
AuditorAnimation::computeFrameFromSliderPosition(int slider_pos)
{
  return (slider_pos - 1);
}

int
AuditorAnimation::computeSliderPositionFromFrame(int frame)
{
  return (frame + 1);
}

//---------------------------------------------------------------------------

void
AuditorAnimation::refreshFrame()
{
  m_gui->setFrame(m_frame);
}

void
AuditorAnimation::setFrame(int frame)
{
  m_frame = frame;
  refreshFrame();
}

void
AuditorAnimation::setSliderPosition(int frame)
{
  SoDialogIntegerSlider* slider = (SoDialogIntegerSlider*)m_gui->m_topLevelDialog->searchForAuditorId("animationslider");
  slider->value.setValue(computeSliderPositionFromFrame(frame));
}

//---------------------------------------------------------------------------

void
AuditorAnimation::setPlaybackPosition(int frame)
{
  setSliderPosition(frame);
  setFrame(frame);
}

bool
AuditorAnimation::isPlaybackFinished()
{
  return (m_frame == FRAME_LIMIT);
}

void
AuditorAnimation::advancePlaybackPosition()
{
  if (!isPlaybackFinished())
  {
    m_frame++;

    setSliderPosition(m_frame);
    refreshFrame();
  }
}


//---------------------------------------------------------------------------

void
AuditorAnimation::animationStepCB()
{
  if (!isPlaybackFinished())
  {
    advancePlaybackPosition();

    if (isPlaybackFinished())
    {
      if (m_loop)
      {
        setPlaybackPosition(0);
      }
      else
      {
        stopAnimation();
      }
    }
  }
}

//---------------------------------------------------------------------------

void
AuditorAnimation::startAnimation()
{
  if (!m_playing)
  {
    SoDialogPushButton* button = (SoDialogPushButton*)m_gui->m_topLevelDialog->searchForAuditorId("animationbutton");

    if (!m_timer->isScheduled())
    {
      m_timer->schedule();
    }

    if (isPlaybackFinished())
    {
      setPlaybackPosition(0);
    }

    button->buttonLabel.setValue("Stop");
    m_playing = TRUE;
  }
}

void
AuditorAnimation::stopAnimation()
{
  SoDialogPushButton* button = (SoDialogPushButton*)m_gui->m_topLevelDialog->searchForAuditorId("animationbutton");

  m_timer->unschedule();

  button->buttonLabel.setValue("Play");
  m_playing = FALSE;
}

//---------------------------------------------------------------------------

void
AuditorAnimation::setAnimationDelay(double delay)
{
  m_animation_delay = delay;

  m_timer->setInterval(SbTime(m_animation_delay));
}

//---------------------------------------------------------------------------

void
AuditorAnimation::dialogPushButton(SoDialogPushButton* button)
{
  if (button->auditorID.getValue() == "animationbutton")
  {
    if (m_playing)
    {
      stopAnimation();
    }
    else
    {
      startAnimation();
    }
  }
}

//---------------------------------------------------------------------------
void 
AuditorAnimation::dialogIntegerSlider(SoDialogIntegerSlider* slider)
{
  if (slider->auditorID.getValue() == "animationslider")
  {
    setFrame(computeFrameFromSliderPosition(slider->value.getValue()));
  }
}

void 
AuditorAnimation::dialogRealSlider(SoDialogRealSlider* slider)
{
  if (slider->auditorID.getValue() == "animationspeedslider")
  {
    setAnimationDelay(slider->value.getValue());
  }
}

//---------------------------------------------------------------------------

void 
AuditorAnimation::dialogCheckBox(SoDialogCheckBox* toggle)
{
  if (toggle->auditorID.getValue() == "animationlooptoggle")
  {
    m_loop = toggle->state.getValue();
  }
}


