#include "GuiDialogViz.h"
#include "AuditorCheckSwitch.h"
#include "AuditorGeometry.h"
#include "AuditorProperties.h"
#include "AuditorHull.h"
#include "AuditorFilter.h"
#include "AuditorDrawStyle.h"
#include "AuditorAnimation.h"
#include "AuditorDownSampling.h"
#include "AuditorSnapshot.h"
#include "SaveSceneAuditor.h"
#include "AuditorTimer.h"
#include "NewStaticSceneAuditor.h"
#include "CloseSceneAuditor.h"
#include "SaveSceneAuditor.h"
#include "wrappers/CellFilter.h"
#include "wrappers/CachedTopoPillarMesh.h"
#include "wrappers/SubSampledMesh.h"
#include "pillargrid/PillarGridGenerator.h"
#include "pillargrid/PillarGridFileIO.h"
#include "pillargrid/PillarGridReader.h"
#include "scenegraph/MeshResolutionSwitch.h"

#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <DialogViz/SoDialogVizAll.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <DialogViz/dialog/SoDialogEditText.h>
#include <Inventor/nodes/SoText2.h>


#include <Inventor/STL/iomanip>
#include <Inventor/STL/sstream>

#include <ScaleViz/SoScaleViz.h>
#include <ScaleViz/misc/SoMemoryBuffer.h>
#include <ScaleViz/misc/SoMemoryBufferCompression.h>


//---------------------------------------------------------------------------
GuiDialogViz::GuiDialogViz(std::string ivDialogFile, std::string ivDialogFileForViewer) :
  m_numSnapshot(0),
  m_grid(NULL),
  m_eclipseMesh(NULL),
  m_sceneGraph(NULL),
  m_ivDialogFile(ivDialogFile),
  m_ivDialogFileForViewer(ivDialogFileForViewer),
  m_init(FALSE)
{
  m_sceneGraph = new OivSceneGraph;
}

//---------------------------------------------------------------------------
GuiDialogViz::~GuiDialogViz()
{
  delete m_grid;
  delete m_eclipseMesh;
  delete m_sceneGraph;
  delete m_viewer;
}

//---------------------------------------------------------------------------
void 
GuiDialogViz::start( Widget window)
{
  m_viewerTopLevelDialog = new SoTopLevelDialog;
  m_viewerTopLevelDialog->position.setValue(300,0);
  m_viewerTopLevelDialog->label = "Mesh Viewer";
  
  SoMenuBar* MenuBar = (SoMenuBar *)SoDialogViz::loadFromFile(m_ivDialogFileForViewer);

  SoDialogCustom *ParentViewer = new SoDialogCustom();
  ParentViewer->height = 800;
  ParentViewer->width = 1200; 

  m_viewerTopLevelDialog->addChild(MenuBar);
  m_viewerTopLevelDialog->addChild(ParentViewer);
  m_viewerTopLevelDialog->buildDialog( window, TRUE);
 
  m_viewer = new SoXtExaminerViewer(ParentViewer->getWidget());
  m_viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  m_viewer->setSceneGraph(m_sceneGraph->getRoot());
  m_viewer->show();
  m_viewer->setFramesPerSecondCallback( GuiDialogViz::viewerFPSCB, m_sceneGraph );
  
  
  m_viewerTopLevelDialog->show();

  buildGui();

  m_topLevelDialog->buildDialog( window, FALSE );
  m_topLevelDialog->show();


}

//---------------------------------------------------------------------------
void 
GuiDialogViz::readMesh(const std::string& filename)
{
  delete m_eclipseMesh;
  m_eclipseMesh = NULL;
  delete m_grid;
  m_grid = new PillarGrid;

  m_numSnapshot = 0; 
  m_gridName = filename.substr(filename.find_last_of('/'));
  m_gridName.erase(0,1);
  while (m_gridName.find_last_of('.') != std::string::npos)
  {
    m_gridName.erase(m_gridName.length()-1);
  }

  if (filename.find_last_of('.') != std::string::npos) 
  {
    std::string extension(filename.substr(filename.find_last_of('.')));

    if ( extension == ".txt" )
    {
      PillarGridGenerator generator;
      generator.generate(*m_grid, filename);
    }
    else if ( extension == ".grdecl" || extension == ".GRDECL" )
    {
      PillarGridReader reader;
      reader.read(*m_grid, filename);
    }
    else
    {
      PillarGridFileIO file;
      file.read(*m_grid,filename);
    }
  }




  this->init();

  m_viewer->viewAll();
}

//---------------------------------------------------------------------------
void 
GuiDialogViz::saveMesh(const std::string& filename)
{
  PillarGridFileIO file;
  file.write(*m_grid,filename);
}

void GuiDialogViz::loadMesh(PillarGrid& grid)
{
  delete m_eclipseMesh;
  m_eclipseMesh = new CachedTopoPillarMesh(grid);
  m_sceneGraph->setMesh((CachedTopoPillarMesh*)m_eclipseMesh);
}

//---------------------------------------------------------------------------
void
GuiDialogViz::buildGui()
{
  SoInput myInput;
  if (! myInput.openFile( m_ivDialogFile.c_str() ))
    return;
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup)
    return;
  m_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  buildAuditorGeometryPanel();
  buildAuditorPropertiesPanel();
  buildAuditorHullPanel();
  buildAuditorRoiPanel();
  buildAuditorAnimationPanel();
  buildAuditorDrawStylePanel();

  SoMenuFileSelection* newStaticScene = (SoMenuFileSelection*) m_viewerTopLevelDialog->searchForAuditorId("fileauditor");
  newStaticScene->addAuditor(new NewStaticSceneAuditor(*this));

  SoMenuFileSelection* saveScene = (SoMenuFileSelection*)m_viewerTopLevelDialog->searchForAuditorId("saveauditor");
  saveScene->addAuditor(new SaveSceneAuditor(*this));

  SoMenuPushButton* closeScene = (SoMenuPushButton*)m_viewerTopLevelDialog->searchForAuditorId("closeauditor");
  closeScene->addAuditor(new CloseSceneAuditor);
  
  SoRowDialog* downsampling = (SoRowDialog*)m_topLevelDialog->searchForAuditorId("downsampling");
  downsampling->addAuditor(new AuditorDownSampling(this));

  SoDialogPushButton* snap = (SoDialogPushButton*)m_topLevelDialog->searchForAuditorId("snapshot");
  snap->addAuditor(new AuditorSnapshot(this));

  SoDialogCheckBox* timercheck = (SoDialogCheckBox*)m_topLevelDialog->searchForAuditorId("timercheck");
  timercheck->addAuditor(new AuditorTimer);
  timercheck->enable = m_sceneGraph->availableTimer();
  timercheck->state = timercheck->state.getValue() && m_sceneGraph->availableTimer();
}

//---------------------------------------------------------------------------
Widget
GuiDialogViz::init()
{
  loadMesh(*m_grid);

  bool porosityAvailable = m_grid->porositySize() > 0;
  SoDialogComboBox* combobox = (SoDialogComboBox*)m_topLevelDialog->searchForAuditorId("FilterProperty");
  combobox->enable = porosityAvailable;
  combobox->selectedItem = porosityAvailable? 1 : 0;
  SoDialogComboBox* isocombobox = (SoDialogComboBox*)m_topLevelDialog->searchForAuditorId("IsoProperty");
  isocombobox->enable = porosityAvailable;
  isocombobox->selectedItem = porosityAvailable ? 1 : 0;

  setSliders();
  
  initGeometryPanel();
  initPropertiesPanel();
  initHullPanel();
  initRoiPanel();
  initAnimationPanel();
  initDrawStylePanel();
  initInfoPanel();

  update();

  m_init = TRUE;

  return m_widget;
}

//---------------------------------------------------------------------------
void
GuiDialogViz::update()
{
  // activate downsampling only if available
  SoRowDialog* downsampling = (SoRowDialog*)m_topLevelDialog->searchForAuditorId("downsampling");
  downsampling->searchForAuditorId("autodownsampling")->enable = m_sceneGraph->availableLowResolution();
  downsampling->applyAuditors();
  // update per node dataset for isosurface
  SoDialogCheckBox* isocheck = (SoDialogCheckBox*) m_topLevelDialog->searchForAuditorId("isosurfacecheck");
  isocheck->applyDlgCptAuditor();
}

void 
  GuiDialogViz::snapshot()
{
  time_t t = time(NULL);
  struct tm tm = *localtime(&t);
  SbString month = tm.tm_mon + 1;
  SbString day = tm.tm_mday;
  m_numSnapshot ++;  
  SbString num = m_numSnapshot;
  SbString snapshotFilePath = SoPreferences::getString("OIVHOME",".") + "/examples/source/MeshVizXLM/demonstrators/EclipseMeshViz/Screenshots/"+m_gridName+"_"+month+day+"_"+num+".png";
  m_viewer->saveSnapshot(snapshotFilePath);
  std::cout << "screenshot saved in : " << snapshotFilePath << std::endl;
}

//---------------------------------------------------------------------------
void 
GuiDialogViz::setSliders(int defaultDimI, int defaultDimJ, int defaultDimK, float /*defaultMax*/)
{
  SbVec3s meshDim(defaultDimI,defaultDimJ,defaultDimK);
  if(defaultDimK<0)
    meshDim = m_sceneGraph->getMeshDim();

  initRoiSlider("roiimin", 1, 0,meshDim[0]-1);
  initRoiSlider("roiimax", 1, meshDim[0]-1,meshDim[0]-1);
  initRoiSlider("roijmin", 2, 0,meshDim[1]-1);
  initRoiSlider("roijmax", 3, meshDim[1]-1,meshDim[1]-1);
  initRoiSlider("roikmin", 4, 0,meshDim[2]-1);
  initRoiSlider("roikmax", 5, meshDim[2]-1,meshDim[2]-1);

    if(m_eclipseMesh != NULL)  
  {
    SoDialogComboBox* combobox = (SoDialogComboBox*)m_topLevelDialog->searchForAuditorId("FilterProperty");
    std::string property(combobox->items[combobox->selectedItem.getValue()].getString());
    this->updateDataRangeSliders(property);
    SoDialogComboBox* isocombobox = (SoDialogComboBox*)m_topLevelDialog->searchForAuditorId("IsoProperty");
    std::string isoproperty(isocombobox->items[isocombobox->selectedItem.getValue()].getString());
    this->updateIsosurfSlider(isoproperty);
  }

  // slice number
  for(int id=0; id<3;id++){
    std::ostringstream numberId;
    numberId << "logicalslicenumber" << id;
    SoDialogIntegerSlider* sliceNumberDialog  = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId(numberId.str().c_str());
    if (sliceNumberDialog != NULL)
      sliceNumberDialog->max = meshDim[id] - 1;
      sliceNumberDialog->value = meshDim[id]/2;
    std::ostringstream thickId;
    thickId << "logicalslicethickness" << id;
    SoDialogIntegerSlider* sliceThicknessDialog  = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId(thickId.str().c_str());
    if (sliceThicknessDialog != NULL)
      sliceThicknessDialog->max = meshDim[id];
    std::ostringstream stepId;
    stepId << "multipleslab" << id;
    SoDialogIntegerSlider* sliceStepDialog  = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId(stepId.str().c_str());
    if (sliceStepDialog != NULL)
      sliceStepDialog->max = meshDim[id];
      sliceStepDialog->min = 1;  
      sliceStepDialog->value = meshDim[id];
  }
}

void
GuiDialogViz::updateResolution(double resolutionRatio)
{
  m_sceneGraph->adjustLowResolution(resolutionRatio);
  // activate low resolution if ratio below 0.5
  if (resolutionRatio <= 0.5)
    m_sceneGraph->switchToLowResolutionMesh();
  else
    m_sceneGraph->switchToFullResolutionMesh();
}

bool
GuiDialogViz::updateDataRangeSliders(const std::string& propertyName)
{
  const GridPropertyIjk* dataset = m_eclipseMesh->getProperty( propertyName );
  float vmax = dataset != NULL ? (float)dataset->getMax() : 0.f;
  float vmin = dataset != NULL ? (float)dataset->getMin() : 0.f;
  initDataRangeSlider("pmin", 1, vmin, vmin, vmax);
  initDataRangeSlider("pmax", 1, vmin, vmax, vmax);
  return m_sceneGraph->setDataRangeFiltering(vmin, vmax, propertyName);
}

void 
GuiDialogViz::updateIsosurfSlider(const std::string& propertyName)
{ 
  const GridPropertyIjk* dataset = m_eclipseMesh->getProperty( propertyName );
  float vmin = 0.f, vmax = 0.f; 
  if (dataset != NULL)
  {
    vmax = (float)dataset->getMax();
    vmin = (float)dataset->getMin();
  }
  initDataRangeSlider("isovalue", 1, vmin,(vmax+vmin)/2.f,vmax); 
}

void 
GuiDialogViz::updateTransparencyType(SoGLRenderAction::TransparencyType newType)
{
  m_viewer->setTransparencyType(newType);
}

void
GuiDialogViz::setFSSA(float quality)
{
  m_viewer->setAntialiasing(quality,SoSceneManager::FSAA);
}

void 
GuiDialogViz::buildAuditorGeometryPanel()
{
  SoDialogGroup* geometryTab = (SoDialogGroup *)m_topLevelDialog->searchForAuditorId("geometrytab");
  if (geometryTab != NULL)
    geometryTab->addAuditor(new AuditorGeometry(this));

  for (int i=0;i<NUM_LOGICALSLICE;i++)
	  buildAuditorLogicalSlice(i);
	
  // skin check 
  SoDialogCheckBox* skinCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("skincheck");
  if (skinCheck != NULL) 
    skinCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchSkin()));

  // plane slice check 
  SoDialogCheckBox* sliceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("planeslicecheck");
  if (sliceCheck != NULL)
    sliceCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchPlaneSlice()));

  // plane slice clipping 
  SoDialogCheckBox* sliceClip = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("planesliceclipping");
  if (sliceClip != NULL && m_sceneGraph->getClipPlane()!=NULL)
  {
    SoClipPlane* clipPlane = m_sceneGraph->getClipPlane();
    clipPlane->on.connectFrom(&sliceClip->state);
  }

  // fence slice check 
  SoDialogCheckBox* fenceSliceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("fenceslicecheck");
  if (fenceSliceCheck != NULL)
    fenceSliceCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchFenceSlice()));

  // isosurface check 
  SoDialogCheckBox* isosurfaceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("isosurfacecheck");
  if (isosurfaceCheck != NULL) 
    isosurfaceCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchIsoSurface()));

  // mesh bounding box check 
  SoDialogCheckBox* boundingBoxCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("fencesliceselection");
  if (boundingBoxCheck != NULL) 
    boundingBoxCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchMeshBoundingBox()));

}

void 
GuiDialogViz::initGeometryPanel()
{
  // geometry auditor
  SoDialogGroup* geomPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("geometrytab"); 
  if (geomPanel != NULL) 
      geomPanel->applyAuditors();

  for (int i=0;i<NUM_LOGICALSLICE;i++)
	  initLogicalSlice(i);

  // skin check 
  SoDialogCheckBox* skinCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("skincheck");
  if (skinCheck != NULL)
    skinCheck->applyDlgCptAuditor();

      // plane slice check 
  SoDialogCheckBox* sliceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("planeslicecheck");
  if (sliceCheck != NULL)
    sliceCheck->applyDlgCptAuditor();

  // plane slice WireFrame check 
  SoDialogCheckBox* planeSliceWireFrameCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("planeslicewireframe");
  if (planeSliceWireFrameCheck != NULL)
    planeSliceWireFrameCheck->applyDlgCptAuditor();

  // isosurf init
  SoDialogCheckBox* isosurfaceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("isosurfacecheck");
  if (isosurfaceCheck != NULL)
    isosurfaceCheck->applyDlgCptAuditor();
}

void
GuiDialogViz::buildAuditorPropertiesPanel()
{
  // Properties auditor
  SoDialogGroup* propPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("proptab");
  if (propPanel != NULL)
    propPanel->addAuditor(new AuditorProperties(*this));
}

void
GuiDialogViz::initPropertiesPanel()
{
  bool porosityAvailable = m_grid->porositySize() > 0;
  // porosity display check 
  SoDialogCheckBox* poroCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("poroCheck");
  if (poroCheck != NULL)
  {
    poroCheck->enable = porosityAvailable;
    poroCheck->state = porosityAvailable;
    poroCheck->applyDlgCptAuditor();
  }

  // depth display check 
  SoDialogCheckBox* depthCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("depthCheck");
  if (depthCheck != NULL)
  {
    depthCheck->state = !porosityAvailable;
    depthCheck->applyDlgCptAuditor();
  }

  // properties auditor
  SoDialogGroup* propPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("proptab");
  if (propPanel != NULL)
    propPanel->applyAuditors();
}

void
GuiDialogViz::buildAuditorHullPanel()
{

    // outline display check 
  SoDialogCheckBox* hullOutlineCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("hulloutlinecheck");
  if (hullOutlineCheck != NULL) 
    hullOutlineCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchHullOutline()));

  // hull display check 
  SoDialogCheckBox* hullCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("hullcheck");
  if (hullCheck != NULL) 
    hullCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchHull()));

  // Hull auditor
  SoDialogGroup* hullPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("hulltab"); 
  if (hullPanel != NULL) 
    hullPanel->addAuditor(new AuditorHull(this));
}

void
GuiDialogViz::initHullPanel()
{

    // outline display check 
  SoDialogCheckBox* hullOutlineCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("hulloutlinecheck");
  if (hullOutlineCheck != NULL) 
    hullOutlineCheck->applyDlgCptAuditor();

    // hull display check 
  SoDialogCheckBox* hullCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId("hullcheck");
  if ( hullCheck!= NULL)
    hullCheck->applyDlgCptAuditor();

  // Hull auditor
  SoDialogGroup* hullPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("hulltab"); 
  if (hullPanel != NULL) 
      hullPanel->applyAuditors();
}

void
GuiDialogViz::buildAuditorRoiPanel()
{
  // Roi auditor
  SoDialogGroup* roiPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("roitab"); 
  if (roiPanel != NULL) 
    roiPanel->addAuditor(new AuditorFilter(this));
}

void
GuiDialogViz::initRoiPanel()
{
  // Roi auditor
  SoDialogGroup* roiPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("roitab"); 
  if (roiPanel != NULL) 
      roiPanel->applyAuditors();
}

void 
GuiDialogViz::buildAuditorLogicalSlice(int id)
{
  // slice check 
  std::ostringstream checkId;
  checkId << "logicalslicecheck" << id;
  SoDialogCheckBox* sliceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId(checkId.str().c_str());
  if (sliceCheck != NULL)
    sliceCheck->addAuditor(new AuditorCheckSwitch(m_sceneGraph->getSwitchLogicalSlice(id)));
}

void 
GuiDialogViz::initLogicalSlice(int id)
{
    // slice check 
    std::ostringstream checkId;
    checkId << "logicalslicecheck" << id;
    SoDialogCheckBox* sliceCheck = (SoDialogCheckBox *)m_topLevelDialog->searchForAuditorId(checkId.str().c_str());
    if (sliceCheck != NULL)
      sliceCheck->applyDlgCptAuditor();

    //slice number
    std::ostringstream numberId;
    numberId << "logicalslicenumber" << id;
    SoDialogIntegerSlider* sliceNumberDialog  = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId(numberId.str().c_str());
    if (sliceNumberDialog != NULL)
      sliceNumberDialog->applyDlgCptAuditor();
}

void 
GuiDialogViz::initRoiSlider(std::string sliderId, int /*roiBoundId*/, int roiInitValue, int roiMaxValue)
{
  SoDialogIntegerSlider *roiSlider = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId(sliderId.c_str());
  if (roiSlider != NULL)
  {
    roiSlider->min = 0;
    roiSlider->max = roiMaxValue;
    roiSlider->value = roiInitValue;
  }
}

void 
GuiDialogViz::initDataRangeSlider(std::string sliderId, int /*roiBoundId*/, float dataMinValue, float dataInitValue, float dataMaxValue)
{
  SoDialogRealSlider *dataSlider = (SoDialogRealSlider*)m_topLevelDialog->searchForAuditorId(sliderId.c_str());
  if (dataSlider != NULL)
  {
    dataSlider->min = dataMinValue;
    dataSlider->max = dataMaxValue;
    dataSlider->value = dataInitValue;
  }
}

void
GuiDialogViz::buildAuditorAnimationPanel()
{
  SoDialogGroup* animationPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("animationtab"); 
  if (animationPanel != NULL)
    animationPanel->addAuditor(new AuditorAnimation(this));
}

void
GuiDialogViz::initAnimationPanel()
{
  if (!m_init)
  {
    SoDialogGroup* animationPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("animationtab"); 
    if (animationPanel != NULL)
      animationPanel->applyAuditors();
  }
  else
  {
    SoDialogIntegerSlider* animationSlider = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId("animationslider"); 
    if (animationSlider != NULL)
      animationSlider->applyDlgCptAuditor();
  }
}

void 
GuiDialogViz::buildAuditorDrawStylePanel()
{
  // DrawStyle auditor
  SoDialogGroup* dsPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("drawstyletab"); 
  if (dsPanel != NULL) 
    dsPanel->addAuditor(new AuditorDrawStyle(this));
}

void
GuiDialogViz::initDrawStylePanel()
{
  // DrawStyle auditor
  SoDialogGroup* dsPanel = (SoDialogGroup*)m_topLevelDialog->searchForAuditorId("drawstyletab"); 
  if (dsPanel != NULL) 
    dsPanel->applyAuditors();
}

void 
GuiDialogViz::initInfoPanel()
{
  SoDialogEditText *textDialog = (SoDialogEditText*)m_topLevelDialog->searchForAuditorId("info"); 
  if (textDialog != NULL && m_grid != NULL && m_eclipseMesh != NULL)
  {
    const char* memUnit[3] = { "KB", "MB", "GB"};
    size_t unit = 0;
    double gridMemoryFp = m_grid->getMemoryFootPrint()/1024.0;
    double meshMemoryFp = m_eclipseMesh->getMemoryFootPrint()/ 1024.0;
    while (gridMemoryFp > 1024 && unit < 3)
    {
      gridMemoryFp /= 1024.0;
      meshMemoryFp /= 1024.0;
      ++unit;
    };
    std::ostringstream gridMemFpString;
    gridMemFpString.precision(1);
    gridMemFpString << std::fixed << gridMemoryFp;
    std::ostringstream meshMemFpString;
    meshMemFpString.precision(1);
    meshMemFpString << std::fixed << meshMemoryFp;

    textDialog->editText = SbString("Grid dimensions I/J/K: ") + int(m_grid->get_dimI()) + " x " + int(m_grid->get_dimJ()) + " x " + int(m_grid->get_dimK())
      + "\n#Active cells : " +  int(m_eclipseMesh->getNumActiveCells())
      + "\nGrid memory footprint : " + gridMemFpString.str() + memUnit[unit]
      + "\nWrappers memory footprint : " + meshMemFpString.str() + memUnit[unit];
  }
}

void
GuiDialogViz::enableAutoDownSampling(bool enable)
{
  SoDialogIntegerSlider *resolutionSlider = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId("resolutionslider");
  resolutionSlider->enable = !enable && m_sceneGraph->availableLowResolution();
  SoDialogIntegerSlider *delaySlider = (SoDialogIntegerSlider*)m_topLevelDialog->searchForAuditorId("fullresdelay");
  delaySlider->enable = enable;
  m_sceneGraph->autoLowResolutionAdjust(enable,resolutionSlider->value);

}

void 
GuiDialogViz::viewerFPSCB( float fps, void *userData, SoXtViewer* /*viewer*/ )
{
  OivSceneGraph* currentSceneGraph = (OivSceneGraph*) userData;
  currentSceneGraph->OivSceneGraph::viewerFPSCB(  fps );
}
