#ifndef _GRID_H_
#define _GRID_H_

#include "DemoSettings.h"
#include <vector>

#define UNDEF     -999.0

class Grid
{
public:
  friend class PillarGridReader;
  friend class PillarGridGenerator;
  friend class PillarGridFileIO;

  virtual ~Grid() {}

  void get_dim(size_t dim[3]) const ;
  size_t get_dimI() const ;
  size_t get_dimJ() const ;
  size_t get_dimK() const ;
  void getMinMax(CoordT min[3], CoordT max[3]) const;
  void getMinMax(CoordT min[3], CoordT max[3]);
  size_t getNumZCorns() const { return m_numZcorns; }
  size_t getNumNonActiveCells() const;
  bool empty() const;

  static CoordT m_undef;

  size_t getMemoryFootPrint() const;

  float getPoro(int i) const { return m_poro[i]; }
  size_t porositySize() const {return m_poro.size();}


protected:
  Grid();
  bool isValid(size_t id) const;
  int generate_pillar_vertex(CoordT P12[6],CoordT zcorn, CoordT xyz[3]) const;
  void clear();

  // Size of the mesh 
  size_t m_iDim;
  size_t m_jDim;
  size_t m_kDim;
  size_t m_numZcorns;
  //number of horizontal faces at a K level on pillars for a given I and J cell position (equals 1 if not storing Horizontal Faults, 2 otherwise)
  //Default is 2 like EGRID storage supporting horizontal faults, but PillarGridGenerator set it to 1 when no HORIZONTAL_FAULTS is selected
  size_t m_numHFacesAtKLevel;

  // mesh data
  std::vector<CoordT> m_coord; //6*(ny+1)*(nx+1)
  std::vector<CoordT> m_zcorn;
  std::vector<ActiveT> m_actnum; //nx*ny*nz

  // porosity data
  std::vector<float> m_poro;

  size_t m_numNonActiveCells;    // number of non active cells

private:
  virtual void setDim(size_t numCellI, size_t numCellJ, size_t numCellK) = 0;
  void updateMinMax();

  // computed meshdata
  bool m_minMaxUpdated;
  CoordT m_min[3];
  CoordT m_max[3];
};

inline void
Grid::get_dim(size_t dim[3]) const
{
  dim[0] = m_iDim;
  dim[1] = m_jDim;
  dim[2] = m_kDim;
}

inline size_t
Grid::get_dimI() const
{
  return m_iDim;
}

inline size_t
Grid::get_dimJ() const
{
  return m_jDim;
}

inline size_t
Grid::get_dimK() const
{
  return m_kDim;
}

inline void
Grid::getMinMax(CoordT min[3], CoordT max[3]) const
{
  min[0] = m_min[0];
  min[1] = m_min[1];
  min[2] = m_min[2];
  max[0] = m_max[0];
  max[1] = m_max[1];
  max[2] = m_max[2];
}

inline void
Grid::getMinMax(CoordT min[3], CoordT max[3])
{
  if (!m_minMaxUpdated)
    updateMinMax();
  min[0] = m_min[0];
  min[1] = m_min[1];
  min[2] = m_min[2];
  max[0] = m_max[0];
  max[1] = m_max[1];
  max[2] = m_max[2];
}

inline size_t 
Grid::getNumNonActiveCells() const
{
  return m_numNonActiveCells;
}

inline bool 
Grid::isValid(size_t id) const
{
  if(m_actnum[id]) return true;
  else return false;
}

inline size_t
Grid::getMemoryFootPrint() const
{
  return (sizeof(CoordT) * (m_coord.size() + m_zcorn.size()) + sizeof(ActiveT) * m_actnum.size());
}

#endif


