#include <cstdio>
#include <string.h>
#include <iostream>

#include <Inventor/STL/string>
#include <Inventor/STL/cassert>
#include <Inventor/STL/limits>
#include <Inventor/STL/algorithm>
#include "PillarEGrid.h"

void PillarEGrid::setDim(size_t numCellI, size_t numCellJ, size_t numCellK)
{
  clear();
  m_iDim = numCellI;
  m_jDim = numCellJ;
  m_kDim = numCellK;
  m_2DimI = 2*m_iDim;
  m_4DimIDimJ = 2*m_2DimI*m_jDim;
  // if horizontal faults enable count 2 times more zcorns (8 * ni * nj *nk)
  // otherwise only (4 * ni * nj * (nk+1))
  m_numZcorns = m_numHFacesAtKLevel * m_4DimIDimJ * (m_kDim + 2 - m_numHFacesAtKLevel);

  //allocate pillar coordinates
  size_t npillars = (m_iDim+1) * (m_jDim+1);
  size_t ndata = 6*npillars;
  m_coord.resize(ndata,0);

  //allocate z corner
  m_zcorn.resize(m_numZcorns,0);

  //allocate actnum
  ndata = m_iDim * m_jDim * m_kDim;
  m_actnum.resize(ndata,1);

  //allocate porosity 
  m_poro.resize(ndata,0);
}

void PillarEGrid::computePillarVertex(size_t zcornId, CoordT xyz[3]) const
{
  CoordT P12[6];
  size_t pillar_index;
  size_t dij = zcornId % m_4DimIDimJ;
  size_t dj = dij / m_2DimI;
  size_t j = dj / 2;
  size_t jPos = dj % 2;
  size_t di = dij % m_2DimI;
  size_t i = di / 2;
  size_t iPos = di % 2;

  pillar_index = (j+jPos) * (m_iDim+1) + (i+iPos);
  memcpy(P12,&m_coord[pillar_index*6],sizeof(CoordT)*6);

  generate_pillar_vertex(P12,m_zcorn[zcornId],xyz);
}

void PillarEGrid::getPillarZCorns(size_t i, size_t j, size_t k, std::pair<CoordT,size_t> indexedZCorns[8]) const
{
  size_t zcorn_index = m_numHFacesAtKLevel*k*m_4DimIDimJ + 2*j* m_2DimI + 2*i;
#if HORIZONTAL_FAULTS
  for (size_t n = 0; n < 2; ++n)
  {
    indexedZCorns[0].first = indexedZCorns[1].first = indexedZCorns[2].first = indexedZCorns[3].first = m_undef;
    if (k < m_kDim)
    {
#else
  indexedZCorns[0].first = indexedZCorns[1].first = indexedZCorns[2].first = indexedZCorns[3].first = m_undef;
  if (k == m_kDim && m_numHFacesAtKLevel == 2)
    zcorn_index -= m_4DimIDimJ;
#endif
  if (j < m_jDim)
  {
    if(i < m_iDim)
    {
      indexedZCorns[0].second = (NodeIdT) zcorn_index;
      indexedZCorns[0].first = m_zcorn[zcorn_index];
    }
    if (i > 0)
    {
      indexedZCorns[1].second = (NodeIdT) zcorn_index - 1;
      indexedZCorns[1].first = m_zcorn[zcorn_index - 1];
    }
  }
  if (j > 0)
  {
    if (i > 0)
    {
      indexedZCorns[2].second =  (NodeIdT) (zcorn_index - m_2DimI - 1);
      indexedZCorns[2].first = m_zcorn[indexedZCorns[2].second];
    }
    if (i < m_iDim)
    {
      indexedZCorns[3].second =  (NodeIdT) (zcorn_index - m_2DimI);
      indexedZCorns[3].first = m_zcorn[indexedZCorns[3].second];
    }
  }
#if HORIZONTAL_FAULTS
    }
    // collect zcorns belonging to the lower face at the same K level
    indexedZCorns += 4;
    --k;
    zcorn_index -= m_4DimIDimJ;
  }
#endif
}

CoordT PillarEGrid::get_zcorn(size_t i, size_t j, size_t k) const
{
  size_t zcorn_index = m_numHFacesAtKLevel*k*m_4DimIDimJ + 2*j * m_2DimI + 2*i;
  return m_zcorn[zcorn_index];
}

int PillarEGrid::generate_cell_vertices(size_t i,size_t j,size_t k,CoordT xyz[24], const size_t reindex[2][2][2]) const
{
  CoordT P12[6];
  size_t pillar_index;
  size_t zcorn_index;
  CoordT zcorn;
  CoordT v[3];
  int res=0;

  for (size_t jPos=0;jPos<=1;jPos++)
  {
    for (size_t iPos=0;iPos<=1;iPos++)
    {
      for (size_t kPos=0;kPos < 2; kPos++)
      {
        pillar_index = (j+jPos) * (m_iDim+1) + (i+iPos);
        assert( pillar_index < ((m_iDim+1) * (m_jDim+1)) );

        memcpy(P12,&m_coord[pillar_index*6],sizeof(CoordT)*6);
        zcorn_index = (m_numHFacesAtKLevel*k+kPos)*m_4DimIDimJ + (2*j+jPos) * m_2DimI + (2*i+iPos);
        assert(zcorn_index< (m_numHFacesAtKLevel * m_kDim + 1) * m_4DimIDimJ);

        zcorn = m_zcorn[zcorn_index];

        res = generate_pillar_vertex(P12,zcorn,v);
        memcpy(xyz+(3*reindex[iPos][jPos][kPos]),v,sizeof(CoordT)*3);

        if (res!=1)
          return(res);
      }
    }
  }

  return(1);
}

CoordT PillarEGrid::get_porosity(size_t i, size_t j, size_t k) const
{
  size_t poro_index = m_jDim*m_iDim*k+ m_iDim*j + i;
  return m_poro[poro_index];
}
