#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/string>
#include <Inventor/SbElapsedTime.h> 
#include <Inventor/helpers/SbFileHelper.h>
#include "PillarGridFileIO.h"

void display_time(const std::string &what, double timeElapsed);

void PillarGridFileIO::write(const Grid& grid, const std::string& filename) const
{
  std::cout << "\nsaving pillar grid into " << filename << std::endl;
  SbElapsedTime localTime;
  double t1 = localTime.getElapsed();
  std::ofstream infile(filename.c_str(),std::ofstream::binary);

  // writes grid dimensions
  infile.write((char*)&grid.m_iDim, sizeof(size_t));
  infile.write((char*)&grid.m_jDim, sizeof(size_t));
  infile.write((char*)&grid.m_kDim, sizeof(size_t));
  infile.write((char*)&grid.m_numHFacesAtKLevel, sizeof(size_t));

  // writes datatype size
  size_t CoorTSize = sizeof(CoordT);
  infile.write((char*)&CoorTSize, sizeof(size_t));
  size_t ActiveTSize = sizeof(ActiveT);
  infile.write((char*)&ActiveTSize, sizeof(size_t));

  // writes num zcorn
  infile.write((char*)&grid.m_numZcorns, sizeof(size_t));
  
  // writes data
  infile.write((char*)&grid.m_zcorn[0], grid.m_numZcorns*CoorTSize);
  infile.write((char*)&grid.m_coord[0], grid.m_coord.size()*CoorTSize);
  infile.write((char*)&grid.m_numNonActiveCells, sizeof(size_t));
  if (grid.m_numNonActiveCells > 0)
    infile.write((char*)&grid.m_actnum[0], grid.m_iDim*grid.m_jDim*grid.m_kDim*ActiveTSize);

  // writes porosity 

	bool flag = grid.porositySize() !=0;

	infile.write((char*)&flag, sizeof(bool));

  if (flag)
  {
    infile.write((char*)&grid.m_poro[0], grid.m_iDim*grid.m_jDim*grid.m_kDim*sizeof(float));
  }


  double t1bis = localTime.getElapsed();
  display_time("pillar grid saved",t1bis-t1);
}

void PillarGridFileIO::read(Grid& grid, const std::string& filename) const
{
  std::cout << "\nreading mesh from " << filename << std::endl;
  SbElapsedTime localTime;
  double t1 = localTime.getElapsed();
  FILE* fp = SbFileHelper::open( filename.c_str() , "rb" );

  uint64_t idim, jdim, kdim;
  // read grid dimensions
  fread (&idim, sizeof(uint64_t), 1, fp);
  fread(&jdim, sizeof(uint64_t), 1, fp);
  fread(&kdim, sizeof(uint64_t), 1, fp);
  fread(&grid.m_numHFacesAtKLevel, sizeof(uint64_t), 1, fp);
  grid.setDim(idim,jdim,kdim);

  // read number of zcorn and datatype size then check with pillargrid type
  uint64_t CoorTSize;
  fread(&CoorTSize, sizeof(uint64_t), 1, fp);
  uint64_t ActiveTSize;
  fread(&ActiveTSize, sizeof(uint64_t), 1, fp);

  uint64_t storedNumZcord;
  fread(&storedNumZcord, sizeof(uint64_t), 1, fp);

  // check file: required number of zcorns 
  if (storedNumZcord == grid.m_numZcorns && (!HORIZONTAL_FAULTS || grid.m_numHFacesAtKLevel == 2)
    && CoorTSize == sizeof(CoordT) && ActiveTSize == sizeof(ActiveT))
  {
    fread (&grid.m_zcorn[0] , CoorTSize, grid.m_numZcorns, fp);
    fread (&grid.m_coord[0] , CoorTSize, 6*(jdim+1)*(idim+1), fp);
    fread(&grid.m_numNonActiveCells, sizeof(uint64_t), 1, fp);
    if (grid.m_numNonActiveCells > 0)
      fread (&grid.m_actnum[0] , ActiveTSize, grid.m_iDim*grid.m_jDim*grid.m_kDim, fp);

    //read porosity
    bool flag ;
    fread (&flag, sizeof(bool), 1, fp);
    if (flag == true)
    {
      grid.m_poro.resize(grid.m_iDim*grid.m_jDim*grid.m_kDim);
      fread (&grid.m_poro[0], sizeof(float), grid.m_iDim*grid.m_jDim*grid.m_kDim , fp);
    }

    grid.updateMinMax();

    double t1bis = localTime.getElapsed();
    display_time("pillar grid read",t1bis-t1);
  }
  else
  {
    std::cout << "Wrong File format" << std::endl;
    grid.clear();
  }
  SbFileHelper::close(fp);
}
