#include <cstdio>
#include <Inventor/STL/string>
#include <Inventor/STL/algorithm>
#include <Inventor/STL/limits>
#include "pillargrid/PillarGridGenerator.h"

#include <Inventor/SbElapsedTime.h> 
#include <Inventor/helpers/SbFileHelper.h>

void display_time(const std::string &what, double timeElapsed);

size_t PillarGridGenerator::m_neighbourPillar[4][2] = 
{
  {0,0},
  {1,0},
  {1,1},
  {0,1}
};

int PillarGridGenerator::m_neighbourCells[4][2] = 
{
  {0,0},
  {-1,0},
  {-1,-1},
  {0,-1}
};

PillarGridGenerator::PillarGridGenerator()
{
}

PillarGridGenerator::~PillarGridGenerator()
{
  clear();
}

void PillarGridGenerator::clear()
{
  for ( size_t i=0; i<m_levelFctors.size(); ++i)
    delete m_levelFctors[i];
  m_levelFctors.clear();
  for ( size_t i=0; i<m_crackFctors.size(); ++i)
    delete m_crackFctors[i];
  m_crackFctors.clear();
}

void PillarGridGenerator::addLevelFunctor( const LevelFunctor* levelFct)
{
  m_levelFctors.push_back(levelFct);
}

void PillarGridGenerator::addCrackFunctor( const CrackFunctor* crackFct)
{
  m_crackFctors.push_back(crackFct);
}

size_t PillarGridGenerator::generateNonActiveCells(PillarGridJIK& mesh, float /*maxX*/, float /*maxY*/)
{
  mesh.m_numNonActiveCells = mesh.m_iDim * mesh.m_jDim * mesh.m_kDim;
  mesh.m_actnum.assign(mesh.m_numNonActiveCells,0);
  for (size_t j = 2; j < mesh.m_jDim-2; ++j)
    for (size_t i = 2; i < mesh.m_iDim-2; ++i)
    {
        size_t pos = (j*mesh.m_iDim*mesh.m_kDim) + i*mesh.m_kDim;
        for (size_t k = 0; k < mesh.m_kDim; ++k, ++pos)
        {
          mesh.m_actnum[pos] = 1;
          --mesh.m_numNonActiveCells;
        }
    }
  return mesh.m_numNonActiveCells;
}

size_t PillarGridGenerator::generateNonActiveCells(PillarEGrid& mesh, float /*maxX*/, float /*maxY*/)
{
   return mesh.m_numNonActiveCells;
}

size_t PillarGridGenerator::generateCoord(PillarGrid& mesh, float maxX, float maxY)
{
	size_t nfloats = 0;
  size_t numLevel = m_levelFctors.size()-1;
  float x, y;
  float stepI = maxX / (float) mesh.m_iDim;
  float stepJ = maxY / (float) mesh.m_jDim;
  x = y = 0.f;

  for (size_t j = 0; j < (size_t) mesh.m_jDim+1; ++j)
    for (size_t i = 0; i < (size_t) mesh.m_iDim+1; ++i)
    {
      x = i * stepI;
      y = j * stepJ;
      mesh.m_coord[nfloats++] = (CoordT) x;
      mesh.m_coord[nfloats++] = (CoordT) y;
      mesh.m_coord[nfloats++] = (CoordT) (*m_levelFctors[0])(x/maxX,y/maxY);
      mesh.m_coord[nfloats++] = (CoordT) x;
      mesh.m_coord[nfloats++] = (CoordT) y;
      mesh.m_coord[nfloats++] = (CoordT) (*m_levelFctors[numLevel])(x/maxX,y/maxY);
    }

	return nfloats;
}

size_t PillarGridGenerator::generateZCornTopDown(PillarEGrid& mesh, float maxX, float maxY)
{
  size_t numCells = 0;
  size_t zcornId;
  float x, y, z, offset;
  size_t numLevels = m_levelFctors.size()-1;
  size_t numLayerPerLevel = mesh.m_kDim / numLevels;
  float height = 0;
  float floor;
  size_t dimIxJ = mesh.m_iDim*mesh.m_jDim;
  float stepI = maxX / (float) mesh.m_iDim;
  float stepJ = maxY / (float) mesh.m_jDim;

  //for each level in mesh (starting from top)
  for( size_t l = numLevels; l > 0; --l)
  {
    size_t kstart = l * numLayerPerLevel - 1;
    size_t kstop = (l-1) * numLayerPerLevel;
    if( l == numLevels )
      kstart = mesh.m_kDim-1;
    float layerFloorHeight = 0;
    float stepK = ((*m_levelFctors[l])(0,0) - (*m_levelFctors[l-1])(0,0) ) / (float) (kstart + 1 - kstop);
    for (size_t k = kstart; k > kstop-1; --k)
    {
      for (size_t j = 0; j < mesh.m_jDim; ++j)
        for (size_t i = 0; i < mesh.m_iDim; ++i)
        {
#if HORIZONTAL_FAULTS
          size_t numPos = 2;
#else
          size_t numPos = k < mesh.m_kDim-1 ? 1 : 2;
#endif
          for ( size_t kPos = 0; kPos < numPos; ++kPos)
          {
            height = (1-kPos) * stepK;
            for (size_t n=0; n<4 ; ++n)
            {
              x = (i+m_neighbourPillar[n][0]) * stepI;
              y = (j+m_neighbourPillar[n][1]) * stepJ;
              offset = computeOffset(i,j);
#if HORIZONTAL_FAULTS
              if(k<mesh.m_kDim-1 && kPos>0)
              {
                zcornId = (mesh.m_numHFacesAtKLevel*(k+1))*(4*dimIxJ) + (2*j+m_neighbourPillar[n][1]) * (2*mesh.m_iDim) + (2*i+m_neighbourPillar[n][0]);
                z = mesh.m_zcorn[zcornId];
              }
              else
#endif
                z = (*m_levelFctors[l])(x/maxX,y/maxY) - height - layerFloorHeight + offset;
              floor = (*m_levelFctors[l-1])(x/maxX,y/maxY) + offset;
              zcornId = (mesh.m_numHFacesAtKLevel*k+kPos)*(4*dimIxJ) + (2*j+m_neighbourPillar[n][1]) * (2*mesh.m_iDim) + (2*i+m_neighbourPillar[n][0]);
              if( z - floor < 1e-4)
                mesh.m_zcorn[zcornId] = (CoordT) floor;
              else
                mesh.m_zcorn[zcornId] = (CoordT) z;
            }
          }
          ++numCells;
        }
        layerFloorHeight += stepK;
    }
  }

  return numCells*8;
}

size_t PillarGridGenerator::generateZCornTopDown(PillarGridJIK& mesh, float maxX, float maxY)
{
  size_t numZcorns = 0;
  size_t zcornId, ni, nj;
  float x, y, z, offset, value;
  size_t numLevels = m_levelFctors.size()-1;
  size_t numLayerPerLevel = mesh.m_kDim / numLevels;
  float floor;
  float stepI = maxX / (float) mesh.m_iDim;
  float stepJ = maxY / (float) mesh.m_jDim;

  //for each level in mesh (starting from top)
  for( size_t l = numLevels; l> 0; --l)
  {
    size_t kstart = l == numLevels ? mesh.m_kDim : l * numLayerPerLevel;
    size_t kstop = (l-1) * numLayerPerLevel;
    float stepK = ((*m_levelFctors[l])(0,0) - (*m_levelFctors[l-1])(0,0) ) / (float) (kstart-kstop);
    for (size_t j = 0; j < (size_t)mesh.m_jDim+1; ++j)
      for (size_t i = 0; i < (size_t)mesh.m_iDim+1; ++i)
      {
        x = i* stepI;
        y = j* stepJ;
        floor = (*m_levelFctors[l-1])(x/maxX,y/maxY); 
        z = (*m_levelFctors[l])(x/maxX,y/maxY);
        zcornId = mesh.computeZCornId(i,j,kstart);
        for ( size_t k = kstart; k >= kstop; --k, z -= stepK)
        {
          value = ( z -floor < 1e-4) ? floor : z;
          for (size_t n=0; n<4 ; ++n, ++zcornId)
          {
            ni = i+m_neighbourCells[n][0];
            nj = j+m_neighbourCells[n][1];
            if (ni < mesh.m_iDim && nj <mesh.m_jDim)
            {
              offset = computeOffset(ni,nj);
              if (!HORIZONTAL_FAULTS || k < mesh.m_kDim)
              {
                mesh.m_zcorn[zcornId] = (CoordT) value + offset;
                ++numZcorns;
              }
#if HORIZONTAL_FAULTS
              if (k != 0)
              {
                mesh.m_zcorn[zcornId+4] = (CoordT) value + offset;
                ++numZcorns;
              }
#endif
            }
          }
          zcornId -= (mesh.m_numCorners+4);
        }
      }
  }
  return numZcorns;
}

float PillarGridGenerator::computeOffset(size_t i, size_t j)
{
	float offset = 0;
  size_t numCracks = m_crackFctors.size();
  for (size_t n=0; n<numCracks; ++n)
  {
    offset += (*m_crackFctors[n])(i,j);
  }
	return offset;
}

void PillarGridGenerator::generate(PillarGrid& pillarGrid, size_t numCellI, size_t numCellJ, size_t numCellK, float maxX, float maxY) 
{
  pillarGrid.clear();

  SbElapsedTime localTime;
  if ( m_levelFctors.size() >= 2 )
  {
    double t1 = localTime.getElapsed();
    // set the number of vertex to generate at a k level for a given i j position
    pillarGrid.m_numHFacesAtKLevel = HORIZONTAL_FAULTS ? 2 : 1;
    pillarGrid.setDim(numCellI,numCellJ,numCellK);

    generateCoord(pillarGrid,maxX,maxY);
    generateZCornTopDown(pillarGrid,maxX,maxY);

    generateNonActiveCells(pillarGrid,maxX,maxY);

    double t1bis = localTime.getElapsed();

    display_time("PillarGrid generated",t1bis-t1);

  }
  else
  {
    pillarGrid.clear();
    std::cout<< "PillarGridGenerator: Bad number of level functions. Can not generate synthetic mesh" << std::endl;
  }
}	

void PillarGridGenerator::generate(PillarGrid& pillarGrid, const std::string& filename)
{
  std::cout << "\ngenerating pillar grid from " << filename << std::endl;
  clear();
  FILE* fp;
  size_t ni,nj,nk;
  float a1,b1,a2,b2,c;
  char keyword[256];
  fp = SbFileHelper::open( filename.c_str() , "r" );
  if (fp == NULL )
  {
    std::cout << "\nno file " << filename << std::endl;
    pillarGrid.clear();
  }
  int d[3];
  int ndata = fscanf(fp,"%d %d %d %f %f",&d[0],&d[1],&d[2],&pillarGrid.m_max[0],&pillarGrid.m_max[1]);
  if ( ndata < 5 )
  {
    std::cout<< "PillarGridGenerator: file format not supported" << std::endl;
    pillarGrid.clear();
  }
  ni = d[0];
  nj = d[1];
  nk = d[2];
  float crackmax = 0;
  float crackmin = 0;

  fscanf(fp,"%s",keyword);
  while (!feof(fp)) {
    if( strcmp(keyword,"const")==0 )
    {
      fscanf(fp,"%f",&c);
      m_levelFctors.push_back(new ConstantFctor(c));
    }
    else if( strcmp(keyword,"cos")==0 )
    {
      fscanf(fp,"%f %f %f",&a1,&b1,&c);   
      m_levelFctors.push_back(new CosFctor(a1,b1,c));
    }
    else if( strcmp(keyword,"sin")==0 )
    {
      fscanf(fp,"%f %f %f",&a1,&b1,&c);   
      m_levelFctors.push_back(new SinFctor(a1,b1,c));
    }
    else if( strcmp(keyword,"cossin")==0 )
    {
      fscanf(fp,"%f %f %f %f %f",&a1,&b1,&a2,&b2,&c);   
      m_levelFctors.push_back(new CosSinFctor(a1,b1,a2,b2,c));
    }
    else if( strcmp(keyword,"stepicrack")==0 )
    {
      fscanf(fp,"%f %f",&a1,&c);
      m_crackFctors.push_back(new StepICrackFctor((size_t)(ni*a1),c));
      if (c > 0) crackmax += c;
      if (c < 0) crackmin += c;
    }
    else if( strcmp(keyword,"stepjcrack")==0 )
    {
      fscanf(fp,"%f %f",&a1,&c);
      m_crackFctors.push_back(new StepJCrackFctor((size_t)(nj*a1),c));
      if (c > 0) crackmax += c;
      if (c < 0) crackmin += c;
    }
    else if( strcmp(keyword,"linearcrack")==0 )
    {
      fscanf(fp,"%f %f %f",&a1,&b1,&c);
      m_crackFctors.push_back(new LinearCrackFctor(b1,nj*a1,c));
      if (c > 0) crackmax += c;
      if (c < 0) crackmin += c;
    }
    fscanf(fp,"%s",keyword);
  }
  fclose(fp);

  pillarGrid.m_min[0] = 0;
  pillarGrid.m_min[1] = 0;
  pillarGrid.m_min[2] = std::numeric_limits<float>::max();
  pillarGrid.m_max[2] = std::numeric_limits<float>::min();
  for (size_t level = 0; level < m_levelFctors.size(); ++level)
  {
    pillarGrid.m_max[2] = std::max(pillarGrid.m_max[2],m_levelFctors[level]->getMax());
    pillarGrid.m_min[2] = std::min(pillarGrid.m_min[2],m_levelFctors[level]->getMin());
  }

  pillarGrid.m_max[2] += crackmax;
  pillarGrid.m_min[2] += crackmin;
  pillarGrid.m_minMaxUpdated = true;

  generate(pillarGrid,ni,nj,nk,pillarGrid.m_max[0],pillarGrid.m_max[1]);

  // no porosity generated
  pillarGrid.m_poro.clear();
}
