#include <cstdio>
#include <iostream>
#include <string.h>
#include <Inventor/STL/string>
#include <Inventor/STL/cassert>
#include <Inventor/helpers/SbFileHelper.h>
#include "PillarGridJIK.h"

PillarGridJIK::PillarGridJIK():
  m_numCorners(HORIZONTAL_FAULTS ? 8 : 4)
{
}

void 
PillarGridJIK::setDim(size_t numCellI, size_t numCellJ, size_t numCellK)
{
  clear();
  m_iDim = numCellI;
  m_jDim = numCellJ;
  m_kDim = numCellK;
  m_2DimI = 2*m_iDim;
  m_4DimIDimJ = 2*m_2DimI*m_jDim;
  m_numCorners = m_numHFacesAtKLevel * 4;
  // gap between a 2 zcorn indices in the same cell is 4 when only 4 corners at a k level, 12 if 8 corners
  m_upperFaceZCornIdShift = m_numHFacesAtKLevel == 1 ? 4 : 12;
  m_numPillarVertices = m_numCorners*(m_kDim+1);
  m_numPillarSliceIVertices = m_numCorners*(m_kDim+1)*(m_iDim+1);
  m_numZcorns = m_numPillarSliceIVertices * (m_jDim+1);


  //allocate pillar coordinates
  size_t npillars = (m_iDim+1) * (m_jDim+1);
  size_t ndata = 6*npillars;
  m_coord.resize(ndata,0);

  //allocate z corner (8 or 4 per each level in each pillar)
  m_zcorn.resize(m_numZcorns,m_undef);

  //allocate actnum
  ndata = m_iDim * m_jDim * m_kDim;
  m_actnum.resize(ndata,1);

  m_numZcornAdded = 0;
  m_numActNumAdded = 0;
  m_numPoroAdded = 0; 
}

void 
PillarGridJIK::computePillarVertex(size_t zcornId, CoordT xyz[3]) const
{
  CoordT P12[6];

  size_t pillar_index = zcornId / m_numPillarVertices;
  memcpy(P12,&m_coord[pillar_index*6],sizeof(CoordT)*6);

  generate_pillar_vertex(P12,m_zcorn[zcornId],xyz);
}

size_t 
PillarGridJIK::getPillarZCorns(size_t i, size_t j, size_t k, const CoordT*& zcorns) const
{
  size_t zcorn_index = computeZCornId(i,j,k);
#if HORIZONTAL_FAULTS == 0
  if (k == m_kDim && m_numCorners == 8)
    zcorn_index += 4;
#endif
  zcorns = &m_zcorn[zcorn_index];
  return zcorn_index;
}

void 
PillarGridJIK::pushBackEGridZCorn(CoordT zcorn)
{
  setEGridZCorn(m_numZcornAdded,zcorn);
  ++m_numZcornAdded;
}

void 
PillarGridJIK::pushBackEGridPoro(float poro)
{
  setEGridPoro(m_numPoroAdded,poro);
  ++m_numPoroAdded;
}

void 
PillarGridJIK::pushBackEGridActNum(ActiveT actnum)
{
  setEGridActNum(m_numActNumAdded,actnum);
  ++m_numActNumAdded;
}

void 
PillarGridJIK::setEGridZCorn(size_t eGridZcornId, CoordT zcorn)
{
  
  size_t dk = eGridZcornId / m_4DimIDimJ;
  size_t k = dk / 2;
  size_t kPos = dk % 2;
  size_t dij = eGridZcornId % m_4DimIDimJ;
  size_t dj = dij / m_2DimI;
  size_t j = dj / 2;
  size_t jPos = dj % 2;
  size_t di = dij % m_2DimI;
  size_t i = di / 2;
  size_t iPos = di % 2;
  size_t ZcornPos = kPos<<2 | jPos<<1 | (iPos^jPos);
  size_t reindexedZcorn = (j+jPos)*m_numPillarSliceIVertices + (i+iPos)*m_numPillarVertices + m_numCorners*(k + kPos)  + ZcornPos;

  m_zcorn[reindexedZcorn] = zcorn;
 
}

void 
PillarGridJIK::setEGridPoro(size_t eGridPoroId, CoordT poro)
{
  size_t k = eGridPoroId / (m_iDim * m_jDim);
  size_t ij = eGridPoroId % (m_iDim * m_jDim);
  size_t j = ij / m_iDim;
  size_t i = ij % m_iDim;
  size_t reindexedPoro = (j*m_iDim*m_kDim)+i*m_kDim+k;

  m_poro[reindexedPoro] = poro;
}

void 
PillarGridJIK::setEGridActNum(size_t eGridActNumId, ActiveT actnum)
{
  size_t k = eGridActNumId / (m_iDim * m_jDim);
  size_t ij = eGridActNumId % (m_iDim * m_jDim);
  size_t j = ij / m_iDim;
  size_t i = ij % m_iDim;
  size_t reindexedActNum = (j*m_iDim*m_kDim)+i*m_kDim+k;

  m_actnum[reindexedActNum] = actnum;
}




