#ifndef _MeshResolutionSwitch_H
#define _MeshResolutionSwitch_H

#include <Inventor/nodes/SoSwitch.h>

class SoAlarmSensor;

/**
 *  This custom SoSwitch handles an automatic switch between its 2 children:
 *   - children 0 corresponds to the full resolution mesh.
 *   - children 1 corresponds to the low resolution mesh.
 *
 *  The low resolution mesh is selected by calling
 *  switchToLowResolution(). This methods select the children 1
 *  (low resolution mesh) and schedules an alarm to go back automatically
 *  to the full resolution after a delay defined by setFullResolutionDelay().
 *
 *  If enableFullResolutionScheduling(false) is called,
 *  the alarm is not scheduled after calling switchToLowResolution()
 *  and the application stays in low resolution mesh.
 */
class MeshResolutionSwitch : public SoSwitch
{
public:
  MeshResolutionSwitch();
  ~MeshResolutionSwitch();

  /**
  * Set the scene graph used for the full resolution rendering mode.
  */
  void setFullResolutionSceneGraph(SoNode* fullResolutionSceneGraph);

  /**
  * Set the scene graph used as the low resolution rendering mode.
  */
  void setLowResolutionSceneGraph(SoNode* lowResolutionSceneGraph);

  /**
  * Defines the delay to go to full resolution
  * after the low resolution has been selected.
  */
  void setFullResolutionDelay(int second);

  /**
  * If FullResolutionScheduling is not enabled the alarm 
  * is unscheduled, thus the switch back to the full 
  * resolution is not automatic and it could occur
  * only when calling switchToFullResolution().
  */
  void enableFullResolutionScheduling(bool active);

  /**
  * Selects the low resolution mesh if FullResolutionScheduling
  * is enabled and schedules the alarm to go back to
  * the full resolution mesh after a delay defined by
  * setFullResolutionDelay()
  */
  void switchToLowResolution();

  /**
  * Selects the full resolution mesh.
  * The alarm is unscheduled.
  */
  void switchToFullResolution();
  
  /**
  * Schedules the alarm or reschedules it.
  * This method can be called for instance while moving the
  * camera to maintain the application in low resolution.
  */
  void scheduleFullResolution();

  bool lowResolutionEnabled() const;
  
protected:
  static void restoreToFullResolution(void* scenegraph, SoSensor*);
  
  // method which is called when the full resolution has been reactivated.
  // overriding it allows to update the text showing the % of resolution used.
  virtual void fullResolutionIsBack() {}

  SoAlarmSensor* m_sensor;
  int m_delayToFullResolution;
  bool m_fullResolutionSchedulingEnabled;
};


#endif

