#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoAnnoText3Property.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoGradientBackground.h> 
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/details/SoCubeDetail.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMarkerSet.h>
#include <Inventor/events/SoMouseButtonEvent.h>

#include <Inventor/STL/algorithm>

#include <cmath> 
#include "OivSceneGraph.h" 

#include "wrappers/IndexedPillarMesh.h"
#include "wrappers/SubSampledMesh.h"
#include "wrappers/CellFilter.h"

#include <DialogViz/dialog/SoDialogCheckBox.h>
#include <DialogViz/dialog/SoDialogIntegerSlider.h>
#include <Inventor/SbViewportRegion.h>

#include <Inventor/STL/iomanip>
#include <Inventor/STL/sstream>

#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/SoPickedPoint.h>

#include <Inventor/elements/SoInteractionElement.h>

#include <MeshVizXLM/extrmesh/MeXSurfaceMeshUnstructured.h>

double OivSceneGraph::LowResCallback::s_lowResolutionExtrationTimeReference = 0.1;

//------------------------------------------------------------------------
// Function called each time the mouse button is pressed in the viewer window
void mousePressedCallback(void *_this, SoEventCallback *eventCB) 
{
  ((OivSceneGraph*)_this)->mousePressed(eventCB);
}

OivSceneGraph::OivSceneGraph() :
m_fullResMesh(NULL),
m_lowResMesh(NULL),
m_lowResCB(*this),
m_timer(*this),
m_asyncProcessing(SoPreferences::getBool("MESHVIZ_ASYNC_PROCESSING", false))
{
  m_FPSTimersensor = new SoTimerSensor(&OivSceneGraph::FPSDisplay, this);
  m_FPSTimersensor->setInterval(1.0);
  m_FPSTimersensor->schedule();

  m_ROIDataRangeFilter = new CellFilter;
  m_lowResROIDataRangeFilter = new CellFilter;
  createSceneGraph();
  assembleSceneGraph();
}


OivSceneGraph::~OivSceneGraph()
{
  m_root->unref();
  delete m_lowResMesh;
  delete m_ROIDataRangeFilter;
  delete m_lowResROIDataRangeFilter;
  delete m_planeSensor;

  m_FPSVector.clear();
  delete m_FPSTimersensor;
}

void 
OivSceneGraph::updateMeshBoundingBox(PillarMesh* mesh)
{
  m_meshBoundingBoxPos->translation.setValue((SbVec3f)mesh->getCenter());
  MbVec3f meshsize = mesh->getBBSize();
  m_meshBoundingBox->width = meshsize[0];
  m_meshBoundingBox->height = meshsize[1];
  m_meshBoundingBox->depth = meshsize[2];
}


void 
OivSceneGraph::setFullResMesh(PillarMesh* mesh)
{
  if (m_fullResSceneGraph != NULL && mesh != NULL)
  {
    MbVec3f center = mesh->getCenter();
    if (m_fullResMesh == NULL || center[0] != m_meshCenter[0] || center[1] != m_meshCenter[1] || center[2] != m_meshCenter[2])
    {
      m_meshCenter.setValue(center[0],center[1],center[2]);
      m_transform->center.setValue(m_meshCenter);
      m_planeManip->draggerPosition.setValue(m_meshCenter);
      m_planeManip->plane.setValue(SbPlane(SbVec3f(1,0,0),m_meshCenter));
    }
    // useless when those 2 planes will be connected (eBug #4628)
    m_fullResSceneGraph->m_planeSlice->plane = m_planeManip->plane;
    m_fullResMesh = mesh;

    m_meshDim.setValue((short)m_fullResMesh->getDimI(),(short)m_fullResMesh->getDimJ(),(short)m_fullResMesh->getDimK());

    m_fullResSceneGraph->m_moMesh->setMesh(mesh);

    m_fullResSceneGraph->m_moDepthScalarSet->setScalarSet(&m_fullResMesh->getDepth());
    m_depthColorMapping->minValue = (float)m_fullResMesh->getDepth().getMin();
    m_depthColorMapping->maxValue = (float)m_fullResMesh->getDepth().getMax();
    const PillarPorosity& porosity = m_fullResMesh->getPorosity();
    m_fullResSceneGraph->m_moPorosityScalarSet->setScalarSet(&porosity);
    m_fullResSceneGraph->m_skin->colorScalarSetId = porosity.getSize() > 0 ? 1 : 0;
    m_colorMappingSwitch->whichChild = porosity.getSize() > 0 ? 1 : 0;
    m_porosityColorMapping->minValue = (float)porosity.getMin();
    m_porosityColorMapping->maxValue = (float)porosity.getMax();
    m_weightedAverageMapping.setMinMaxValues(m_depthColorMapping->minValue.getValue(), m_depthColorMapping->maxValue.getValue(),
      m_porosityColorMapping->minValue.getValue(), m_porosityColorMapping->maxValue.getValue());

    m_ROIDataRangeFilter->setMesh(m_fullResMesh);

    SoJackDragger* dragger = (SoJackDragger *) m_planeManip->getDragger(); 
    SbVec3f bbSize(m_fullResMesh->getBBSize());
    dragger->scaleFactor.setValue(bbSize/15);

    const MiTopologyIjk& topology = m_fullResMesh->getTopology();
    std::string unit("");
    int numActiveCells = (int) m_fullResMesh->getNumActiveCells();
    if (numActiveCells > 1e6)
    {
      unit = " million";
      numActiveCells /= (size_t)1e6;
    }
    m_gridInfoDisplay->string.set1Value(0,SbString("I/J/K: ") + int(topology.getNumCellsI()) + " x " + int(topology.getNumCellsJ()) + " x " + int(topology.getNumCellsK()));
    m_gridInfoDisplay->string.set1Value(1,SbString("#Active cells: ") +  numActiveCells + unit.c_str());
    m_gridInfoDisplay->string.set1Value(2,"Resolution ");

    m_gridFPSDisplay->string.set1Value(3, "FPS : ");

    updateMeshBoundingBox(mesh);
    initFence();

    // reset timer;
    m_timer.reset();
  }
}

void 
OivSceneGraph::setMesh(IndexedPillarMesh* mesh)
{
  setFullResMesh(mesh);
  setLowResMesh(mesh);
}

void 
OivSceneGraph::setMesh(PillarMesh* mesh)
{
  setFullResMesh(mesh);
  setLowResMesh(NULL);
}

//---------------------------------------------------------------------------
void 
OivSceneGraph::setLowResMesh(IndexedPillarMesh* mesh)
{
  bool lowResActive = m_lowResSceneGraph != NULL && mesh != NULL && ACTIVE_LOWRESOLUTION && !m_asyncProcessing;
  delete m_lowResMesh;
  m_lowResMesh = lowResActive ? new SubSampledMesh(*mesh) : NULL;
  m_lowResSceneGraph->m_moMesh->setMesh(m_lowResMesh);
  m_lowResROIDataRangeFilter->setMesh(m_lowResMesh);
  m_lowResSceneGraph->m_moDepthScalarSet->setScalarSet(lowResActive ? &m_lowResMesh->getDepth() : NULL);
  m_lowResSceneGraph->m_moPorosityScalarSet->setScalarSet(lowResActive ? &m_lowResMesh->getPorosity() : NULL);
}

void 
OivSceneGraph::autoLowResolutionAdjust(bool enable, SoSFInt32& resolution)
{
  m_switchMesh->enableFullResolutionScheduling(enable && m_lowResMesh != NULL);
  m_lowResCB.evalLowResolution(enable);
  m_timer.evalLowResolution(enable);
  if (enable)
    m_timer.apply();
  resolution = m_lowResMesh ? (int)(100 / (double) m_lowResMesh->getStep()) : 100;
}

void 
OivSceneGraph::adjustLowResolution(double resolution)
{
  if(m_lowResMesh != NULL)
  {
    if(resolution <= 0.5)
    {
      m_lowResMesh->adjust(resolution);
      m_lowResROIDataRangeFilter->setMesh(m_lowResMesh);

      //Group I
      for (int i = 0; i<m_lowResSceneGraph->m_slabGroup[0]->getNumChildren(); ++i)
      {
        ((MoMeshSlab*)m_lowResSceneGraph->m_slabGroup[0]->getChild(i))->index = ((unsigned int) (((MoMeshSlab*)m_fullResSceneGraph->m_slabGroup[0]->getChild(i))->index.getValue() / m_lowResMesh->getStep()));
        ((MoMeshSlab*)m_lowResSceneGraph->m_slabGroup[0]->getChild(i))->thickness = (unsigned int) std::ceil(((MoMeshSlab*)m_fullResSceneGraph->m_slabGroup[0]->getChild(i))->thickness.getValue() / (float) m_lowResMesh->getStep());
      }
      //Group J
      for (int i = 0; i<m_lowResSceneGraph->m_slabGroup[1]->getNumChildren(); ++i)
      {
        ((MoMeshSlab*)m_lowResSceneGraph->m_slabGroup[1]->getChild(i))->index = ((unsigned int) (((MoMeshSlab*)m_fullResSceneGraph->m_slabGroup[1]->getChild(i))->index.getValue() / m_lowResMesh->getStep()));
        ((MoMeshSlab*)m_lowResSceneGraph->m_slabGroup[1]->getChild(i))->thickness = (unsigned int) std::ceil(((MoMeshSlab*)m_fullResSceneGraph->m_slabGroup[1]->getChild(i))->thickness.getValue() / (float) m_lowResMesh->getStep());
      }
      //Group K
      for (int i = 0; i<m_lowResSceneGraph->m_slabGroup[2]->getNumChildren(); ++i)
      {
        ((MoMeshSlab*)m_lowResSceneGraph->m_slabGroup[2]->getChild(i))->index = ((unsigned int) (((MoMeshSlab*)m_fullResSceneGraph->m_slabGroup[2]->getChild(i))->index.getValue() / m_lowResMesh->getStep()));
        ((MoMeshSlab*)m_lowResSceneGraph->m_slabGroup[2]->getChild(i))->thickness = (unsigned int) std::ceil(((MoMeshSlab*)m_fullResSceneGraph->m_slabGroup[2]->getChild(i))->thickness.getValue() / (float) m_lowResMesh->getStep());
      }
      m_lowResSceneGraph->m_moMesh->touch();
    }
    else
      m_lowResMesh->adjust(1.0);
  }
}

void
OivSceneGraph::changeResolutionInfoDisplay(size_t resolutionStep)
{
  std::ostringstream res;
  res << "Resolution " << std::right << std::setw(4) << (int)(100 / (double) resolutionStep) << "%";

  if (m_gridInfoDisplay->string.getNum() > 2)
  {
    if (m_gridInfoDisplay->string[2] != res.str())
      // check if the displayed resolution must be updated
        m_gridInfoDisplay->string.set1Value(2,res.str());
  }
}

void
OivSceneGraph::setFrame(int frame) 
{ 
  m_fullResMesh->setFrame(frame);
  m_ROIDataRangeFilter->touch();
  m_lowResROIDataRangeFilter->touch();
  m_fullResSceneGraph->m_moCellFilter->touch();
  m_lowResSceneGraph->m_moCellFilter->touch();
  m_fullResSceneGraph->m_moPorosityScalarSet->touch();
  m_fullResSceneGraph->m_moDepthScalarSet->touch();
  m_lowResSceneGraph->m_moPorosityScalarSet->touch();
  m_lowResSceneGraph->m_moDepthScalarSet->touch();
}

void
OivSceneGraph::setSkinTransparency(float value)
{
	m_skinMaterial->transparency.setValue(value);
}

void
OivSceneGraph::setHullTransparency(float value)
{
	m_hullMaterial->transparency.setValue(value);
}

void
OivSceneGraph::enableROIFiltering(bool enable)
{
  m_fullResSceneGraph->m_switchROIDataRangeFilter->whichChild = (enable == true)
    ? SO_SWITCH_ALL 
    : SO_SWITCH_NONE;
  if (enable && m_ROIDataRangeFilter->isActive())
  {
    m_fullResSceneGraph->m_moCellFilter->setCellFilter(m_ROIDataRangeFilter);
    m_lowResSceneGraph->m_moCellFilter->setCellFilter(m_lowResROIDataRangeFilter);
  }
  else
  {
    m_fullResSceneGraph->m_moCellFilter->reset();
    m_lowResSceneGraph->m_moCellFilter->reset();
  }
}

bool
OivSceneGraph::setROIFiltering(size_t imin,size_t imax,size_t jmin,size_t jmax,size_t kmin,size_t kmax)
{
  m_ROIDataRangeFilter->setROI(imin,imax,jmin,jmax,kmin,kmax);
  m_lowResROIDataRangeFilter->setROI(imin,imax,jmin,jmax,kmin,kmax);
  if (m_ROIDataRangeFilter->isActive())
  {
    m_fullResSceneGraph->m_moCellFilter->setCellFilter(m_ROIDataRangeFilter);
    m_lowResSceneGraph->m_moCellFilter->setCellFilter(m_lowResROIDataRangeFilter);
  }
  else
  {
    m_fullResSceneGraph->m_moCellFilter->reset();
    m_lowResSceneGraph->m_moCellFilter->reset();
  }

  return (m_fullResSceneGraph->m_switchROIDataRangeFilter->whichChild.getValue() == SO_SWITCH_ALL);
}


bool
OivSceneGraph::setDataRangeFiltering(double min, double max, const std::string& propertyName)
{
  if (propertyName != "")
  {
    m_ROIDataRangeFilter->setDataSet(m_fullResMesh->getProperty(propertyName));
    m_lowResROIDataRangeFilter->setDataSet(m_lowResMesh ? m_lowResMesh->getProperty(propertyName) : NULL);
  }

  m_ROIDataRangeFilter->setDataRange(min,max);
  m_lowResROIDataRangeFilter->setDataRange(min,max);

  if (m_ROIDataRangeFilter->isActive())
  {
    m_fullResSceneGraph->m_moCellFilter->setCellFilter(m_ROIDataRangeFilter);
    m_lowResSceneGraph->m_moCellFilter->setCellFilter(m_lowResROIDataRangeFilter);
  }
  else
  {
    m_fullResSceneGraph->m_moCellFilter->reset();
    m_lowResSceneGraph->m_moCellFilter->reset();
  }

  return m_fullResSceneGraph->m_switchROIDataRangeFilter->whichChild.getValue() == SO_SWITCH_ALL;
}

void
OivSceneGraph::updateFilter()
{
  m_ROIDataRangeFilter->touch();
  m_lowResROIDataRangeFilter->touch();
  m_fullResSceneGraph->m_moCellFilter->touch();
  m_lowResSceneGraph->m_moCellFilter->touch();
}


void 
OivSceneGraph::createSceneGraph() 
{
  m_switchMesh = new MyMeshResolutionSwitch(this);
  m_switchMesh->setName("MeshResolutionSwitch");

  // Define data mapping
  m_colorMappingSwitch = new SoSwitch;
  m_depthColorMapping = new MoPredefinedColorMapping;
  m_depthColorMapping->predefColorMap = MoPredefinedColorMapping::BLUE_WHITE_RED;
  m_porosityColorMapping = new MoPredefinedColorMapping;
  m_porosityColorMapping->predefColorMap = MoPredefinedColorMapping::BLUE_WHITE_RED;
  m_combineColorMapping = new MoCombineColorMapping;
  m_combineColorMapping->setColorMapping(&m_weightedAverageMapping);

  m_hullMaterial = new MoMaterial; 
  m_hullMaterial->faceColoring = MoMaterial::COLOR;

  m_drawStyle = new MoDrawStyle;

  m_skinMaterial = new MoMaterial; 
  m_skinMaterial->lineColoring = MoMaterial::COLOR;
  m_skinMaterial->lineColor.setValue(0,0,0);
  m_skinMaterial->pointColoring = MoMaterial::COLOR;
  m_skinMaterial->pointColor.setValue(1,0,0);

  m_transform = new SoTransform;
  m_transform->scaleFactor = SbVec3f(1,1,1);

  m_planeManip = new SoClipPlaneManip;
  m_planeManip->on.setValue(FALSE);
  m_clipPlane = new SoClipPlane;
  m_clipPlane->on.setValue(FALSE);
  m_clipPlane->plane.connectFrom(&m_planeManip->plane);
  
  m_lowResSceneGraph = new MeshSceneGraph(m_skinMaterial, m_hullMaterial, !m_asyncProcessing ? &m_lowResCB : NULL);
  m_lowResSceneGraph->setName("Low_Resolution");
  m_fullResSceneGraph = new MeshSceneGraph(m_skinMaterial, m_hullMaterial, !m_asyncProcessing ? &m_timer : NULL);
  m_fullResSceneGraph->setName("Full_Resolution");

  // Workaround eBug #4628: Due to OIV Draggers.
  // When moving a dragger, all fields representing its position are updated with a margin of error, 
  // thus losing information of its motion (translation, rotation ...).
  // To avoid an overload of computation, parameters are then checked within a threshold before being updated the slice plane
  // When the bug is fixed, simply connect the two planes
  //m_fullResSceneGraph->m_planeSlice->plane.connectFrom(&m_planeManip->plane);
  m_planeSensor = new SoFieldSensor(updateGeomSlice,this);
  m_planeSensor->attach(&m_planeManip->plane);
  
  m_lowResSceneGraph->connectFrom(*m_fullResSceneGraph);

  m_switchDragger = new SoSwitch;
  m_switchDragger->whichChild.connectFrom(&m_fullResSceneGraph->m_switchPlaneSlice->whichChild);

  setGeomSlice(2);

  m_gridInfoDisplay = new SoText2;
  m_gridInfoDisplay->string.setNum(3);

  m_gridFPSDisplay = new SoText2;
  // set callback on dragger for down sampling
  m_planeManip->getDragger()->addMotionCallback(OivSceneGraph::switchToLowResolutionMesh,this);
  
  m_switchMeshBoundingBox = new SoSwitch;
  m_meshBoundingBox = new SoCube;
  m_meshBoundingBoxPos = new SoTranslation;
  m_meshBoundingBoxSep = new SoSeparator;
  m_switchMeshBoundingBox = new SoSwitch;

  m_meshBoundingMaterial = new SoMaterial;
  m_meshBoundingMaterial->transparency = 0.9f;
  m_meshBoundingMaterial->diffuseColor = SbColor(0,1,1);



}

int
OivSceneGraph::getGapBetweenSlabs(int sliceId, int numSlice)
{
  int gap;
  if (numSlice == 1)
    gap = 0;
  else
    gap = ((m_meshDim[sliceId])-1)/(numSlice-1); 

  return gap; 
}

void
OivSceneGraph::updateSlabs(int sliceId, int step, int currentIndex, int currentThickness)
{
  int numSlice = ((getMeshDim()[sliceId]) /step)+1;
  int stepLow = m_lowResMesh != NULL ? int(float(step)/float(m_lowResMesh->getStep())) : 2;
  if ( stepLow > 1)
  {
    m_lowResSceneGraph->updateSlabs(sliceId, stepLow, numSlice, currentIndex, currentThickness);
  }
  else
  {
    m_lowResSceneGraph->updateSlabs(sliceId, stepLow + 1, numSlice, currentIndex, currentThickness);
    for (int numSlab = 0 ; numSlab < m_lowResSceneGraph->m_slabGroup[sliceId]->getNumChildren(); ++numSlab)
    {
      ((MoMeshSlab*) m_lowResSceneGraph->m_slabGroup[sliceId]->getChild(numSlab))->index = (unsigned int)  ((currentIndex + numSlab * step) / m_lowResMesh->getStep());
    }
  }
  m_fullResSceneGraph->updateSlabs(sliceId, step, numSlice, currentIndex, currentThickness);
}

OivSceneGraph::MeshSceneGraph::MeshSceneGraph(MoMaterial* skinMat, MoMaterial* hullMat, MiExtractorCallback* extractorCB)
{
  m_moMesh = new MoMesh;
  m_moMesh->setName("Mesh");
  m_moDepthScalarSet = new MoScalarSet;
  m_moPorosityScalarSet = new MoScalarSet;

  m_switchROIDataRangeFilter  = new SoSwitch;
  m_moCellFilter = new MoCellFilter;
  m_switchROIDataRangeFilter->addChild(m_moCellFilter);

  m_switchSkin  = new SoSwitch;
  m_switchSkin->setName("Skin");

  m_switchHullOutline = new SoSwitch;
  m_switchHullOutline->setName("Hull_Outline");

  m_switchHull  = new SoSwitch;
  m_switchHull->setName("Hull");

  // Create the solid contour visualization node.
  m_skin = new MoMeshSkin;
  m_skin->setExtractorCallback(extractorCB);
  m_skin->colorScalarSetId = 0;
  m_skin->parallel = MULTITHREADING;

  m_hullOutline = new MoMeshOutline;
  m_hullOutline->colorScalarSetId = -1;
  m_hullOutline->setExtractorCallback(extractorCB);
  m_hullOutline->parallel = MULTITHREADING;

  m_hullSkin = new MoMeshSkin;
  m_hullSkin->setExtractorCallback(extractorCB);
  m_hullSkin->colorScalarSetId = -1;
  m_hullSkin->parallel = MULTITHREADING;

  m_switchPlaneSlice  = new SoSwitch;
  m_switchPlaneSlice->setName("PlaneSlice");
  m_planeSlice = new MoMeshPlaneSlice;
  m_planeSlice->colorScalarSetId.connectFrom(&m_skin->colorScalarSetId);
  m_planeSlice->setExtractorCallback(extractorCB);
  m_planeSlice->parallel = MULTITHREADING;

  m_switchFenceSlice  = new SoSwitch;
  m_switchFenceSlice->setName("FenceSlice");
  m_fenceSlice = new MoMeshFenceSlice;
  m_fenceSlice->colorScalarSetId.connectFrom(&m_skin->colorScalarSetId);
  m_fenceSlice->setExtractorCallback(extractorCB);
  m_fenceSlice->parallel = MULTITHREADING;

  for (int i=0; i<NUM_LOGICALSLICE; ++i)
  {
    m_switchLogicalSlice[i] = new SoSwitch;
    m_switchLogicalSlice[i]->setName("Slab");
    m_slabGroup[i] = new SoGroup;
    MoMeshSlab *slab = new MoMeshSlab;
    slab->parallel = MULTITHREADING;
    m_slabGroup[i]->addChild(slab);
    ((MoMeshSlab*)m_slabGroup[i]->getChild(0))->dimension =  (MiMesh::Dimension) i;
    ((MoMeshSlab*)m_slabGroup[i]->getChild(0))->index = 0;
    ((MoMeshSlab*)m_slabGroup[i]->getChild(0))->colorScalarSetId.connectFrom(&m_skin->colorScalarSetId);
    ((MoMeshSlab*)m_slabGroup[i]->getChild(0))->setExtractorCallback(extractorCB);
  }

  m_switchIsosurface = new SoSwitch();
  m_switchIsosurface->setName("Isosurface");

  m_isosurface = new MoMeshIsosurface;
  m_isosurface->colorScalarSetId.connectFrom(&m_skin->colorScalarSetId);
  m_isosurface->isoScalarSetId = 0;
  m_isosurface->setExtractorCallback(extractorCB);
  m_isosurface->parallel = MULTITHREADING;

  assembleSceneGraph(skinMat, hullMat);
}

void
OivSceneGraph::assembleSceneGraph()
{
  m_root = new SoSeparator;
  m_root->ref();
  SoSeparator* mainscene = new SoSeparator;
  m_root->addChild(mainscene);

  {
    // Assemble the scene graph
    SoPerspectiveCamera *camera3D = new SoPerspectiveCamera ;
    mainscene->addChild(camera3D);
    mainscene->addChild(new SoGradientBackground);
    SoShapeHints* shapeHints = new SoShapeHints;
    shapeHints->vertexOrdering.setValue(SoShapeHints::COUNTERCLOCKWISE );
    mainscene->addChild(shapeHints);
    mainscene->addChild(m_transform );

    // due to bug #28549 can not use copy constructor with m_skinMaterial
    MoMaterial* material = new MoMaterial;
    material->faceColoring.connectFrom(&m_skinMaterial->faceColoring);
    material->faceColor.connectFrom(&m_skinMaterial->faceColor);
    material->lineColoring.connectFrom(&m_skinMaterial->lineColoring);
    material->lineColor.connectFrom(&m_skinMaterial->lineColor);
    material->pointColoring.connectFrom(&m_skinMaterial->pointColoring);
    material->pointColor.connectFrom(&m_skinMaterial->pointColor);
    ////
    mainscene->addChild(material);
    mainscene->addChild(m_drawStyle);
    mainscene->addChild(m_colorMappingSwitch);
    {
      m_colorMappingSwitch->addChild(m_depthColorMapping);
      m_colorMappingSwitch->addChild(m_porosityColorMapping);
      m_colorMappingSwitch->addChild(m_combineColorMapping);
    }
    MoDataBinding* dataBinding = new MoDataBinding;
    dataBinding->dataBinding = MoDataBinding::PER_CELL;
    mainscene->addChild(dataBinding);
    m_switchMesh->switchToFullResolution();
    mainscene->addChild(m_switchMesh);
    {
      m_fullResSceneGraph->m_switchPlaneSlice->addChild(m_clipPlane);
      m_lowResSceneGraph->m_switchPlaneSlice->addChild(m_clipPlane);
      m_switchMesh->setFullResolutionSceneGraph(m_fullResSceneGraph);
      m_switchMesh->setLowResolutionSceneGraph(m_lowResSceneGraph);
    }
    mainscene->addChild(m_switchDragger);
    {
      m_switchDragger->addChild(m_planeManip);
    }

    mainscene->addChild(m_switchMeshBoundingBox);
    {
      m_switchMeshBoundingBox->addChild(m_meshBoundingBoxSep);
      {
        SoVertexProperty* vertexProp = new SoVertexProperty;
        vertexProp->vertex.connectFrom(&m_fullResSceneGraph->m_fenceSlice->polyline);
        SoDrawStyle* polylineDS = new SoDrawStyle;
        polylineDS->lineWidth = 2;

        SoLineSet* polyline = new SoLineSet;
        polyline->vertexProperty.setValue(vertexProp);

        SoMarkerSet* pointSet = new SoMarkerSet;
        pointSet->vertexProperty.setValue(vertexProp);
        pointSet->markerIndex = SoMarkerSet::CIRCLE_FILLED_7_7;

        SoMaterial* polylineMaterial = new SoMaterial;
        polylineMaterial->diffuseColor = SbColor(1,0,0);

        m_meshBoundingBoxSep->addChild(polylineDS);
        m_meshBoundingBoxSep->addChild(polylineMaterial);
        m_meshBoundingBoxSep->addChild(polyline);
        m_meshBoundingBoxSep->addChild(pointSet); 

        m_meshBoundingBoxSep->addChild(m_meshBoundingMaterial);
        m_meshBoundingBoxSep->addChild(m_meshBoundingBoxPos);
        m_meshBoundingBoxSep->addChild(m_meshBoundingBox);

        SoDrawStyle* boundingBoxDrawStyle = new SoDrawStyle;
        boundingBoxDrawStyle->style = SoDrawStyle::LINES;
        boundingBoxDrawStyle->lineWidth = 5;
        m_meshBoundingBoxSep->addChild(boundingBoxDrawStyle);
        m_meshBoundingBoxSep->addChild(m_meshBoundingBox);

      }
    }

    SoEventCallback* m_mousePressedCallbackNode = new SoEventCallback;
    m_mousePressedCallbackNode->addEventCallback(SoMouseButtonEvent::getClassTypeId(),mousePressedCallback, this); 
    mainscene->addChild(m_mousePressedCallbackNode);

  }

  SoAnnotation* scene2D = new SoAnnotation;
  scene2D->boundingBoxIgnoring = true;
  m_root->addChild(scene2D);
  {
    SoOrthographicCamera *camera2D = new SoOrthographicCamera ;
    camera2D->viewportMapping = SoCamera::LEAVE_ALONE;
    scene2D->addChild(camera2D);

    SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
    annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;
    annoText3Property->fontSizeHint = SoAnnoText3Property::ANNOTATION ;
    scene2D->addChild(annoText3Property);

    SoPickStyle *pick_style = new SoPickStyle;
    pick_style->style = SoPickStyle::UNPICKABLE;
    scene2D->addChild(pick_style);

    SoFont *font = new SoFont;
    font->size = 15;
    scene2D->addChild(font);
    SoTranslation *trans = new SoTranslation;
    trans->translation.setValue(-0.95F, 0.90F, -0.1F);
    scene2D->addChild(trans);
    m_gridInfoDisplay->justification =  SoText2::LEFT;
    scene2D->addChild(m_gridInfoDisplay);
    m_gridFPSDisplay->justification =  SoText2::LEFT;
    scene2D->addChild(m_gridFPSDisplay);
  }

  // set a callback node to reschedule the full resolution when interacting
  SoCallback* cb = new SoCallback;
  cb->setCallback(OivSceneGraph::scheduleFullResolutionMesh,this);
  m_root->addChild(cb);

  // set a callback node to count triangles in scenegraph
  SoCallback* triangleCountCb = new SoCallback;
  triangleCountCb->setCallback(OivSceneGraph::getNumTriangles, this);
  m_root->addChild(triangleCountCb);
}


void 
OivSceneGraph::MeshSceneGraph::updateSlabs(int sliceId, int step, int numSlice, int /*currentIndex*/, int currentThickness)
{
  int numChildren = m_slabGroup[sliceId]->getNumChildren(); 
  for (int j = 1; j<numChildren; ++j)
  { 
    m_slabGroup[sliceId]->removeChild(1);
  }
  for (int i=1; i<numSlice; ++i)
  {
    MoMeshSlab *slab = new MoMeshSlab; 
    m_slabGroup[sliceId]->addChild(slab);
    slab->dimension =  (MiMesh::Dimension) sliceId;
    slab->thickness = currentThickness;
    slab->colorScalarSetId.connectFrom(&m_skin->colorScalarSetId);
    slab->index = ((MoMeshSlab*) m_slabGroup[sliceId]->getChild(0))->index.getValue() + (i * step);
  }
}


void
OivSceneGraph::MeshSceneGraph::assembleSceneGraph(MoMaterial* skinMaterial, MoMaterial* hullMaterial)
{
  this->addChild(m_moMesh);

  m_switchHullOutline->setName("Mesh_Hull_Outline");
  this->addChild( m_switchHullOutline );
  {
    SoSeparator* app = new SoSeparator;
    m_switchHullOutline->addChild( app );
    MoDrawStyle* oultineDrawStyle = new MoDrawStyle;
    oultineDrawStyle->displayEdges = true;
    app->addChild(oultineDrawStyle);
    app->addChild(m_hullOutline);
  }

  m_switchHull->setName("Mesh_Hull");
  this->addChild( m_switchHull );
  {
    SoSeparator* app = new SoSeparator;
    SoPickStyle* pickStyle = new SoPickStyle;
    MoDrawStyle* hullDrawStyle = new MoDrawStyle;
    hullDrawStyle->displayEdges = FALSE;
    hullDrawStyle->displayPoints = FALSE;
    pickStyle->style.setValue(SoPickStyle::UNPICKABLE);
    m_switchHull->addChild( app );
    app->addChild(hullMaterial);
    app->addChild(pickStyle);
    app->addChild(hullDrawStyle);
    app->addChild(m_hullSkin);
  }
  
  this->addChild(m_moDepthScalarSet);
  this->addChild(m_moPorosityScalarSet);

  this->addChild(m_switchROIDataRangeFilter);

  this->addChild( m_switchPlaneSlice );
  {
    m_switchPlaneSlice->addChild( m_planeSlice );
  }

  this->addChild( m_switchFenceSlice );
  {
    m_switchFenceSlice->addChild( m_fenceSlice );
  }

  SoSeparator* sep = new SoSeparator;
  sep->setName("Slabs_Group");
  this->addChild(sep);
  {
    for (int i=0; i<NUM_LOGICALSLICE; ++i)
    {
      sep->addChild(m_switchLogicalSlice[i]);
      {
        m_switchLogicalSlice[i]->addChild( m_slabGroup[i] );
      }
    }
  }

  m_switchSkin->setName("Mesh_Skin");
  this->addChild( m_switchSkin );
  {
    SoSeparator* app = new SoSeparator;
    m_switchSkin->addChild( app );
    app->addChild(skinMaterial);
    app->addChild( m_skin );
  }


  this->addChild( m_switchIsosurface );
  {
    SoSeparator* app = new SoSeparator;
    m_switchIsosurface->addChild( app );
    app->addChild(m_isosurface);
  }
}

void
OivSceneGraph::MeshSceneGraph::connectFrom(const MeshSceneGraph& sg)
{
  m_skin->colorScalarSetId.connectFrom(&sg.m_skin->colorScalarSetId);
  m_planeSlice->plane.connectFrom(&sg.m_planeSlice->plane);
  m_isosurface->isovalue.connectFrom(&sg.m_isosurface->isovalue);
  m_isosurface->isoScalarSetId.connectFrom(&sg.m_isosurface->isoScalarSetId);
  m_fenceSlice->polyline.connectFrom(&sg.m_fenceSlice->polyline);
  m_fenceSlice->direction.connectFrom(&sg.m_fenceSlice->direction);

  m_switchSkin->whichChild.connectFrom(&sg.m_switchSkin->whichChild);
  m_switchHullOutline->whichChild.connectFrom(&sg.m_switchHullOutline->whichChild);
  m_switchHull->whichChild.connectFrom(&sg.m_switchHull->whichChild);
  m_switchPlaneSlice->whichChild.connectFrom(&sg.m_switchPlaneSlice->whichChild);
  m_switchFenceSlice->whichChild.connectFrom(&sg.m_switchFenceSlice->whichChild);

  for (int i=0; i<NUM_LOGICALSLICE; ++i)
  {
    m_switchLogicalSlice[i]->whichChild.connectFrom(&sg.m_switchLogicalSlice[i]->whichChild);
  }

  m_switchROIDataRangeFilter->whichChild.connectFrom(&sg.m_switchROIDataRangeFilter->whichChild);
  m_switchIsosurface->whichChild.connectFrom(&sg.m_switchIsosurface->whichChild);
}

size_t
OivSceneGraph::MeshSceneGraph::getNumTriangles() const
{
  size_t numTriangles = 0;
  MeshType meshType;
  // Skin, Hull and Slab are only made of quadrangles, each containing 2 triangles.
  // The number of triangles is 2 times the number of cells
  const MeXSurfaceMeshUnstructured* surface = (dynamic_cast<const MeXSurfaceMeshUnstructured*>(m_skin->getExtractedMesh(meshType)));
  if (surface != NULL && m_switchSkin->whichChild.getValue() == SO_SWITCH_ALL)
    numTriangles += 2 * surface->getTopology().getNumCells();
  surface = (dynamic_cast<const MeXSurfaceMeshUnstructured*>(m_hullSkin->getExtractedMesh(meshType)));
  if (surface != NULL && m_switchHull->whichChild.getValue() == SO_SWITCH_ALL)
    numTriangles += 2 * surface->getTopology().getNumCells();
  for (int i = 0; i < NUM_LOGICALSLICE; ++i)
    if (m_switchLogicalSlice[i]->whichChild.getValue() == SO_SWITCH_ALL)
    {
      int numChildren = m_slabGroup[i]->getNumChildren();
      for (int j = 0; j < numChildren; ++j)
      {
        surface = (dynamic_cast<const MeXSurfaceMeshUnstructured*>(((MoMeshSlab*)m_slabGroup[i]->getChild(j))->getExtractedMesh(meshType)));
        if (surface != NULL)
          numTriangles += 2 * surface->getTopology().getNumCells();
      }
    }

  // Isosurface, plane slice and fence slice are made of polygons.
  // Each polygon contains (number of nodes - 2) triangles
  surface = (dynamic_cast<const MeXSurfaceMeshUnstructured*>(m_isosurface->getExtractedMesh(meshType)));
  if (surface != NULL && m_switchIsosurface->whichChild.getValue() == SO_SWITCH_ALL)
    for (size_t i = 0; i < surface->getTopology().getNumCells(); ++i)
      numTriangles += (surface->getTopology().getCell(i)->getNumNodes() - 2);

  surface = (dynamic_cast<const MeXSurfaceMeshUnstructured*>(m_planeSlice->getExtractedMesh(meshType)));
  if (surface != NULL && m_switchPlaneSlice->whichChild.getValue() == SO_SWITCH_ALL)
    for (size_t i = 0; i < surface->getTopology().getNumCells(); ++i)
      numTriangles += (surface->getTopology().getCell(i)->getNumNodes() - 2);

  surface = (dynamic_cast<const MeXSurfaceMeshUnstructured*>(m_fenceSlice->getExtractedMesh(meshType)));
  if (surface != NULL && m_switchFenceSlice->whichChild.getValue() == SO_SWITCH_ALL)
    for (size_t i = 0; i < surface->getTopology().getNumCells(); ++i)
      numTriangles += (surface->getTopology().getCell(i)->getNumNodes() - 2);

  return numTriangles;
}

/*---------------------------------------------------------------------------*/
bool
OivSceneGraph::setIsosurfaceValue(float value)
{
	m_fullResSceneGraph->m_isosurface->isovalue.setValue(value);
  return m_fullResSceneGraph->m_switchIsosurface->whichChild.getValue() == SO_SWITCH_ALL;
}

//---------------------------------------------------------------------------
void 
OivSceneGraph::updateGeomSlice(void* scenegraph, SoSensor* /*sensor*/)
{
  OivSceneGraph* sg = (OivSceneGraph*)scenegraph;
  sg->updateGeomSlice();
}

//---------------------------------------------------------------------------
void 
OivSceneGraph::updateGeomSlice()
{
  SbPlane currentPlane = m_fullResSceneGraph->m_planeSlice->plane.getValue();
  SbVec3f currentNormal = currentPlane.getNormal();
  SbPlane newPlane = m_planeManip->plane.getValue();
  float newDistance = newPlane.getDistanceFromOrigin();
  if (!areCollinear(MbVec3d(newPlane.getNormal()),MbVec3d(currentNormal),1e-6))
    m_fullResSceneGraph->m_planeSlice->plane.setValue(newPlane);
  else if (!SbMathHelper::isCoincRelativeOrAbsolute(currentPlane.getDistanceFromOrigin(),newDistance,(float)1e-4,(float)1e-8))
    m_fullResSceneGraph->m_planeSlice->plane.setValue(SbPlane(currentNormal,newDistance));
}


/*---------------------------------------------------------------------------*/
void
OivSceneGraph::setGeomSlice(int axis)
{
  m_sliceNormal = m_rotAxis = SbVec3f(0,0,0);
  m_sliceNormal[axis] = 1;
  m_rotAxis[(axis+1)%3] = 1;
  m_planeManip->plane = SbPlane(m_sliceNormal,m_meshCenter);
  m_planeManip->draggerPosition = m_meshCenter;
}

/*---------------------------------------------------------------------------*/
bool
OivSceneGraph::rotateGeomSlice(float angle)
{
  angle *= (float)(M_PI/180.);
  SbVec3f sliceNormal = m_sliceNormal;
  SbRotation(m_rotAxis,angle).multVec(sliceNormal,sliceNormal);
  m_planeManip->plane = SbPlane(sliceNormal,m_meshCenter);
  return m_fullResSceneGraph->m_switchPlaneSlice->whichChild.getValue() == SO_SWITCH_ALL;
}

bool
OivSceneGraph::moveSlab(size_t sliceId, size_t index, int currentStep)
{
  for (int numSlab = 0; numSlab < m_fullResSceneGraph->m_slabGroup[sliceId]->getNumChildren(); numSlab++)
  {
    ((MoMeshSlab*) m_fullResSceneGraph->m_slabGroup[sliceId]->getChild(numSlab))->index = (unsigned int) index + (numSlab * currentStep);
    if (m_lowResMesh)
      ((MoMeshSlab*) m_lowResSceneGraph->m_slabGroup[sliceId]->getChild(numSlab))->index = (unsigned int) ( index / m_lowResMesh->getStep() + (numSlab * currentStep) / m_lowResMesh->getStep());
  }
  return m_fullResSceneGraph->m_switchLogicalSlice[sliceId]->whichChild.getValue() == SO_SWITCH_ALL;
}

bool
OivSceneGraph::changeSlabThickness(size_t sliceId, size_t thickness)
{
  for (int numSlab = 0; numSlab < m_fullResSceneGraph->m_slabGroup[sliceId]->getNumChildren(); numSlab++)
  {
    ((MoMeshSlab*) m_fullResSceneGraph->m_slabGroup[sliceId]->getChild(numSlab))->thickness = (unsigned int) thickness;
    if (m_lowResMesh)
      ((MoMeshSlab*) m_lowResSceneGraph->m_slabGroup[sliceId]->getChild(numSlab))->thickness =  (unsigned int) std::ceil(thickness / (float) m_lowResMesh->getStep());
  }
  return m_fullResSceneGraph->m_switchLogicalSlice[sliceId]->whichChild.getValue() == SO_SWITCH_ALL;
}

void
OivSceneGraph::selectIsoScalar(const std::string& name)
{
  if (name == "Depth")
    m_fullResSceneGraph->m_isosurface->isoScalarSetId = 0;
  else if (name == "Porosity")
    m_fullResSceneGraph->m_isosurface->isoScalarSetId = 1;
}

void
OivSceneGraph::setCombiningWeight(double weight)
{
  m_weightedAverageMapping.setWeight(weight);
  m_combineColorMapping->touch();
}

void 
OivSceneGraph::displayColor()
{
  m_skinMaterial->faceColoring = MoMaterial::COLOR;
}

void 
OivSceneGraph::displayPorosity()
{
  m_skinMaterial->faceColoring = MoMaterial::CONTOURING;
  m_colorMappingSwitch->whichChild = 1;
  m_fullResSceneGraph->m_skin->colorScalarSetId = 1;
}

void 
OivSceneGraph::displayDepth()
{
  m_skinMaterial->faceColoring = MoMaterial::CONTOURING;
  m_colorMappingSwitch->whichChild = 0;
  m_fullResSceneGraph->m_skin->colorScalarSetId = 0;
}

void
OivSceneGraph::displayCombined()
{
  m_skinMaterial->faceColoring = MoMaterial::CONTOURING;
  m_colorMappingSwitch->whichChild = 2;
}

void 
OivSceneGraph::updateColorMap(MoPredefinedColorMapping::PredefColorMapping newMap)
{
  m_depthColorMapping->predefColorMap = newMap;
  m_porosityColorMapping->predefColorMap = newMap; 
  m_weightedAverageMapping.setPredefinedColorMapping(newMap);
}

/*---------------------------------------------------------------------------*/
void 
OivSceneGraph::LowResCallback::beginExtract(const std::string /*name*/, bool geomChanged, bool topoChanged, bool dataSetChanged, size_t)
{
  if ( m_count == 0 && (topoChanged || geomChanged || dataSetChanged))
  {
    m_geomChanged = geomChanged;
    m_topoChanged = topoChanged;
    m_dataSetChanged = dataSetChanged;
  }
  m_startTime.push_back(m_locatTime.getElapsed());
  ++m_count;
}

/*---------------------------------------------------------------------------*/
void 
OivSceneGraph::LowResCallback::endExtract() 
{
  --m_count;
  if (m_count == 0)
  {
    double time = m_locatTime.getElapsed() - m_startTime.back();
    if (LOW_RES_TIMINGS) display_time("low resolution extraction", time);
    if (m_evalLowResolution && (m_topoChanged || m_geomChanged) && time > s_lowResolutionExtrationTimeReference)
      ++m_exceedCounter;
    else
      m_exceedCounter = 0;
    // if the number of consecutive time that extraction exceeds the reference time
    if (m_exceedCounter > 3)
    {
      m_exceedCounter = 0;
      // low resolution is not fast enough, reduce low resolution
      m_scenegraph.adjustLowResolution(1 / (double)(m_scenegraph.m_lowResMesh->getStep() + 1));
    }

    m_geomChanged = false;
    m_topoChanged = false;
  }
  m_startTime.pop_back();
}

/*---------------------------------------------------------------------------*/
void 
OivSceneGraph::FullResCallback::beginExtract(const std::string name, bool geomChanged, bool topoChanged, bool dataSetChanged, size_t numPhases) 
{
  m_name.push_back(name);
  if (topoChanged || geomChanged || dataSetChanged)
    display(std::string(m_count*3,' ') + "start " + name);
  LowResCallback::beginExtract(name,geomChanged,topoChanged,dataSetChanged,numPhases);
}

/*---------------------------------------------------------------------------*/
void 
OivSceneGraph::FullResCallback::endExtract() 
{
  --m_count;
  double time = m_locatTime.getElapsed() - m_startTime.back();
  if(m_topoChanged || m_geomChanged)
    display_time(std::string(m_count*3,' ')  + m_name.back() + " extracted",time);
  else if(m_dataSetChanged)
    display_time(std::string(m_count*3,' ')  + m_name.back() + " dataset extracted",time);
  if (m_count == 0)
  {
    if (m_evalLowResolution && (m_topoChanged || m_geomChanged))
    {
      // max extraction is set to 10% above so that resolution is only reevaluated if a slower 
      // representation is extracted
      if (time > m_maxExtractionTime * 1.1)
      {
        m_maxExtractionTime = time;
        apply();
      }
    }
    m_geomChanged = false;
    m_topoChanged = false;
    m_dataSetChanged = false;
  }
  m_startTime.pop_back();
  m_name.pop_back();
}

/*---------------------------------------------------------------------------*/
void
OivSceneGraph::FullResCallback::apply()
{
  if (m_maxExtractionTime < s_lowResolutionExtrationTimeReference)
  {
    // low resolution not needed, extraction is fast enough
    m_scenegraph.adjustLowResolution(1.0);
  }
  else
  {
    double timeFactor = m_maxExtractionTime / s_lowResolutionExtrationTimeReference;
    // Since extraction is not linear regarding to the size of the grid
    // the step to group cells for low resolution grid is computed based on nlog(n) 
    // so that it increases faster as the size of the grid does. This is lowered by a step of 2.
    double step = ceil(std::pow(timeFactor*log(timeFactor), 1 / 3.0));
    m_scenegraph.adjustLowResolution(1 / step);
  }
}


/*---------------------------------------------------------------------------*/
bool 
OivSceneGraph::FullResCallback::beginPhase(size_t ,  std::string phaseName, size_t)
{
  m_phaseName.push_back(phaseName);
  m_startPhase.push_back(m_locatTime.getElapsed());
  return true;
}

/*---------------------------------------------------------------------------*/
bool 
OivSceneGraph::FullResCallback::endPhase()
{
#if PHASE_TIMINGS
  display_time(std::string(m_count*3,' ') + m_phaseName.back(),m_locatTime.getElapsed() - m_startPhase.back());
#endif
  m_phaseName.pop_back();
  m_startPhase.pop_back();
  return true; 
}

//------------------------------------------------------------------------------------

void
OivSceneGraph::getNumTriangles(void * userData, SoAction* /*action*/)
{
  OivSceneGraph* currentSceneGraph = (OivSceneGraph*)userData;
  currentSceneGraph->getNumTriangles();
}

void
OivSceneGraph::getNumTriangles()
{
  size_t numTriangle = m_switchMesh->lowResolutionEnabled() ? m_lowResSceneGraph->getNumTriangles()
                                                               : m_fullResSceneGraph->getNumTriangles();
  std::string numTrianglesString; 
  std::ostringstream convert; 
  convert << numTriangle; 
  numTrianglesString = convert.str(); 
  m_gridFPSDisplay->string.set1Value(4, "# Triangles : " + numTrianglesString );
}

void 
OivSceneGraph::FPSDisplay(void * userData, SoSensor* sensor)
{
  OivSceneGraph* currentSceneGraph = (OivSceneGraph*) userData;
  currentSceneGraph->FPSDisplay();
  sensor->schedule();
}


void 
OivSceneGraph::FPSDisplay()
{
  float displayedValue = 0; 
  if (!m_FPSVector.empty())
  {
    float somme = 0; 
    for (size_t i = 0; i<m_FPSVector.size(); i++)
      somme+= m_FPSVector[i];
    displayedValue = somme/m_FPSVector.size(); 
  }
  std::string FPSString; 
  std::ostringstream convert; 
  convert << (int) ceil(displayedValue); 
  FPSString = convert.str(); 
  m_gridFPSDisplay->string.set1Value(3, "FPS : " + FPSString);
  m_FPSVector.clear();
}

void 
OivSceneGraph::viewerFPSCB( float fps )
{
  m_FPSVector.push_back((int) fps);
}

void 
OivSceneGraph::setFullResolutionDelay(int second)
{
  m_switchMesh->setFullResolutionDelay(second);
}

//------------------------------------------------------------------------
// Function called each time the mouse button is pressed in the viewer window
void OivSceneGraph::mousePressed(SoEventCallback *eventCB) 
{
  if (SoMouseButtonEvent::isButtonPressEvent(eventCB->getEvent(),SoMouseButtonEvent::BUTTON1))
  {
    const SoPickedPoint* pickedPoint = eventCB->getPickedPoint();
    if (pickedPoint != NULL)
    {
      const SoDetail* detail = pickedPoint->getDetail();
      // check if the transparent mesh bounding box (displayed with an SoCube) has been picked 
      if (detail->isOfType(SoCubeDetail::getClassTypeId()))
      {
        switchToLowResolutionMesh();

        SbVec3f pickedPointCoord = pickedPoint->getPoint();
        if (m_fullResSceneGraph->m_fenceSlice->polyline.getNum() == 0)
        {
          //the first point of a fence has been picked: we use the normal of the first 
          //picked face on the bounding box as the direction of the fence.
          SoCubeDetail* cubedetail = (SoCubeDetail*)detail;
          SbVec3f fenceDirection;
          switch (cubedetail->getPart()) {
            case 0: case 1: fenceDirection.setValue(0,0,1); break; 
            case 2: case 3: fenceDirection.setValue(1,0,0); break;
            case 4: case 5: fenceDirection.setValue(0,1,0); break;
          }
          m_fullResSceneGraph->m_fenceSlice->direction = fenceDirection;
        }
        m_fullResSceneGraph->m_fenceSlice->polyline.set1Value(m_fullResSceneGraph->m_fenceSlice->polyline.getNum(),pickedPointCoord);
      }
    }
  }
}

//------------------------------------------------------------------------
void OivSceneGraph::clearFence()
{
   m_fullResSceneGraph->m_fenceSlice->polyline.setNum(0);
}


//------------------------------------------------------------------------
void OivSceneGraph::initFence()
{
  SbVec3f boxsize = (SbVec3f)m_fullResMesh->getBBSize();
  SbVec3f boxcenter = (SbVec3f)m_fullResMesh->getCenter();

  float dx = boxsize[0]/4;
  float dy = boxsize[1]/4;
  float dz = boxsize[2]/2;

  SbVec3f po = boxcenter - SbVec3f(dx*2,dy*2,-dz);
  SbVec3f p0 = po + SbVec3f(dx*3,0,0);
  SbVec3f p1 = po + SbVec3f(dx,dy,0);
  SbVec3f p2 = po + SbVec3f(dx*3,dy*3,0);
  SbVec3f p3 = po + SbVec3f(dx,dy*4,0);

  m_fullResSceneGraph->m_fenceSlice->direction = SbVec3f(0,0,1);
  m_fullResSceneGraph->m_fenceSlice->polyline.setNum(4);
  m_fullResSceneGraph->m_fenceSlice->polyline.set1Value(0,p0);
  m_fullResSceneGraph->m_fenceSlice->polyline.set1Value(1,p1);
  m_fullResSceneGraph->m_fenceSlice->polyline.set1Value(2,p2);
  m_fullResSceneGraph->m_fenceSlice->polyline.set1Value(3,p3);
}

void OivSceneGraph::switchToLowResolutionMesh()
{
  size_t lowResolutionStep = (m_lowResMesh != NULL) ? m_lowResMesh->getStep() : 1;
  changeResolutionInfoDisplay(lowResolutionStep);
  if (lowResolutionStep > 1)
    m_switchMesh->switchToLowResolution();
}

void OivSceneGraph::switchToFullResolutionMesh()
{
  m_switchMesh->switchToFullResolution();
}

void OivSceneGraph::scheduleFullResolutionMesh()
{
  m_switchMesh->scheduleFullResolution();
}

void OivSceneGraph::scheduleFullResolutionMesh(void *sg, SoAction* action) 
{ 
  if (action->isOfType(SoGLRenderAction::getClassTypeId()) && SoInteractionElement::isInteracting(action->getState()))
    ((OivSceneGraph*)sg)->scheduleFullResolutionMesh(); 
}
