#ifndef _OIVSCENEGRAPH_H
#define _OIVSCENEGRAPH_H

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/manips/SoClipPlaneManip.h>
#include <Inventor/SbElapsedTime.h> 
#include <Inventor/SbString.h>

#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoCombineColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSlab.h>
#include <MeshVizXLM/mapping/nodes/MoMeshPlaneSlice.h>
#include <MeshVizXLM/mapping/nodes/MoMeshFenceSlice.h>
#include <MeshVizXLM/mapping/nodes/MoCellFilter.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSet.h>
#include <MeshVizXLM/mapping/nodes/MoDataBinding.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoMeshIsosurface.h>
#include <MeshVizXLM/mapping/nodes/MoMeshOutline.h>

#include <MeshVizXLM/extractors/MiExtractorCallback.h>

#include "MeshResolutionSwitch.h" 
#include "WeightedAverageToColorMapping.h" 

class SoJackDragger;
class SoDialogCheckBox;
class SoDialogIntegerSlider;
class SoText2;
class SoFieldSensor;
class SoSensor;
class SoSwitch;
class SoCube;
class SoTranslation;
class SoMaterial;

class GuiDialogViz;
class LogicalSliceFilter;
class CellFilter;
class EclipseMesh;
class PillarMesh;
class IndexedPillarMesh;
class SubSampledMesh;
class MeshResolutionSwitch;
class SoEventCallback;

#define NUM_LOGICALSLICE 3

void display_time(const std::string &what, double timeElapsed);
void display(const std::string &what);

class OivSceneGraph 
{

public:
  OivSceneGraph();
  ~OivSceneGraph();

  SoSeparator* getRoot() const
  {
    return m_root;
  }

  void setMesh(IndexedPillarMesh* mesh);
  void setMesh(PillarMesh* mesh);
  void autoLowResolutionAdjust(bool enable, SoSFInt32& resolution);
  void adjustLowResolution(double resolution);
  void changeResolutionInfoDisplay(size_t resolutionStep);
  void updateGeomSlice();
  void setGeomSlice(int axis);
  bool rotateGeomSlice(float angle);
  void setFrame(int frame);
  void setSkinTransparency(float value);
  void setHullTransparency(float value);
  void enableROIFiltering(bool enable);
  // return true if enabled
  bool setROIFiltering(size_t imin,size_t imax,size_t jmin,size_t jmax,size_t kmin,size_t kmax);
  bool setDataRangeFiltering(double min, double max, const std::string& propertyName = "");
  int getGapBetweenSlabs(int sliceId, int sliceNumber);
  bool moveSlab(size_t sliceId, size_t index, int currentStep);
  void updateSlabs(int sliceId, int sliceNumber, int currentIndex, int currentThickness);
  void updateFilter();
  bool changeSlabThickness(size_t sliceId, size_t thickness);
  bool setIsosurfaceValue(float value);
  void selectIsoScalar(const std::string& name);
  void setCombiningWeight(double weight);

  void updateMeshBoundingBox(PillarMesh* mesh);

  void displayColor();
  void displayPorosity();
  void displayDepth();
  void displayCombined();

  void viewerFPSCB( float fps);

  void updateColorMap(MoPredefinedColorMapping::PredefColorMapping newMap);
  

  SbVec3s getMeshDim();
  SoTransform* getTransform();

  bool availableLowResolution() const {return m_lowResMesh != NULL; }
  bool availableTimer() const { return !m_asyncProcessing;  }

  MoDrawStyle* getDrawStyle();
  SoSwitch* getSwitchSkin();
  SoSwitch* getSwitchHull();
  SoSwitch* getSwitchHullOutline();
  SoSwitch* getSwitchPlaneSlice();
  SoSwitch* getSwitchFenceSlice();
  SoSwitch* getSwitchLogicalSlice(size_t sliceId);
  SoSwitch* getSwitchROIDataRangeFilter();
  SoSwitch* getSwitchIsoSurface();
  SoSwitch* getSwitchMeshBoundingBox();

  const CellFilter& getROIDataRangeFilter() const;
  const CellFilter& getlowROIDataRangeFilter() const;

  void clearFence();

  SoClipPlane* getClipPlane();

  void mousePressed(SoEventCallback *eventCB);

  void switchToLowResolutionMesh();
  void switchToFullResolutionMesh();
  void scheduleFullResolutionMesh();
  void setFullResolutionDelay(int second); 

  static void updateGeomSlice(void* scenegraph, SoSensor*);

  static void switchToLowResolutionMesh(void *sg, SoDragger *) { ((OivSceneGraph*)sg)->switchToLowResolutionMesh(); }
  static void scheduleFullResolutionMesh(void *sg, SoAction* action);

protected:
  void createSceneGraph();
  void assembleSceneGraph();
  void setFullResMesh(PillarMesh* mesh);
  void setLowResMesh(IndexedPillarMesh* mesh);
  void initFence();

protected:
  PillarMesh* m_fullResMesh;
  SubSampledMesh* m_lowResMesh;
  CellFilter* m_ROIDataRangeFilter;
  CellFilter* m_lowResROIDataRangeFilter;

  SoSeparator* m_root;
  SoSwitch* m_colorMappingSwitch;
  MoPredefinedColorMapping* m_depthColorMapping;
  MoPredefinedColorMapping* m_porosityColorMapping;
  MoCombineColorMapping* m_combineColorMapping;
  // Interface implementation for combining values for color mapping
  WeightedAverageToColorMapping m_weightedAverageMapping;
  MoDrawStyle* m_drawStyle;
  MoMaterial* m_skinMaterial;
  MoMaterial* m_hullMaterial;

  SoTransform *m_transform;
  SoClipPlaneManip* m_planeManip;;
  SoClipPlane* m_clipPlane;
  SoText2* m_gridInfoDisplay;
  SoText2* m_gridFPSDisplay;
  SoText2* m_gridTrianglesDisplay;


  MeshResolutionSwitch* m_switchMesh; // toggle low resolution
  SoSwitch* m_switchDragger;

  SoCube* m_meshBoundingBox;
  SoTranslation* m_meshBoundingBoxPos;
  SoSeparator* m_meshBoundingBoxSep;
  SoSwitch* m_switchMeshBoundingBox;
  SoMaterial* m_meshBoundingMaterial;

  class MyMeshResolutionSwitch : public MeshResolutionSwitch
  {
  public:
    MyMeshResolutionSwitch(OivSceneGraph* sg):oivSceneGraph(sg)  {}
  protected:
    virtual void fullResolutionIsBack()
    {
      oivSceneGraph->changeResolutionInfoDisplay(1);
    }
    OivSceneGraph* oivSceneGraph;
  };

  class MeshSceneGraph : public SoSeparator
  {
    void assembleSceneGraph(MoMaterial* skinMat, MoMaterial* hullMat);
  public:
    MeshSceneGraph(MoMaterial* skinMat,MoMaterial* hullMat, MiExtractorCallback* extractorCB = NULL);
    void connectFrom(const MeshSceneGraph& sg);
    void updateSlabs(int sliceId, int step, int numSlice, int currentIndex, int currentThickness);
    size_t getNumTriangles() const;

    MoMesh* m_moMesh;
    MoScalarSet *m_moDepthScalarSet;
    MoScalarSet *m_moPorosityScalarSet;

    MoMeshSkin *m_skin;
    MoMeshSkin *m_hullSkin;
    SoGroup *m_slabGroup[NUM_LOGICALSLICE];
    MoCellFilter* m_moCellFilter;

    MoMeshOutline *m_hullOutline;

    MoMeshPlaneSlice* m_planeSlice;
    MoMeshFenceSlice* m_fenceSlice;
    MoMeshIsosurface* m_isosurface;

    SoSwitch* m_switchSkin;
    SoSwitch* m_switchHullOutline;
    SoSwitch* m_switchHull;
    SoSwitch* m_switchPlaneSlice;
    SoSwitch* m_switchFenceSlice;
    SoSwitch* m_switchLogicalSlice[NUM_LOGICALSLICE];
    SoSwitch* m_switchROIDataRangeFilter;
    SoSwitch* m_switchIsosurface;
  };
  
  MeshSceneGraph* m_lowResSceneGraph;
  MeshSceneGraph* m_fullResSceneGraph;

  SbVec3s m_meshDim;
  SbVec3f m_meshCenter;

  SoTimerSensor* m_FPSTimersensor;
  std::vector<int> m_FPSVector;

  static void FPSDisplay(void *  userData, SoSensor* sensor);
  static void getNumTriangles(void* userData, SoAction *action);
  void FPSDisplay();
  void getNumTriangles();

  // Workaround eBug #4628: Due to OIV Draggers.
  SoFieldSensor* m_planeSensor;

	SbVec3f m_sliceNormal;
	SbVec3f m_rotAxis;
  
  class LowResCallback : public MiExtractorCallback
  {
  public:
    LowResCallback(OivSceneGraph& sg) : m_scenegraph(sg), m_count(0), m_geomChanged(false), 
                      m_topoChanged(false), m_dataSetChanged(false), m_evalLowResolution(true), m_exceedCounter(0) {}
    void beginExtract(const std::string, bool geomChanged, bool topoChanged, bool dataSetChanged, size_t);
    void endExtract();
    bool beginPhase(size_t , std::string , size_t ) { return true; }
    bool endPhase() { return true; }
    bool endStep(size_t ) { return true; }
    double getEndStepCallPeriod() { return 1.0; }

    void evalLowResolution(bool enable) { m_evalLowResolution = enable; }

  protected:
    OivSceneGraph& m_scenegraph;
    static double s_lowResolutionExtrationTimeReference;
    size_t m_count;
    SbElapsedTime m_locatTime;
    std::list<double> m_startTime;
    bool m_geomChanged, m_topoChanged, m_dataSetChanged;
    bool m_evalLowResolution;
  private:
    size_t m_exceedCounter;
  };
  LowResCallback m_lowResCB;

  class FullResCallback : public LowResCallback
  {
  public:
    FullResCallback(OivSceneGraph& sg) : LowResCallback(sg), m_maxExtractionTime(0) {}
    void beginExtract(const std::string, bool geomChanged, bool topoChanged, bool dataSetChanged, size_t);
    void endExtract();
    bool beginPhase(size_t , std::string , size_t );
    bool endPhase();

    void apply();
    void reset() { m_maxExtractionTime = 0; }

  private:
    double m_maxExtractionTime;
     std::list<std::string> m_name;
     std::list<std::string> m_phaseName;
     std::list<double> m_startPhase;
  };
  FullResCallback m_timer;
  bool m_asyncProcessing;

};


inline SbVec3s 
OivSceneGraph::getMeshDim()
{
  return m_meshDim;
}

inline MoDrawStyle* 
  OivSceneGraph::getDrawStyle()
{
  return m_drawStyle;
}

inline SoSwitch* 
OivSceneGraph::getSwitchSkin()
{
  return m_fullResSceneGraph->m_switchSkin;
}

inline SoSwitch* 
OivSceneGraph::getSwitchHull()
{
  return m_fullResSceneGraph->m_switchHull;
}

inline SoSwitch* 
OivSceneGraph::getSwitchHullOutline()
{
  return m_fullResSceneGraph->m_switchHullOutline;
}

inline SoTransform* 
OivSceneGraph::getTransform()
{
  return m_transform;
}

inline SoSwitch* 
OivSceneGraph::getSwitchPlaneSlice()
{
  return m_fullResSceneGraph->m_switchPlaneSlice;
}

inline SoSwitch* 
OivSceneGraph::getSwitchFenceSlice()
{
  return m_fullResSceneGraph->m_switchFenceSlice;
}


inline SoSwitch* 
OivSceneGraph::getSwitchMeshBoundingBox()
{
  return m_switchMeshBoundingBox;
}

inline SoSwitch*
OivSceneGraph::getSwitchLogicalSlice(size_t sliceId)
{
  return m_fullResSceneGraph->m_switchLogicalSlice[sliceId];
}

inline SoSwitch* 
OivSceneGraph::getSwitchIsoSurface()
{
  return m_fullResSceneGraph->m_switchIsosurface;
}

inline SoSwitch*
OivSceneGraph::getSwitchROIDataRangeFilter()
{
  return m_fullResSceneGraph->m_switchROIDataRangeFilter;
}

inline const CellFilter&
OivSceneGraph::getROIDataRangeFilter() const
{
  return *m_ROIDataRangeFilter;
}

inline const CellFilter&
  OivSceneGraph::getlowROIDataRangeFilter() const
{
  return *m_lowResROIDataRangeFilter;
}

inline SoClipPlane*
OivSceneGraph::getClipPlane()
{
  return m_clipPlane;
}

#endif

