#pragma once

#include <MeshVizXLM/mapping/interfaces/MiColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/MxTimeStamp.h>
#include <Inventor/SbColorRGBA.h>

#include <vector> 

/**
 */
class WeightedAverageToColorMapping : public MiColorMapping<std::vector<double>, SbColorRGBA>
{
public:
  WeightedAverageToColorMapping() : m_combiningWeight(0.5), m_timeStamp(MxTimeStamp::getTimeStamp()) 
  {
    m_predColorMapping = new MoPredefinedColorMapping;
    m_predColorMapping->predefColorMap = MoPredefinedColorMapping::BLUE_WHITE_RED;
    m_predColorMapping->minValue = 0.f;
    m_predColorMapping->maxValue = 1.f;
    m_predColorMapping->ref();
    m_linearColorMapping = m_predColorMapping->getColorMapping();
  }

  ~WeightedAverageToColorMapping()
  {
    m_predColorMapping->unref();
  }

  /**
  * Returns the color associated with the given value.
  */
  virtual SbColorRGBA getColor(std::vector<double> values) const
  {
    double normalizedV0 = (values[0] - min[0]) / rlength[0];
    double normalizedV1 = (values[1] - min[1]) / rlength[1];
    double combinedValue = (1 - m_combiningWeight) * normalizedV0 + m_combiningWeight * normalizedV1;

    // then use the linear colormap implementation
    return m_linearColorMapping->getColor(combinedValue);
  }

  void setWeight(double weight)
  {
    m_combiningWeight = weight;
    m_timeStamp = MxTimeStamp::getTimeStamp();
  }

  void setPredefinedColorMapping(MoPredefinedColorMapping::PredefColorMapping predColor)
  {
    m_predColorMapping->predefColorMap = predColor;
    m_linearColorMapping = m_predColorMapping->getColorMapping();
    m_timeStamp = MxTimeStamp::getTimeStamp();
  }

  void setMinMaxValues(float min1, float max1, float min2, float max2)
  {
    min[0] = min1;
    min[1] = min2;
    rlength[0] = max1 - min1;
    rlength[1] = max2 - min2;
    m_timeStamp = MxTimeStamp::getTimeStamp();
  }

  /**
  * Returns for this colormap a unique time stamp across all colormaps in the application.
  * When the content of the colormap changes, the time stamp must increase. The time stamp
  * allows representations to identify this colormap and to be aware of any changes to it.
  * @note MxTimeStamp can be used to generate a valid time stamp.
  */
  virtual size_t getTimeStamp() const { return m_timeStamp; }
  
protected:
  MoPredefinedColorMapping* m_predColorMapping;
  const MiColorMapping<double, SbColorRGBA>* m_linearColorMapping;
  double m_combiningWeight;
  float min[2], rlength[2];
  size_t m_timeStamp;
};


