#include "CachedPillarTopology.h"
#include "CachedPillarGeometry.h"
#include "pillargrid/PillarGrid.h"
#include "DemoSettings.h"

//-----------------------------------------------------------------------------
CachedPillarTopology::CachedPillarTopology(const PillarGrid& grid, CachedPillarGeometry& geometry)
: PillarTopology<PillarGrid>(grid), m_endNodeId(0), m_numJ(0)
{
  if (grid.getNumZCorns() <= std::numeric_limits<NodeIdT>::max())
  {
    m_numJ = grid.get_dimJ();
#if HORIZONTAL_FAULTS
    m_numK = grid.get_dimK();
    m_indices.resize(getNumCells()*8,0);
#else
    m_numK = grid.get_dimK()+1;
    m_indices.resize(grid.get_dimI()*grid.get_dimJ()*(grid.get_dimK()+1)*4,0);
#endif
    m_numKxI = m_numK*grid.get_dimI();
    init();
    generateCellNodeIndices(grid,geometry);
  }
  else
  {
    std::cout << "NodeIdT not sufficient to support " << grid.getNumZCorns() << " vertex indices in the topology cache." << std::endl;
  }
}

//-----------------------------------------------------------------------------
void
CachedPillarTopology::init()
{
  m_next_direction[0] = 0;
  m_next_direction[1] = -(int) m_numK;
  m_next_direction[3] = -(int) m_numKxI;
  m_next_direction[2] = m_next_direction[3] + m_next_direction[1];
  
  m_next_direction[4] = -1;
  m_next_direction[5] = m_next_direction[1]-1;
  m_next_direction[6] = m_next_direction[2]-1;
  m_next_direction[7] = m_next_direction[3]-1;

  for (size_t c = 0; c < m_numCorners; ++c)
  {
    m_next_direction[c] *= (int) m_numCorners;
    m_next_direction[c] += (int) c;
  }
}

//-----------------------------------------------------------------------------
void 
CachedPillarTopology::getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                    size_t& n4, size_t& n5, size_t& n6, size_t& n7) const
{
  std::vector<NodeIdT>::const_iterator it = m_indices.begin() + (j*m_numKxI + i*m_numK + k)* m_numCorners;
  n0 = (size_t)*it++;
  n1 = (size_t)*it++;
  n2 = (size_t)*it++;
  n3 = (size_t)*it++;
  n4 = (size_t)*it++;
  n5 = (size_t)*it++;
  n6 = (size_t)*it++;
  n7 = (size_t)*it;
}

//-----------------------------------------------------------------------------
size_t 
CachedPillarTopology::getEndNodeId() const
{
  return m_endNodeId;
}

//-----------------------------------------------------------------------------
void
CachedPillarTopology::generateCellNodeIndices(const PillarEGrid& grid, CachedPillarGeometry& geometry)
{
  std::pair<CoordT,size_t> zcorns[8];
  CoordT current_zcorn;
  int previous_corner;
  size_t offset;
  size_t refCell2dId = 0, refcellId = 0;
  size_t nPillarI = grid.get_dimI()+1;
  size_t nPillarJ = grid.get_dimJ()+1;
  size_t nLayerK = grid.get_dimK()+1;
  
  for (size_t j = 0; j < nPillarJ; ++j, refCell2dId-=m_numK) 
    for (size_t i = 0; i < nPillarI; ++i, refCell2dId+=m_numK)
    {
      refcellId = refCell2dId;
      for (size_t k = 0; k < nLayerK; ++k,++refcellId)
      {
        grid.getPillarZCorns(i,j,k,zcorns);

        // check each corner with another and use
        // use same index if shared
        for (int cn = 0; cn < (int) m_numCorners; ++cn) 
        {
          if (zcorns[cn].first != PillarGrid::m_undef)
          {
            offset = refcellId*m_numCorners + m_next_direction[cn];
            current_zcorn = zcorns[cn].first; 
            previous_corner = cn;
            do
            {
              --previous_corner;
            }
            while (previous_corner >= 0 && current_zcorn != zcorns[previous_corner].first);
            if (previous_corner < 0)
              zcorns[cn].second = geometry.addPoint((NodeIdT)zcorns[cn].second);
            else
              zcorns[cn].second = zcorns[previous_corner].second;
            m_indices[offset] = (NodeIdT) zcorns[cn].second;
          }
        }
      }
    }
  m_endNodeId = geometry.size();
}

//-----------------------------------------------------------------------------
void
  CachedPillarTopology::generateCellNodeIndices(const PillarGridJIK& grid, CachedPillarGeometry& geometry)
{
  const CoordT* zcorns;
  CoordT current_zcorn;
  int previous_corner;
  size_t refZcornId, zcornIds[8], offset;
  size_t refCell2dId = 0, refcellId = 0;
  size_t nPillarI = grid.get_dimI()+1;
  size_t nPillarJ = grid.get_dimJ()+1;
  size_t nLayerK = grid.get_dimK()+1;

  for (size_t j = 0; j < nPillarJ; ++j, refCell2dId-=m_numK) 
    for (size_t i = 0; i < nPillarI; ++i, refCell2dId+=m_numK)
    {
      refcellId = refCell2dId;
      for (size_t k = 0; k < nLayerK; ++k,++refcellId)
      {
        refZcornId = grid.getPillarZCorns(i,j,k,zcorns);

        // check each corner with another and use
        // use same index if shared
        for (int cn = 0; cn < (int)m_numCorners; ++cn)
        {
          if (zcorns[cn] != PillarGrid::m_undef)
          {
            offset = refcellId*m_numCorners + m_next_direction[cn];
            current_zcorn = zcorns[cn]; 
            previous_corner = cn;
            do
            {
              --previous_corner;
            }
            while (previous_corner >= 0 && current_zcorn != zcorns[previous_corner]);
            if (previous_corner < 0)
              zcornIds[cn] = geometry.addPoint((NodeIdT) refZcornId+cn);
            else
              zcornIds[cn] = zcornIds[previous_corner];
            m_indices[offset] = (NodeIdT) zcornIds[cn];
          }
        }
      }
    }
    m_endNodeId = geometry.size();
}
