#ifndef  _PillarTopology_H
#define  _PillarTopology_H

#include <MeshVizXLM/mesh/topology/MiHexahedronTopologyExplicitIjk.h>
#include <MeshVizXLM/MxTimeStamp.h>

#include "DemoSettings.h"
#include "CachedPillarGeometry.h"
#include "pillargrid/PillarGrid.h"

template <typename PillaGridT>
class PillarTopology : public MiHexahedronTopologyExplicitIjk
{
public:
  PillarTopology(const PillaGridT& grid);

  MbVec3d getCenter(size_t i, size_t j, size_t k, const MiGeometryI& geometry) const;

  /**
  * Returns 8 indices of a cell of this topology. The cell 
  * is identified by three id i,j,k.
  */
  virtual void getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                                                size_t& n4, size_t& n5, size_t& n6, size_t& n7) const;

  virtual MiMeshIjk::StorageLayout getStorageLayout() const { return MiMeshIjk::LAYOUT_JIK; }

  /**
  * Returns the max node id +1 used by this topology. This topology used only 
  * node index in the interval [beginNodeId,EndNodeId[. Thus the maximum node
  * index used by the topology is getEndNodeId() -1 and the number of 
  * nodes used by this topology is getEndNodeId()-getBeginNodeId().
  */
  size_t getEndNodeId() const;

    /**
  * Returns the number of cells on the first logical axis.
  */
  size_t getNumCellsI() const { return m_grid.get_dimI(); }

  /**
  * Returns the number of cells on the second logical axis.
  */
  size_t getNumCellsJ() const { return m_grid.get_dimJ(); }

  size_t getNumCells() const { return getNumCellsI()*getNumCellsJ()*getNumCellsK();}

  /**
  * Returns the number of cells on the third logical axis.
  */
  size_t getNumCellsK() const { return m_grid.get_dimK(); }

  virtual size_t getTimeStamp() const {return m_timeStamp;}

  bool hasDeadCells()const
  {
    return m_grid.getNumNonActiveCells() != 0;  
  }

  /** 
  * Returns true if the cell of index (i,j,k) should be ignored.
  * This value is ignored if the hasDeadCell() method returns false.
  * The default implementation returns always false.
  * @param (i,j,k) the index of the cell to check
  */ 
  bool isDead(size_t i, size_t j, size_t k) const 
  {
    return !m_grid.isValid(i,j,k);
  }

  /**
  * Returns the number of activecells
  */
  size_t getNumActiveCells() const;

  virtual size_t getMemoryFootPrint() const {return 0;}

protected:
  size_t m_numCorners;

private:
  const PillaGridT& m_grid;
  size_t m_timeStamp;

  static size_t s_direction[8][3];
};

template <typename PillaGridT> size_t PillarTopology<PillaGridT>::s_direction[8][3] = 
{
  {0,0,0},{1,0,0},{1,1,0},{0,1,0},{0,0,1},{1,0,1},{1,1,1},{0,1,1}
};

//-----------------------------------------------------------------------------
template <typename PillaGridT>
PillarTopology<PillaGridT>::PillarTopology( const PillaGridT& grid )
  : MiHexahedronTopologyExplicitIjk()
  , m_numCorners( HORIZONTAL_FAULTS ? 8 : 4 )
  , m_grid( grid )
  , m_timeStamp( MxTimeStamp::getTimeStamp() )
{
}

//-----------------------------------------------------------------------------
template <typename PillaGridT>
inline size_t
PillarTopology<PillaGridT>::getNumActiveCells() const 
{
  return m_grid.get_dimI()*m_grid.get_dimJ()*m_grid.get_dimK() - m_grid.getNumNonActiveCells();
}

//-----------------------------------------------------------------------------
template <>
inline void 
PillarTopology<PillarEGrid>::getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                                                              size_t& n4, size_t& n5, size_t& n6, size_t& n7) const
{
  std::pair<CoordT,size_t> zcorns[8];
  size_t zcornId[8], corner;
  CoordT zcorn;

  m_grid.getPillarZCorns(i,j,k,zcorns);
  n0 = zcornId[0] = zcorns[0].second;

  m_grid.getPillarZCorns(i+1,j,k,zcorns);
  corner = 0;
  if (zcorns[1].first != zcorns[0].first)
    corner = 1;
  n1 = zcornId[1] = zcorns[corner].second;

  for (size_t n = 2; n < m_numCorners; ++n)
  {
    m_grid.getPillarZCorns(i+s_direction[n][0],j+s_direction[n][1],k+s_direction[n][2],zcorns);
    corner = 0;
    zcorn = zcorns[n].first;
    while(zcorn != zcorns[corner].first)
      ++corner;
    zcornId[n] = zcorns[corner].second;
  }

  n2 = zcornId[2];
  n3 = zcornId[3];

#if HORIZONTAL_FAULTS
  n4 = zcornId[4];
  n5 = zcornId[5];
  n6 = zcornId[6];
  n7 = zcornId[7];
#else
  if (k+1 < m_grid.get_dimK())
  {
    n4 = m_grid.getNextUpperCellFaceZCornId(zcornId[0]);
    n5 = m_grid.getNextUpperCellFaceZCornId(zcornId[1]);
    n6 = m_grid.getNextUpperCellFaceZCornId(zcornId[2]);
    n7 = m_grid.getNextUpperCellFaceZCornId(zcornId[3]);
  }
  else
  {
    n4 = m_grid.getNextUpperFaceZCornId(zcornId[0]);
    n5 = m_grid.getNextUpperFaceZCornId(zcornId[1]);
    n6 = m_grid.getNextUpperFaceZCornId(zcornId[2]);
    n7 = m_grid.getNextUpperFaceZCornId(zcornId[3]);
  }
#endif
}

//-----------------------------------------------------------------------------
template <>
inline void 
PillarTopology<PillarGridJIK>::getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                                                                      size_t& n4, size_t& n5, size_t& n6, size_t& n7) const
{
  const CoordT* zcorns;
  size_t zcornId[8], corner;
  CoordT zcorn;

  n0 = zcornId[0] = m_grid.getPillarZCorns(i,j,k,zcorns);
  zcornId[1] = m_grid.getPillarZCorns(i+1,j,k,zcorns);
  if (zcorns[1] != zcorns[0])
    ++zcornId[1];
  n1 = zcornId[1];

  for (size_t n = 2; n < m_numCorners; ++n)
  {
    zcornId[n] = m_grid.getPillarZCorns(i+s_direction[n][0],j+s_direction[n][1],k+s_direction[n][2],zcorns);
    corner = 0;
    zcorn = zcorns[n];
    while(zcorn != zcorns[corner])
      ++corner;
    zcornId[n] += corner;
  }

  n2 = zcornId[2];
  n3 = zcornId[3];

#if HORIZONTAL_FAULTS
  n4 = zcornId[4];
  n5 = zcornId[5];
  n6 = zcornId[6];
  n7 = zcornId[7];
#else
  if (k+1 < m_grid.get_dimK())
  {
    n4 = m_grid.getNextUpperCellFaceZCornId(zcornId[0]);
    n5 = m_grid.getNextUpperCellFaceZCornId(zcornId[1]);
    n6 = m_grid.getNextUpperCellFaceZCornId(zcornId[2]);
    n7 = m_grid.getNextUpperCellFaceZCornId(zcornId[3]);
  }
  else
  {
    n4 = m_grid.getNextUpperFaceZCornId(zcornId[0]);
    n5 = m_grid.getNextUpperFaceZCornId(zcornId[1]);
    n6 = m_grid.getNextUpperFaceZCornId(zcornId[2]);
    n7 = m_grid.getNextUpperFaceZCornId(zcornId[3]);
  }
#endif
}

//-----------------------------------------------------------------------------
template <typename PillaGridT>
inline size_t
PillarTopology<PillaGridT>::getEndNodeId() const
{
  return m_grid.getNumZCorns();
}

//-----------------------------------------------------------------------------
template <typename PillaGridT>
inline MbVec3d
PillarTopology<PillaGridT>::getCenter(size_t i, size_t j, size_t k, const MiGeometryI& geometry) const
{
  size_t nodes[8];
  MbVec3d center(0);
  getCellNodeIndices(i, j, k, nodes[0], nodes[1], nodes[2], nodes[3], nodes[4], nodes[5], nodes[6], nodes[7]);
  for (size_t n = 0; n < 8; ++n)
    center += geometry.getCoord(nodes[n]);

  center /= 8.0;
  return center;
}


#endif







