//------------------------------------------------------------------------
//                                                                        
// demonstration classes to visualize 3D volumic mesh with MeshViz XLM.
//                                                                        
//  author : J-Michel Godinaud                                            
//------------------------------------------------------------------------
#include "Mesh3DVecViewer.h"
#include "commonAuditor.h"

class LengthSliderAuditor : public SoDialogRealSliderAuditor {
  Mesh3DVecViewer *mViewer;
public:
  LengthSliderAuditor(Mesh3DVecViewer *mvw) { this->mViewer = mvw; }
  void dialogRealSlider(SoDialogRealSlider *slider) {
    mViewer->setLengthValue(slider->value.getValue());
  }  
};

//------------------------------------------------------------------------
Mesh3DVecViewer::Mesh3DVecViewer()
{
}/*---------------------------------------------------------------------------*/

void Mesh3DVecViewer::buildSceneGraph( const MyMesh &myMesh, SoGroup *scene_mesh3D, SoSFInt32* /*which_dataset*/)
{
  SbBox3f mesh_bounding_box;

  scene_mesh3D->setName("Vector_Scene");

  // get the mesh bounding box
  mesh_bounding_box = myMesh.getBoundingBoxf();

  float sx,sy,sz; mesh_bounding_box.getSize(sx,sy,sz);
  float maxs = (sx > sy) ? sx : sy; if (sz > maxs) maxs = sz;

  v_vecLengthFactor = 0.0025f*maxs;

  // define the vector field
  v_meshGridPlaneSlice = new MoMeshGridPlaneSlice;
  v_meshGridPlaneSlice->step = v_vecLengthFactor * 4;

  v_meshGrid = new MoMesh;
  v_meshGrid->connectFrom(v_meshGridPlaneSlice);

  v_meshVector = new MoMeshVector;
  v_meshVector->scaleFactor = v_vecLengthFactor;

  v_meshVectorSwitch = new SoSwitch;
  v_meshVectorSwitch->whichChild = SO_SWITCH_NONE;

  SoPickStyle *pickStyle = new SoPickStyle;
  pickStyle->style = SoPickStyle::UNPICKABLE;

  SoDrawStyle* invisibleStyle = new SoDrawStyle;
  invisibleStyle->style = SoDrawStyle::INVISIBLE;

  SoDrawStyle* visibleStyle = new SoDrawStyle;
  visibleStyle->style = SoDrawStyle::FILLED;

  // assemble the scene graph of this 3D vector viewer
  {
    scene_mesh3D->addChild(v_meshVectorSwitch);
    {
      v_meshVectorSwitch->addChild(pickStyle);
      v_meshVectorSwitch->addChild(invisibleStyle);
      v_meshVectorSwitch->addChild(v_meshGridPlaneSlice);
      v_meshVectorSwitch->addChild(v_meshGrid);
      v_meshVectorSwitch->addChild(visibleStyle);
      v_meshVectorSwitch->addChild(v_meshVector);
    }
  }

}/*---------------------------------------------------------------------------*/

void Mesh3DVecViewer::setLengthValue(float val)
{
  v_meshVector->scaleFactor.setValue(v_vecLengthFactor = val);
}/*---------------------------------------------------------------------------*/

SoDialogComponent *
Mesh3DVecViewer::buildDialogBox ()
{
  SbString ivPath = SoPreferences::getString("OIVHOME",".") + "/examples/source/MeshVizXLM/demonstrators/MeshViewer/GuiTabVector3D.iv";

  SoInput myInput;
  if (!myInput.openFile(ivPath.getString())) 
    exit (1);
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (myGroup == NULL) exit (1);
  v_DialogBox = (SoDialogGroup *)myGroup->getChild(0);

  //////////// 
  // vector check enable
  SoDialogCheckBox* vectorCheckEnable = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("VectorEnable");
  vectorCheckEnable->addAuditor(new CheckSwitchAllAuditor(v_meshVectorSwitch));
  vectorCheckEnable->state = (v_meshVectorSwitch->whichChild.getValue() == SO_SWITCH_ALL);
  vectorCheckEnable->applyDlgCptAuditor();

  // vector grid spacing
  SoDialogRealSlider* vectorGridSpacing = (SoDialogRealSlider *)v_DialogBox->searchForAuditorId("VectorGridSpacing");
  vectorGridSpacing->addAuditor(new RealSliderAuditor(v_meshGridPlaneSlice->step));
  vectorGridSpacing->value = v_meshGridPlaneSlice->step.getValue(); 
  vectorGridSpacing->min = v_meshGridPlaneSlice->step.getValue() / 3; 
  vectorGridSpacing->max = v_meshGridPlaneSlice->step.getValue() * 2; 
  vectorGridSpacing->applyDlgCptAuditor();

  // vector length
  SoDialogRealSlider* vectorLength = (SoDialogRealSlider *)v_DialogBox->searchForAuditorId("VectorLength");
  vectorLength->addAuditor(new LengthSliderAuditor(this));
  vectorLength->value = v_meshVector->scaleFactor.getValue(); 
  vectorLength->min = v_meshVector->scaleFactor.getValue() / 3; 
  vectorLength->max = v_meshVector->scaleFactor.getValue() * 2; 
  vectorLength->applyDlgCptAuditor();

  // vector check enable arrow heads
  SoDialogCheckBox* vectorCheckEnableHeads = (SoDialogCheckBox *) v_DialogBox->searchForAuditorId( "VectorHeadEnable" );
  vectorCheckEnableHeads->addAuditor( new CheckBoolAuditor(v_meshVector->arrow) );
  vectorCheckEnableHeads->applyDlgCptAuditor();

  // vector shift
  SoDialogRealSlider* vectorShift = (SoDialogRealSlider *) v_DialogBox->searchForAuditorId( "VectorShift" );
  vectorShift->addAuditor( new RealSliderAuditor( v_meshVector->shiftFactor ) );
  vectorShift->applyDlgCptAuditor();

  // vector thickness
  SoDialogRealSlider* vectorThickness = (SoDialogRealSlider *) v_DialogBox->searchForAuditorId( "VectorThickness" );
  vectorThickness->addAuditor( new RealSliderAuditor( v_meshVector->thicknessFactor ) );
  vectorThickness->applyDlgCptAuditor();

  return v_DialogBox;
}/*---------------------------------------------------------------------------*/

void Mesh3DVecViewer::updateAllConnectedToDragger(SoJackDragger *dragger, SbVec3f plane_normal){
  if (!v_IsDraggerConnected) return;

  // get the dragger position
  SbVec3f drager_position = dragger->translation.getValue();

  // rotate the plane's normal by the dragger rotation
  SbRotation rotation = dragger->rotation.getValue();
  rotation.multVec(SbVec3f(0,1,0),plane_normal);

  // translate vec-field
  v_meshGridPlaneSlice->plane.setValue(SbPlane(plane_normal,drager_position));

}/*---------------------------------------------------------------------------*/



void Mesh3DVecViewer::preWriteAction() {
  enableConnection(v_meshVector,FALSE);
}/*---------------------------------------------------------------------------*/

void Mesh3DVecViewer::postWriteAction() {
  enableConnection(v_meshVector,TRUE);
}/*---------------------------------------------------------------------------*/

void Mesh3DVecViewer::enableConnection(SoNode *node, SbBool flag) {
  SoFieldList fields;
  int num_fields = node->getFields(fields);
  for (int i=0; i<num_fields; i++) {
    SoField *field = fields[i];
    if (field->isConnected()) field->enableConnection(flag);
  }
}/*---------------------------------------------------------------------------*/

void Mesh3DVecViewer::updateParallelMode()
{
  v_meshGridPlaneSlice->parallel = getParallelMode();
  v_meshVector->parallel = getParallelMode();
}

