//****************************************************************************
//                                                                         
// demonstration class to get probe informations with MeshViz
//                                                                         
//  author : J-Michel Godinaud                                             
//****************************************************************************
#include <float.h>
#include "MeshProbeViewer.h"
#include "MeshProbeSheet.h"
#include "commonAuditor.h"

#include <Inventor/nodes/SoEventCallback.h> 
#include <Inventor/nodes/SoAnnotation.h> 
#include <Inventor/nodes/SoBaseColor.h> 
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/SoPickedPoint.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoLineDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>

//------------------------------------------------------------------------
class CheckProbeToMouseAuditor : public SoDialogCheckBoxAuditor 
{
  MeshProbeViewer *mv;
public:
  CheckProbeToMouseAuditor(MeshProbeViewer *mpv) { this->mv = mpv; }
  void dialogCheckBox(SoDialogCheckBox *checkBox) {
    mv->connectProbeToMouseMotion(checkBox->state.getValue());
  }
};

//------------------------------------------------------------------------
class CheckProbeToDraggerAuditor : public SoDialogCheckBoxAuditor 
{
  MeshProbeViewer *mv;
public:
  CheckProbeToDraggerAuditor(MeshProbeViewer *mpv) { this->mv = mpv; }
  void dialogCheckBox(SoDialogCheckBox *checkBox) {
    mv->connectProbeToDraggerMotion(checkBox->state.getValue());
  }
};

//-------------------------------------------------------------------------
class FactorSliderAuditor : public SoDialogRealSliderAuditor 
{
  MeshProbeViewer *mpw;
public:
  FactorSliderAuditor(MeshProbeViewer *mpw) { this->mpw = mpw; }
  void dialogRealSlider(SoDialogRealSlider *slider) 
  {
    mpw->setFactorValue(slider->value.getValue());
  }  
};

//------------------------------------------------------------------------
void mouseMovedCallback(void *user_data, SoEventCallback *cb_node) 
{
    MeshProbeSheet *probeSheet =  (MeshProbeSheet *)user_data;
    const SoPickedPoint *picked_point = cb_node->getPickedPoint();
    size_t cellId = -1;
    double value = 0.0;
    std::string dataSetName;
    SbVec3f point;
    if (picked_point) 
    {
      point = picked_point->getPoint();
      const SoDetail* detail = picked_point->getDetail();
      const MoMeshDetail* mdetail = NULL;
      if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
      {
        MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;
        cellId = fdetail->getCellIndex();
        value = fdetail->getValue(point);
        mdetail = fdetail->getMeshDetail();
      }
      else if (detail->isOfType(MoLineDetailI::getClassTypeId()))
      {
        MoLineDetailI* ldetail = (MoLineDetailI*) detail;
        cellId = ldetail->getCellIndex();
        value = ldetail->getValue(point);
        mdetail = ldetail->getMeshDetail();
        const MeXSurfaceMeshUnstructured* inputMesh =  dynamic_cast<const MeXSurfaceMeshUnstructured*>(mdetail->getMesh());
        if (inputMesh != NULL)
        { // if the input mesh is an extracted surface, this means that
          // the picked mesh has been connected to a surface representation
          // and the cellId within the volume mesh must be retrieved
          cellId = inputMesh->getTopology().getInputCellIdI(cellId);
        }
      }
      if(mdetail && mdetail->getColorScalarSet())
        dataSetName = mdetail->getColorScalarSet()->getName();
    }
    probeSheet->setTextInfo(point,cellId,value,dataSetName);
    cb_node->setHandled();
}

//------------------------------------------------------------------------
MeshProbeViewer::MeshProbeViewer()
  : v_mouseMotion( false )
  , v_draggerMotion( false )
  , v_factorValue( 1 )
{
}

//------------------------------------------------------------------------
void MeshProbeViewer::buildSceneGraph(const MyMesh& myMesh, 
				      SoGroup *scene_probe,
				      SoSFInt32 *colorScalarSetId,
				      SoJackDragger *dragger,
				      SoGroup *probe_text_info_group
				      )
{
  v_myMesh = &myMesh;
  v_jackDragger = dragger;

  scene_probe->setName("Probe_Scene");

  // define a cell to draw, it is defined by the position of the probe
  v_cellShape = new MoMeshCellShape;
  v_cellShape->colorScalarSetId.connectFrom(colorScalarSetId);
  v_cellShape->cellIndices.setNum(0);
  v_cellShape->factor = v_factorValue;
  v_cellShape->offset = 0.5;
  v_cellShape->relativeOffset = TRUE;

  SoAnnotation *groupPickedPoint = new SoAnnotation;
  SoBaseColor  *base_color = new SoBaseColor ;
  base_color->rgb.setValue(1,1,1) ;
  SoCoordinate3* pickedPointCoord = new SoCoordinate3;
  SoMarkerSet* pickedPoint = new SoMarkerSet;
  // build probe information text callback
  v_probeSheet = new MeshProbeSheet(probe_text_info_group,v_cellShape,pickedPointCoord);

  v_pointProbe = new MoMeshPointProbe;
  v_pointProbe->scalarSetIds.connectFrom(colorScalarSetId);
  v_pointProbe->position.connectFrom(&v_jackDragger->translation);
  v_pointProbe->vectorSetIds.setNum(0);

  MoMaterial* cellMaterial = new MoMaterial;
  cellMaterial->transparency = 0.5;
  cellMaterial->faceColoring = MoMaterial::CONTOURING;
  cellMaterial->lineColoring = MoMaterial::COLOR;
  cellMaterial->lineColor.setValue(1,1,1);
  cellMaterial->pointColoring = MoMaterial::COLOR;
  cellMaterial->pointColor.setValue(1,1,1);

  v_cellShapeSwitch = new SoSwitch;
  v_cellShapeSwitch->whichChild = SO_SWITCH_NONE;

  v_cellDrawStyle = new MoDrawStyle;
  v_cellDrawStyle->displayFaces = FALSE;
  
  v_probeSwitch = new SoSwitch;
  v_probeSwitch->whichChild = SO_SWITCH_NONE;
  v_pickedPointSwitch = new SoSwitch;
  v_pickedPointSwitch->whichChild = SO_SWITCH_NONE;

  SoPickStyle *pick_style = new SoPickStyle;
  pick_style->style = SoPickStyle::UNPICKABLE;

  v_EventCallback = new SoEventCallback;
  
  // assemble the scene graph of this probe viewer
  {
    scene_probe->addChild(v_EventCallback);
    scene_probe->addChild(pick_style);
    scene_probe->addChild(v_probeSwitch);
    {
      v_probeSwitch->addChild(v_pointProbe);
    }
    scene_probe->addChild(v_cellShapeSwitch);
    {
      SoDrawStyle* ds = new SoDrawStyle;
      ds->pointSize = 5.f;
      v_cellShapeSwitch->addChild(ds);
      v_cellShapeSwitch->addChild(v_cellDrawStyle);  
      v_cellShapeSwitch->addChild(cellMaterial);
      v_cellShapeSwitch->addChild(v_cellShape);
    }
    scene_probe->addChild(v_pickedPointSwitch);
    {
      v_pickedPointSwitch->addChild(groupPickedPoint);
      {
          groupPickedPoint->addChild(base_color);
          groupPickedPoint->addChild(pickedPointCoord);
          groupPickedPoint->addChild(pickedPoint);
      }
    }
  }

}

//------------------------------------------------------------------------
SoDialogComponent *
MeshProbeViewer::buildDialogBox()
{
  SbString ivPath = SoPreferences::getString("OIVHOME",".") + "/examples/source/MeshVizXLM/demonstrators/MeshViewer/GuiTabProbe.iv";

  SoInput myInput;
  if (!myInput.openFile(ivPath.getString())) 
    exit (1);
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (myGroup == NULL) exit (1);
  v_DialogBox = (SoDialogGroup *)myGroup->getChild(0);

  //////////// 
  // mouse motion auditor
  SoDialogCheckBox* mouseMotionConnection = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("MouseMotion");
  mouseMotionConnection->addAuditor(new CheckProbeToMouseAuditor(this));
  mouseMotionConnection->addAuditor(new CheckSwitchAllAuditor(v_cellShapeSwitch));
  connectProbeToMouseMotion(mouseMotionConnection->state.getValue());

  //////////// 
  // dragger motion auditor
  SoDialogCheckBox* draggerMotionConnection = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("DraggerMotion");
  draggerMotionConnection->addAuditor(new CheckProbeToDraggerAuditor(this));
  draggerMotionConnection->addAuditor(new CheckSwitchAllAuditor(v_cellShapeSwitch));
  connectProbeToDraggerMotion(draggerMotionConnection->state.getValue());

  //////////// 
  // cell facets check
  SoDialogCheckBox* cellFacetsCheck = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("CellFacet");
  cellFacetsCheck->addAuditor(new CheckBoolAuditor(v_cellDrawStyle->displayFaces));
  cellFacetsCheck->state = (v_cellDrawStyle->displayFaces.getValue() == TRUE);

  //////////// 
  // cell edge check
  SoDialogCheckBox* cellEdgeCheck = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("CellEdge");
  cellEdgeCheck->addAuditor(new CheckBoolAuditor(v_cellDrawStyle->displayEdges));
  cellEdgeCheck->state = (v_cellDrawStyle->displayEdges.getValue() == TRUE);

  //////////// 
  // cell nodes check
  SoDialogCheckBox* cellNodesCheck = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("CellNodes");
  cellNodesCheck->addAuditor(new CheckBoolAuditor(v_cellShape->showNodeName));
  cellNodesCheck->addAuditor(new CheckBoolAuditor(v_cellDrawStyle->displayPoints));
  cellNodesCheck->applyDlgCptAuditor();

  ////////////
  // scaling factor
  SoDialogRealSlider* factorSlider = (SoDialogRealSlider *)v_DialogBox->searchForAuditorId("scalingFactorSlider");
  factorSlider->addAuditor(new FactorSliderAuditor(this));
  factorSlider->value = v_factorValue; 

  return v_DialogBox;
}

//---------------------------------------------------------------------------------------------------------------------
void MeshProbeViewer::setFactorValue(float factorValue) 
{
  v_factorValue = factorValue;
  v_cellShape->factor = v_factorValue;
}

//------------------------------------------------------------------------
void MeshProbeViewer::updateAllConnectedToDragger(SoJackDragger* /*dragger*/, SbVec3f /*plane_normal*/)
{
  // the connection is realized by the Inventor's field connexion
}

//------------------------------------------------------------------------
void MeshProbeViewer::connectProbeToMouseMotion(bool connect) 
{
  if (connect) 
  {
    SoDialogCheckBox* draggerMotionConnection = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("DraggerMotion");
    draggerMotionConnection->state = false;
    connectProbeToDraggerMotion(false);
    v_EventCallback->addEventCallback(SoLocation2Event::getClassTypeId(),
                                      mouseMovedCallback,
                                      v_probeSheet);
    v_probeSheet->setInMeshSwitch(v_pickedPointSwitch);
    v_probeSheet->switchOnText();
  }
  else 
  {
    v_EventCallback->removeEventCallback(SoLocation2Event::getClassTypeId(),
                                         mouseMovedCallback,
                                         v_probeSheet);
    v_probeSheet->setInMeshSwitch(NULL);
    v_probeSheet->switchOffText();
  }
  v_mouseMotion = connect;
  v_probeSwitch->whichChild = SO_SWITCH_NONE;
  
}

//------------------------------------------------------------------------
void MeshProbeViewer::connectProbeToDraggerMotion(bool connect) 
{
  if (connect)
  {
    SoDialogCheckBox* mouseMotionConnection = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("MouseMotion");
    mouseMotionConnection->state = false;
    connectProbeToMouseMotion(false);
    v_probeSheet->setInMeshSwitch(NULL);
    v_pickedPointSwitch->whichChild = SO_SWITCH_NONE;
    v_probeSheet->switchOnText();
    v_pointProbe->setProbeCallback(*v_probeSheet);
  }
  else
  {
    v_pointProbe->removeProbeCallback();
    v_probeSheet->switchOffText();
  }

  v_draggerMotion = connect;
  v_probeSwitch->whichChild = connect ? SO_SWITCH_ALL : SO_SWITCH_NONE;
}

//------------------------------------------------------------------------
void
MeshProbeViewer::preWriteAction() 
{
  enableConnection(v_cellShape,FALSE);
}

//------------------------------------------------------------------------
void 
MeshProbeViewer::postWriteAction() 
{
  enableConnection(v_cellShape,TRUE);
}

//------------------------------------------------------------------------
void MeshProbeViewer::enableConnection(SoNode *node, bool flag) 
{
  SoFieldList fields;
  int num_fields = node->getFields(fields);
  for (int i=0; i<num_fields; i++) {
    SoField *field = fields[i];
    if (field->isConnected()) field->enableConnection(flag);
  }
}

void
MeshProbeViewer::updateParallelMode()
{
  v_cellShape->parallel = getParallelMode();
}

