//------------------------------------------------------------------------
//                                                                        
// demonstration classes to visualize stream on a mesh with MeshViz XLM.
//                                                                        
//  author : J-Michel Godinaud                                            
//------------------------------------------------------------------------
#include "MeshStreamViewer.h"
#include "commonAuditor.h"
#include <Inventor/nodes/SoPointSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCoordinate3.h>

#define WIDTH_FACTOR 200
#define DEF_MAX_STEP_NUM 10000

class SourceShapeChoice : public SoDialogChoiceAuditor 
{
  MeshStreamViewer *mViewer;
public:
  SourceShapeChoice(MeshStreamViewer *mvw) { this->mViewer = mvw; }
  void dialogChoice (SoDialogChoice *dc) {
    mViewer->setSourceShape(dc->selectedItem.getValue());
  }
};

class SourceShapeSizeSlider : public SoDialogRealSliderAuditor 
{
  MeshStreamViewer *mViewer;
public:
  SourceShapeSizeSlider(MeshStreamViewer *mvw) { this->mViewer = mvw; }
  void dialogRealSlider (SoDialogRealSlider *rs) {
    mViewer->setSourceShapeSize(rs->value.getValue());
  }
};

class SourcestartingPointsSlider : public SoDialogIntegerSliderAuditor 
{
  MeshStreamViewer *mViewer;
public:
  SourcestartingPointsSlider(MeshStreamViewer *mvw) { this->mViewer = mvw; }
  void dialogIntegerSlider (SoDialogIntegerSlider *rs) {
    mViewer->setSourceStartingPoints(rs->value.getValue());
  }
};

class IlluminateAuditor : public SoDialogCheckBoxAuditor
{
  MeshStreamViewer *mViewer;
public:
  IlluminateAuditor(MeshStreamViewer *mvw) { this->mViewer = mvw; }

  void dialogCheckBox(SoDialogCheckBox *checkBox)
  {
    mViewer->enableLighting(checkBox->state.getValue());
  }
};

class AmbientOcclusionAuditor : public SoDialogCheckBoxAuditor
{
  MeshStreamViewer *mViewer;
public:
  AmbientOcclusionAuditor(MeshStreamViewer *mvw) { this->mViewer = mvw; }

  void dialogCheckBox(SoDialogCheckBox *checkBox)
  {
    mViewer->enableAmbientOcclusion(checkBox->state.getValue());
  }
};

MeshStreamViewer::MeshStreamViewer() :
  v_ViewFrame(0),
  v_numStartingPoints(7),
  v_SourceShape(SOURCE_CIRCLE),
  v_sourceSizeFactor(0.25)
{
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::buildSceneGraph(const MyMesh &myMesh, 
				       SoGroup *scene_stream,
				       SoSFInt32 *colorScalarSetId
               )
{
  scene_stream->setName("Streamlines_Scene");

  if (myMesh.vecSets.empty()) return;

  // define StreamLine
  v_MeshStreamLines = new MoMeshStreamline;
  v_MeshStreamLines->vec3SetId.setValue(0);
  v_MeshStreamLines->maxStepNumber = DEF_MAX_STEP_NUM;
  v_MeshStreamLines->minSpeed = 0.0009f;
  v_MeshStreamLines->colorScalarSetId.connectFrom(colorScalarSetId);
  //define the streamline material
  v_material = new MoMaterial;
  SoMaterial* streamMaterial = new SoMaterial;
  streamMaterial->specularColor.set1Value(0, 0.85f, 0.85f, 0.85f);
  // lighting for streamlines
  v_lightModel = new SoLightModel;
  v_lightModel->model = SoLightModel::PER_VERTEX_PHONG;

  // define stream sources representation
  SoMaterial *material = new SoMaterial;
  material->diffuseColor.set1Value(0, 1.,0.,0.);
  SoDrawStyle *draw_style = new SoDrawStyle;
  draw_style->pointSize = 4;
  SoCoordinate3 *source_coord = new SoCoordinate3;
  source_coord->point.connectFrom(&v_MeshStreamLines->startingPoints);
  SoPointSet *source_points = new SoPointSet;
  source_points->numPoints = -1; // use all points in source_coord
  v_StreamSource = new SoSeparator;
  v_StreamSource->addChild(material);
  v_StreamSource->addChild(draw_style);
  v_StreamSource->addChild(source_coord);
  v_StreamSource->addChild(source_points);

  v_StreamSourceSwitch = new SoSwitch;
  v_StreamSourceSwitch->addChild(v_StreamSource);
  v_StreamSourceSwitch->whichChild = SO_SWITCH_ALL;

  v_MeshStreamSwitch = new SoSwitch;
  v_MeshStreamSwitch->addChild(v_StreamSourceSwitch);
  v_MeshStreamSwitch->addChild(streamMaterial);
  v_MeshStreamSwitch->addChild(v_material);
  v_MeshStreamSwitch->addChild(v_lightModel);
  v_MeshStreamSwitch->addChild(v_MeshStreamLines);
  v_MeshStreamSwitch->whichChild = SO_SWITCH_NONE;
  
  SoPickStyle *pick_style = new SoPickStyle;
  pick_style->style = SoPickStyle::UNPICKABLE;

  // ambient occlusion for streamlines
  v_environment = new SoEnvironment;
  v_environment->ambientOcclusion = TRUE;

  scene_stream->addChild(pick_style);
  scene_stream->addChild(v_environment);
  //scene_stream->addChild(module_data_mapping);
  scene_stream->addChild(v_MeshStreamSwitch);

}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::setSourceShape(int selected)
{
  v_SourceShape = (MeshStreamViewer::SourceShapeType) selected;
  moveStreamSource();
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::setSourceShapeSize(float size)
{
  v_sourceSizeFactor = size;
  moveStreamSource();
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::setSourceStartingPoints(int num)
{
  v_numStartingPoints = num;
  moveStreamSource();
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::enableLighting(bool enable)
{
  if ( enable )
    v_lightModel->model = SoLightModel::PER_VERTEX_PHONG;
  else
    v_lightModel->model = SoLightModel::BASE_COLOR;
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::enableAmbientOcclusion(bool enable)
{
  v_environment->ambientOcclusion = enable;
}/*---------------------------------------------------------------------------*/

SoDialogComponent *
MeshStreamViewer::buildDialogBox ()
{
  SbString ivPath = SoPreferences::getString("OIVHOME",".") + "/examples/source/MeshVizXLM/demonstrators/MeshViewer/GuiTabStream.iv";

  SoInput myInput;
  if (!myInput.openFile(ivPath.getString())) 
    exit (1);
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (myGroup == NULL) exit (1);
  SoDialogGroup *v_DialogBox = (SoDialogGroup *)myGroup->getChild(0);

  //////////// 
  // StreamLines check
  SoDialogCheckBox* streamLinesCheck = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("StreamLinesEnable");
  streamLinesCheck->addAuditor(new CheckSwitchAllAuditor(v_MeshStreamSwitch));
  streamLinesCheck->state = (v_MeshStreamSwitch->whichChild.getValue() == SO_SWITCH_ALL);

  //////////// 
  // Source Shape Type check
  SoDialogChoice* sourceShapeTypeChoice = (SoDialogChoice *)v_DialogBox->searchForAuditorId("SourceShapeType");
  sourceShapeTypeChoice->addAuditor(new SourceShapeChoice(this));

  //////////// 
  // Source Shape Size Slider
  SoDialogRealSlider* sourceShapeSize = (SoDialogRealSlider *)v_DialogBox->searchForAuditorId("SourceShapeSize");
  sourceShapeSize->addAuditor(new SourceShapeSizeSlider(this));
  sourceShapeSize->value = v_sourceSizeFactor;

  //////////// 
  // Source Start Points Slider
  SoDialogIntegerSlider* sourceStart = (SoDialogIntegerSlider *)v_DialogBox->searchForAuditorId("SourceStartingPoints");
  sourceStart->addAuditor(new SourcestartingPointsSlider(this));
  sourceStart->applyDlgCptAuditor();

  // coloring check
  SoDialogComboBox* colorType = (SoDialogComboBox *)v_DialogBox->searchForAuditorId("ColoringType");
  colorType->addAuditor(new ColoringTypeAuditor(v_material->lineColoring));
  colorType->applyDlgCptAuditor();

  // illuminated streamlines
  SoDialogCheckBox* illuminate = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("IlluminateLines");
  illuminate->state = (v_lightModel->model.getValue() != SoLightModel::BASE_COLOR);
  illuminate->addAuditor(new IlluminateAuditor(this));
  illuminate->applyDlgCptAuditor();

  // Ambient occlusion
  SoDialogCheckBox* ambientOcclusion = (SoDialogCheckBox *)v_DialogBox->searchForAuditorId("AmbientOcclusion");
  ambientOcclusion->state = v_environment->ambientOcclusion;
  ambientOcclusion->addAuditor(new AmbientOcclusionAuditor(this));
  ambientOcclusion->applyDlgCptAuditor();

  return v_DialogBox;
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::updateAllConnectedToDragger(SoJackDragger *dragger, SbVec3f plane_normal)
{
  if (!v_IsDraggerConnected) return;

  // get the dragger position
  v_DraggerPos = dragger->translation.getValue();

  // get the dragger scale factor
  v_DraggerScale = dragger->scaleFactor.getValue();

  // rotate the plane's normal by the dragger rotation
  v_DraggerRotation = dragger->rotation.getValue();
  v_DraggerRotation.multVec(SbVec3f(0,1,0),plane_normal);

  // move stream lines, steam points & stream spheres
  moveStreamSource();
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::moveStreamSource()
{
  // move stream lines, steam points & stream spheres

  SbVec3f start_point;
  SbVec3d start_pointd;

  SbMatrix m;
  m.setTransform (v_DraggerPos, v_DraggerRotation, v_DraggerScale);
  v_MeshStreamLines->startingPoints.setNum(v_numStartingPoints);

  if (v_SourceShape == SOURCE_CIRCLE) {
    double alpha = 0;
    double d_alpha = 2*3.1415927/v_numStartingPoints;
    for (int i=0; i<v_numStartingPoints; i++, alpha += d_alpha) 
    {
      start_point.setValue (v_sourceSizeFactor * float(cos(alpha)), 0, v_sourceSizeFactor * float(sin(alpha)));
      m.multVecMatrix(start_point,start_point);
      start_pointd.setValue(start_point);
      v_MeshStreamLines->startingPoints.set1Value(i,start_pointd);
    } 
  } else {
    float xc,dx;
    xc = -v_sourceSizeFactor; dx = 2*v_sourceSizeFactor/float(v_numStartingPoints-1);
    for (int i=0; i<v_numStartingPoints; i++, xc += dx) 
    {
      start_point.setValue(xc,0,0);
      m.multVecMatrix(start_point,start_point);
      start_pointd.setValue(start_point);
      v_MeshStreamLines->startingPoints.set1Value(i,start_pointd);
    } 
  }

}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::preWriteAction() 
{
  enableConnection(v_MeshStreamLines,FALSE);
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::postWriteAction() 
{
  enableConnection(v_MeshStreamLines,TRUE);
}/*---------------------------------------------------------------------------*/

void MeshStreamViewer::enableConnection(SoNode *node, SbBool flag) 
{
  SoFieldList fields;
  int num_fields = node->getFields(fields);
  for (int i=0; i<num_fields; i++) 
  {
    SoField *field = fields[i];
    if (field->isConnected()) field->enableConnection(flag);
  }
}/*---------------------------------------------------------------------------*/

float
MeshStreamViewer::getMaxVec(const MyMesh& /*myMesh*/) const
{
  return 0;
}

void
MeshStreamViewer::updateParallelMode()
{
  v_MeshStreamLines->parallel = getParallelMode();
}

//---------------------------------------------------------------------------------------------------------------------
void 
MeshStreamViewer::enableProgressCallback(bool enable)
{
  if (enable)
    v_MeshStreamLines->setExtractorCallback(&v_progressCallback);
  else
    v_MeshStreamLines->setExtractorCallback(NULL);
}
