//------------------------------------------------------------------------
//                                                                        
// abstract demonstration class to visualize mesh with MeshViz
//                                                                        
//  author : J-Michel Godinaud                                            
//------------------------------------------------------------------------

#include "MeshViewer.h"
#include "commonAuditor.h"

#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>

#include <MeshVizXLM/mapping/nodes/MoTessellator.h>
#include <MeshVizXLM/tessellator/MiTessellator.h>

#include <Inventor/nodes/SoPolygonOffset.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/SoPath.h>

using std::vector;
using std::string;

//---------------------------------------------------------------------------------------------------------------------
class TessellateSliderAuditor : public SoDialogRealSliderAuditor 
{
public:
  TessellateSliderAuditor( MeshViewer* meshViewer) : m_meshViewer(meshViewer) {}
  void dialogRealSlider(SoDialogRealSlider *slider) 
  {
    m_meshViewer->updateTessellationError(double(slider->value.getValue()));
  }
private:
  MeshViewer* m_meshViewer;
};

//---------------------------------------------------------------------------------------------------------------------
class CheckProgressAuditor : public SoDialogCheckBoxAuditor 
{
public:
  CheckProgressAuditor(MeshViewerComponent *viewer)
    : v_viewer(viewer)
  {
  }
  void dialogCheckBox(SoDialogCheckBox *checkBox) 
  {
    v_viewer->enableProgressCallback(checkBox->state.getValue() == TRUE);
  }
private:
  MeshViewerComponent *v_viewer;
};


//---------------------------------------------------------------------------------------------------------------------
MeshViewer::MeshViewer() :
  v_TabWindow(NULL),
  v_Root(NULL),
  v_DraggerSwitch(NULL),
  v_userTessellator(NULL),
  v_edgeErrorMetricGeometry(NULL),
  v_DataSetIndex(0),
  v_ScaleFactor(1),
  v_NumIsoValues(16),
  v_MaxThresholdIsEnable(FALSE),
  v_Mesh3DScalarViewer(NULL),
  v_Mesh3DVecViewer(NULL),
  v_MeshStreamViewer(NULL),
  v_MeshProbeViewer(NULL),
  v_MeshFilterViewer(NULL),
  v_DraggerNormal(0,0,1),
  v_DraggerPos(0,0,0),
  v_parallel(true)
{
  // Initialize Open-Inventor classes
  v_BaseWidget = SoXt::init(v_WindowTitle) ;
  if (v_BaseWidget == NULL) exit(1) ;

  MoMeshViz::init();

  v_basicTessellator = MiTessellator::getNewTessellatorBasic();

  // In order to keep the compatibility with MeshViz 5.0
  //SoPreferences::setFloat("OIV_3DDATA_CREASE_ANGLE",0.8f);

  v_Dragger = new SoJackDragger;
  strcpy(v_WindowTitle,"mesh_viewer");
  strcpy(v_CourtesyTitle," ");
  strcpy(v_IvAsciiFileName,"mesh.iv");
  strcpy(v_IvBinaryFileName,"mesh_bin.iv");
}/*---------------------------------------------------------------------------*/

MeshViewer::~MeshViewer() {
  if(v_Root)
    v_Root->unref();
  if (v_TabWindow)
    v_TabWindow->close();
  delete v_basicTessellator;
  delete v_Viewer;
  delete v_Mesh3DScalarViewer;
  delete v_MeshStreamViewer;
  delete v_Mesh3DVecViewer;
  delete v_MeshProbeViewer;
  delete v_MeshFilterViewer;
  SoDialogViz::finish();
  MoMeshViz::finish();
  SoXt::finish();
}/*---------------------------------------------------------------------------*/

void MeshViewer::setDraggerNormal(SbVec3f normal) {
  v_DraggerNormal = normal;
  setDraggerPos(v_DraggerPos);
}/*---------------------------------------------------------------------------*/

void MeshViewer::setIvFileName(const char *filename) {
  if (filename) strcpy(v_IvAsciiFileName,filename);
}/*---------------------------------------------------------------------------*/

void MeshViewer::setWindowTitle(const char *window_title) {
  if (window_title) {
    strcpy(v_WindowTitle,window_title);
    strcat(strcpy(v_IvAsciiFileName,window_title),".iv");
  }
  
}/*---------------------------------------------------------------------------*/

void MeshViewer::setCourtesyTitle(const char *courtesy_title) {
  if (courtesy_title) strcpy(v_CourtesyTitle,courtesy_title);
}/*---------------------------------------------------------------------------*/

void MeshViewer::setNumIsoValues(int num_iso) {
  if (num_iso>0) v_NumIsoValues = num_iso;
}/*---------------------------------------------------------------------------*/

void MeshViewer::setDataSetIndex(int index) {
  v_DataSetIndex = index;
}/*---------------------------------------------------------------------------*/

void MeshViewer::enableMaxThreshold(SbBool is_enable) {
  v_MaxThresholdIsEnable = is_enable;
}/*---------------------------------------------------------------------------*/

void MeshViewer::buildDataMappings(const MyMesh &myMesh) 
{
  // define the data-mapping associated to each set of data-set
  v_colorMappingSwitch = new SoSwitch;
  v_colorMappingSwitch->whichChild = v_DataSetIndex;

  for (size_t i=0; i<myMesh.scalarSets.size(); ++i) 
  {
    // define the data-mapping associated to i-th data-set
    v_colorMapping = new MoPredefinedColorMapping;
    v_colorMapping->minValue = myMesh.getScalarMin(i); 
    v_colorMapping->maxValue = myMesh.getScalarMax(i); 
    v_colorMapping->predefColorMap = MoPredefinedColorMapping::STANDARD;
    v_colorMappingSwitch->addChild(v_colorMapping);
  }     

}/*---------------------------------------------------------------------------*/

void MeshViewer::buildScene2D(SoGroup *probe_text_info_group)
{
  // build courtesy title
  SoFont *font = new SoFont;
  font->size = 12;
  SoTranslation *courtesy_trans = new SoTranslation;
  courtesy_trans->translation.setValue(0.95F, -0.95F, -1);
  SoText2 *courtesy_title = new SoText2;
  courtesy_title->string = v_CourtesyTitle;
  courtesy_title->justification = SoText2::RIGHT;

  // pick style node
  SoPickStyle *pick_style = new SoPickStyle;
  pick_style->style = SoPickStyle::UNPICKABLE;

  // Text property for the 2D scene
  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;
  annoText3Property->fontSizeHint = SoAnnoText3Property::ANNOTATION ;

  v_Scene2D = new SoAnnotation ;
  v_Scene2D->addChild(pick_style);
  v_Scene2D->addChild(annoText3Property);
  v_Scene2D->addChild(font);
  v_Scene2D->addChild(courtesy_trans);
  v_Scene2D->addChild(courtesy_title);
  v_Scene2D->addChild(probe_text_info_group);
}/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
void MeshViewer::show(const MiVolumeMeshUnstructured& mesh, 
                      vector<const MiScalardSetI*>& scalarSets,
                      vector<const MiVec3dSetI*>& vecSets,
                      MiTessellator* tessellator,
                      MxEdgeErrorMetricGeometry* edgeErrorMetricGeometry
                      ) 
{
  MyMesh myMesh(mesh,scalarSets,vecSets,tessellator);

  v_userTessellator = tessellator;
  v_edgeErrorMetricGeometry = edgeErrorMetricGeometry;
  buildMeshGroup(myMesh);

  // build the dialog box
  buildMainWindow();
  buildTabWindow(myMesh);
  
  addAuditors(myMesh);
  startShow(v_BaseWidget);
}

/*---------------------------------------------------------------------------*/
void
MeshViewer::buildMeshGroup(const MyMesh& myMesh) 
{
  // build the mesh node group
  v_meshGroup = new SoGroup;

  MoMesh* v_meshNode = new MoMesh;
  v_meshNode->setMesh(&myMesh.mesh);
  v_meshGroup->addChild(v_meshNode);

  for (size_t i=0; i<myMesh.scalarSets.size(); ++i)
  {
     MoScalarSetI* scalarSetNode = new MoScalarSetI;
     scalarSetNode->setScalarSet(myMesh.scalarSets[i]);
     v_meshGroup->addChild(scalarSetNode);
  }

  for (size_t i=0; i<myMesh.vecSets.size(); ++i)
  {
    MoVec3SetI* vecSetNode = new MoVec3SetI;
    vecSetNode->setVec3Set(myMesh.vecSets[i]);
    v_meshGroup->addChild(vecSetNode);
  }

  MoTessellator* basicTessellatorNode = new MoTessellator;
  basicTessellatorNode->setTessellator(v_basicTessellator);
  v_userTessellatorNode = new MoTessellator;
  v_userTessellatorNode->setTessellator(v_userTessellator);
  v_tessellatorSwitch = new SoSwitch;
  if (v_userTessellator)
  {
    v_tessellatorSwitch->addChild(basicTessellatorNode);
    v_tessellatorSwitch->addChild(v_userTessellatorNode);
  }
  else
  {
    v_tessellatorSwitch->addChild(v_userTessellatorNode);
    v_tessellatorSwitch->addChild(basicTessellatorNode);
  }
  v_tessellatorSwitch->whichChild = v_userTessellator ? 1 : 0;
  v_meshGroup->addChild(v_tessellatorSwitch);
}

/*---------------------------------------------------------------------------*/
void 
MeshViewer::closeTabWindow()
{
  //if (v_TabWindow) v_TabWindow->removeAllChildren();
  if (v_TabWindow) v_TabWindow->close();
}

/*---------------------------------------------------------------------------*/
void 
MeshViewer::buildTabWindow(const MyMesh &myMesh)
{
  v_TabWindow = new SoTopLevelDialog();
  v_TabWindow->label = "Menu";

  v_TabDialog = new SoTabDialog();
  v_TabWindow->addChild(v_TabDialog);
  
  buildDataMappings(myMesh);

  // get the mesh bounding box
  v_BoundingBox = myMesh.getBoundingBoxf();

  float sx,sy,sz; 
  v_BoundingBox.getSize(sx,sy,sz);
  float maxs = (sx > sy) ? sx : sy; 
  if (sz > maxs) maxs = sz;

  v_ScaleFactor = maxs/10;

  // build the dragger
  v_Dragger->addMotionCallback((SoDraggerCB *)&MeshViewer::motionCallback,this);

  SoSeparator* draggerSep = new SoSeparator;

  // Make sure the dragger is not impacted by any previous ShapeHints
  draggerSep->addChild(new SoShapeHints);
  draggerSep->addChild(v_Dragger);

  v_DraggerSwitch = new SoSwitch;
  v_DraggerSwitch->addChild(draggerSep) ;
  v_DraggerSwitch->whichChild = SO_SWITCH_ALL;
   
  // build the scene to visualize scalar data 
  v_SceneScalar = new SoSeparator;

  v_SceneVec = new SoSeparator;

  // Mesh3DScalarViewer
  v_Mesh3DScalarViewer = new Mesh3DScalarViewer(myMesh,v_TabWindow,v_Viewer);
  ((SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchScalar"))->enable = TRUE;
  v_Mesh3DScalarViewer->buildSceneGraph(v_SceneScalar, &v_colorMappingSwitch->whichChild,v_ParentViewer->getWidget(),v_meshGroup);
  v_DialogBoxScalar = v_Mesh3DScalarViewer->buildDialogBox();
  v_TabDialog->addChild(v_DialogBoxScalar);
  if (v_userTessellator != NULL && v_edgeErrorMetricGeometry != NULL)
  {
    // Bug: AuditorID cannot be found.
    //SoRowDialog* isosurfaceDialog = (SoRowDialog *)v_DialogBoxScalar->searchForAuditorId("IsosurfaceDialog");
    SoRowDialog* isosurfaceDialog = (SoRowDialog *)v_DialogBoxScalar->getChild(1);
    isosurfaceDialog->enable = false;
  }

  // Mesh3DVecViewer
  if (!myMesh.vecSets.empty()) 
  {
    v_Mesh3DVecViewer = new Mesh3DVecViewer;
    ((SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchVectors"))->enable = TRUE;
    v_Mesh3DVecViewer->buildSceneGraph(myMesh,v_SceneVec,&v_colorMappingSwitch->whichChild);
    v_DialogBoxVec = v_Mesh3DVecViewer->buildDialogBox();
    v_TabDialog->addChild(v_DialogBoxVec);
  }

  // MeshStreamViewer
  v_SceneStream = new SoSeparator;

  if (!myMesh.vecSets.empty()) 
  {
    v_MeshStreamViewer = new MeshStreamViewer;
    ((SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchStream"))->enable = TRUE;
    v_MeshStreamViewer->buildSceneGraph(myMesh,v_SceneStream,&v_colorMappingSwitch->whichChild);
    v_DialogBoxStream = v_MeshStreamViewer->buildDialogBox();
    v_TabDialog->addChild(v_DialogBoxStream);
  }

  // MeshProbeViewer
  v_SceneProbe = new SoSeparator;
  v_SceneProbe->boundingBoxCaching = SoSeparator::OFF;

  SoGroup *probe_text_info = new SoGroup;

  v_MeshProbeViewer = new MeshProbeViewer;
  ((SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchProbe"))->enable = TRUE;
  v_MeshProbeViewer->buildSceneGraph(myMesh,
				     v_SceneProbe,
				     &v_colorMappingSwitch->whichChild,
				     v_Dragger,
             probe_text_info);
  v_DialogBoxProbe = v_MeshProbeViewer->buildDialogBox();
  v_TabDialog->addChild(v_DialogBoxProbe);

  // MeshFilterViewer
  v_SceneFilter = new SoGroup;
  v_MeshFilterViewer = new MeshFilterViewer;
  ((SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchFilter"))->enable = TRUE;
  v_MeshFilterViewer->buildSceneGraph(myMesh, v_SceneFilter);
  v_DialogBoxFilter = v_MeshFilterViewer->buildDialogBox(myMesh);
  v_TabDialog->addChild(v_DialogBoxFilter);
  
  
  SbString ivPath = SoPreferences::getString("OIVHOME",".") + "/examples/source/MeshVizXLM/demonstrators/MeshViewer/GuiTabGlobal.iv";
  SoInput myInput;
  if (!myInput.openFile(ivPath.getString())) 
    exit (1);
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (myGroup == NULL) exit (1);
  v_DialogBoxGlobal = (SoDialogGroup *)myGroup->getChild(0);
  v_TabDialog->addChild(v_DialogBoxGlobal);


  // Build the tab panel dialog
  v_TabWindow->buildDialog(v_BaseWidget,FALSE);
  
  // move the dragger at its initial position
  setDraggerPos(v_BoundingBox.getCenter());

  // build any shape displayed on the screen space, as the courtesy title.
  buildScene2D(probe_text_info);

  SoPerspectiveCamera *camera3D = new SoPerspectiveCamera ;

  SoOrthographicCamera *camera2D = new SoOrthographicCamera ;
  camera2D->viewportMapping = SoCamera::LEAVE_ALONE;

  // necessary to activate 2 sided lighting
  SoShapeHints* shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering.setValue(SoShapeHints::CLOCKWISE );
  shapeHints->creaseAngle = 0.8f;

  v_Scene3D = new SoSeparator ;
  v_Scene3D->setName("Scene3D");
  v_Scene3D->addChild(camera3D);

  v_Scene3D->addChild(v_meshGroup);
  v_Scene3D->addChild(v_colorMappingSwitch);
  v_Scene3D->addChild(v_DraggerSwitch);
  v_Scene3D->addChild(new SoPolygonOffset);
  v_Scene3D->addChild(v_SceneFilter);
  v_Scene3D->addChild(v_SceneScalar);
  v_Scene3D->addChild(v_SceneProbe);
  v_Scene3D->addChild(v_SceneStream);
  v_Scene3D->addChild(v_SceneVec);

  // Text property for the 2D scene
  SoAnnoText3Property *annoText3Property = new SoAnnoText3Property ;
  annoText3Property->renderPrintType = SoAnnoText3Property::RENDER2D_PRINT_RASTER ;
  annoText3Property->fontSizeHint = SoAnnoText3Property::ANNOTATION ;

  v_Root = new SoSeparator ;
  v_Root->ref() ;
  v_Root->addChild(shapeHints);
  v_Root->addChild(v_Scene3D);
  v_Root->addChild(camera2D);
  v_Root->addChild(annoText3Property);
  v_Root->addChild(v_Scene2D);

}/*---------------------------------------------------------------------------*/

void MeshViewer::startShow(Widget baseWidget)
{
  v_Viewer->setSceneGraph(v_Root);
  v_Viewer->setTitle(v_WindowTitle);

  if (v_Root) v_Viewer->viewAll();
  v_Viewer->show();

  v_MainWindow->show();
  if (v_TabWindow) v_TabWindow->show();

  SoXt::show(baseWidget);
  SoXt::mainLoop();
}/*---------------------------------------------------------------------------*/


void MeshViewer::updateTessellationError(double error)
{
  if (v_edgeErrorMetricGeometry)
  {
    v_edgeErrorMetricGeometry->setMaxError(error);
    v_userTessellatorNode->touch();
  }
}/*---------------------------------------------------------------------------*/

PoXtExaminerViewer::PoXtExaminerViewer(Widget parent, const char *name,
				       SbBool buildInsideParent, 
				       SoXtFullViewer::BuildFlag flag,
				       SoXtViewer::Type typeViewer) :
  
  SoXtExaminerViewer(parent,
                     name, buildInsideParent, flag,
		     typeViewer)
{
}/*---------------------------------------------------------------------------*/

void
PoXtExaminerViewer::viewAll()
{
  SoCamera *my_camera = getCamera();
  my_camera->viewAll(SoNode::getByName("Scene3D"), getViewportRegion()) ;
}/*---------------------------------------------------------------------------*/

class EnumFieldAuditor : public SoMenuRadioButtonsAuditor {
  SoSFEnum *field;
public:
  EnumFieldAuditor(SoSFEnum *myField) { this->field = myField; }
  void  menuRadioButtons(SoMenuRadioButtons *rb) {
    field->setValue(rb->selectedItem.getValue());
  }
};

class IntSwitchAuditor : public SoMenuRadioButtonsAuditor {
  SoSwitch *sw;
public:
  IntSwitchAuditor(SoSwitch *ssw) { this->sw = ssw; }
  void  menuRadioButtons(SoMenuRadioButtons *rb) {
    sw->whichChild = rb->selectedItem.getValue();
  }
};

class CheckMenuSwitchAuditor : public SoMenuCheckBoxAuditor {
  SoSwitch *sw;
public:
  CheckMenuSwitchAuditor(SoSwitch *ssw) { this->sw = ssw; }
  void menuCheckBox(SoMenuCheckBox *checkBox) {
    sw->whichChild = (checkBox->state.getValue() == TRUE)
      ? SO_SWITCH_ALL 
      : SO_SWITCH_NONE;
  }
};

class OrientAuditor : public SoMenuRadioButtonsAuditor {
  MeshViewer *mv;
public:
  OrientAuditor(MeshViewer *mvw) { this->mv = mvw; }
  void  menuRadioButtons(SoMenuRadioButtons *rb) {
    int choice_val = rb->selectedItem.getValue();
    switch (choice_val) {
    case 0 : mv->setDraggerNormal(SbVec3f(1,0,0)); break;
    case 1 : mv->setDraggerNormal(SbVec3f(0,1,0)); break;
    case 2 : mv->setDraggerNormal(SbVec3f(0,0,1)); break;
    }
  }
};

class SelectTabWindowAuditor : public SoMenuPushButtonAuditor {
  MeshViewer *mv;
public:
  SelectTabWindowAuditor(MeshViewer *mvw) { this->mv = mvw; }
  void menuPushButton(SoMenuPushButton* pb)
  {
    mv->setTabWindowSelectedPage(pb->auditorID.getValue());
  }
};

class MergeWindowsCheckAuditor : public SoMenuCheckBoxAuditor {
  MeshViewer *mv;
public:
  MergeWindowsCheckAuditor(MeshViewer *mvw) { this->mv = mvw; }
  void menuCheckBox(SoMenuCheckBox *cb) {
    mv->setWindowsDisposition(cb->state.getValue());
  }
};

class ExitAuditor : public SoMenuPushButtonAuditor {
  MeshViewer *mv;
public:
  ExitAuditor(MeshViewer *mv) {
    this->mv = mv;
  }
  void menuPushButton(SoMenuPushButton* /*pb*/) {
    delete mv;
    exit(0);
  }
};

class ParallelModeAuditor : public SoMenuCheckBoxAuditor {
  MeshViewer *mv;
public:
  ParallelModeAuditor(MeshViewer* meshViewer) : mv(meshViewer) {}
  void menuCheckBox(SoMenuCheckBox *checkBox) {
    mv->setParallelMode((checkBox->state.getValue() == TRUE));
  }
};


void
MeshViewer::setWindowsDisposition(SbBool state)
{
  if (state)
  {
  }
  else
  {
  }
}

void
MeshViewer::setTabWindowSelectedPage(SbString id)
{
  int sel = 0;
  v_TabWindow->show();
  SoDialogGroup* group = NULL;
  if (id == "launchStream")
    group = (SoDialogGroup*)v_TabDialog->searchForAuditorId("StreamTab");
  else if (id == "launchProbe")
    group = (SoDialogGroup*)v_TabDialog->searchForAuditorId("ProbeTab");
  else if (id == "launchVectors")
    group = (SoDialogGroup*)v_TabDialog->searchForAuditorId("VectorTab");
  else if (id == "launchScalar")
    group = (SoDialogGroup*)v_TabDialog->searchForAuditorId("ScalarTab");
  else if (id == "launchFilter")
    group = (SoDialogGroup*)v_TabDialog->searchForAuditorId("FilterTab");

  if (group)
    sel = v_TabDialog->findChild(group);

  v_TabDialog->selectedPage.setValue(sel);
}

/*---------------------------------------------------------------------------*/
void 
MeshViewer::buildMainWindow()
{
  SoDialogViz::init();

  v_MainWindow = new SoTopLevelDialog;
  v_MainWindow->position.setValue(300,0);
  v_MainWindow->label = "Mesh Viewer";
  
  SbString ivPath = SoPreferences::getString("OIVHOME",".") + "/examples/source/MeshVizXLM/demonstrators/MeshViewer/GuiMenuBar.iv";
  v_MenuBar = (SoMenuBar *)SoDialogViz::loadFromFile(ivPath);

  v_ParentViewer = new SoDialogCustom();
  v_ParentViewer->width = 1024;
  v_ParentViewer->height = 768;

  v_MainWindow->addChild(v_MenuBar);
  v_MainWindow->addChild(v_ParentViewer);

  v_MainWindow->buildDialog(v_BaseWidget,TRUE);

  v_Viewer = new PoXtExaminerViewer(v_ParentViewer->getWidget());

}

/*---------------------------------------------------------------------------*/
void
MeshViewer::enableMenuBar(SbBool state)
{
  int num_child = v_MenuBar->getNumChildren();
  int i;
  for (i=1; i<num_child; i++) {
    SoMenuItem *item = (SoMenuItem *)v_MenuBar->getChild(i);
    item->enable = state;
  }
}

/*---------------------------------------------------------------------------*/
void MeshViewer::updateAuditors(const MyMesh &myMesh)
{
  SoMenuCheckBox* draggerCheck = (SoMenuCheckBox *)v_MainWindow->searchForAuditorId("Dragger");
  if (v_DraggerSwitch != NULL) {
    draggerCheck->addAuditor(new CheckMenuSwitchAuditor(v_DraggerSwitch));
    draggerCheck->state = (v_DraggerSwitch->whichChild.getValue() == SO_SWITCH_ALL) ? TRUE : FALSE;
  }

  SoMenuRadioButtons* dataSetChoice = (SoMenuRadioButtons *)v_MainWindow->searchForAuditorId("DataSetUsed");
  dataSetChoice->addAuditor(new IntSwitchAuditor(v_colorMappingSwitch));

  dataSetChoice->items.setNum(int(myMesh.scalarSets.size()));
  SbString scalarSetName;
  for (size_t i=0; i<myMesh.scalarSets.size(); ++i)
  {
    scalarSetName = myMesh.scalarSets[i]->getName().c_str();
    if (scalarSetName.getLength() == 0)
    {
      scalarSetName = "Dataset ";
      scalarSetName += int(i);
    }
    if (  myMesh.scalarSets[i]->getBinding() == MiDataSet::PER_NODE)
      scalarSetName += " PER_NODE";
    else
      scalarSetName += " PER_CELL";
    dataSetChoice->items.set1Value(int(i), scalarSetName);
  }
  dataSetChoice->selectedItem = v_colorMappingSwitch->whichChild.getValue();
}

/*---------------------------------------------------------------------------*/
void MeshViewer::addAuditors(const MyMesh &myMesh) 
{
  SoMenuPushButton* exitButton = (SoMenuPushButton*)v_MainWindow->searchForAuditorId("Exit");
  exitButton->addAuditor(new ExitAuditor(this));

  SoMenuCheckBox* parallelModeButton = (SoMenuCheckBox*)v_MainWindow->searchForAuditorId("ParallelMode");
  parallelModeButton->addAuditor(new ParallelModeAuditor(this));

  SoMenuRadioButtons* orientChoice = (SoMenuRadioButtons *)v_MainWindow->searchForAuditorId("Dragger Orientation");
  orientChoice->addAuditor(new OrientAuditor(this));

  SelectTabWindowAuditor* mySelectTabAuditor = new SelectTabWindowAuditor(this);
  SoMenuPushButton* tabScalarSelection = (SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchScalar");
  tabScalarSelection->addAuditor(mySelectTabAuditor);
  SoMenuPushButton* tabStreamSelection = (SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchStream");
  tabStreamSelection->addAuditor(mySelectTabAuditor);
  SoMenuPushButton* tabVectorsSelection = (SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchVectors");
  tabVectorsSelection->addAuditor(mySelectTabAuditor);
  SoMenuPushButton* tabProbeSelection = (SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchProbe");
  tabProbeSelection->addAuditor(mySelectTabAuditor);
  SoMenuPushButton* tabFilterSelection = (SoMenuPushButton*)v_MainWindow->searchForAuditorId("launchFilter");
  tabFilterSelection->addAuditor(mySelectTabAuditor);

  SoMenuCheckBox* mergeWindowsCheck = (SoMenuCheckBox *)v_MainWindow->searchForAuditorId("MergeWindowsCheck");
  mergeWindowsCheck->addAuditor(new MergeWindowsCheckAuditor(this));

  // set the initial state of check box, choice etc.
  if (v_DraggerNormal == SbVec3f(1,0,0)) 
    orientChoice->selectedItem = 0;
  else if (v_DraggerNormal == SbVec3f(0,1,0)) 
    orientChoice->selectedItem = 1;
  else
    orientChoice->selectedItem = 2;

  if (v_DraggerSwitch != NULL)
    updateAuditors(myMesh);

  // tessellator enable
  SoDialogCheckBox* tessellatorCheck = (SoDialogCheckBox *)v_DialogBoxGlobal->searchForAuditorId("TessellatorCheck");

  // tessellator level value
  SoDialogRealSlider* tessellateSlider = (SoDialogRealSlider *)v_DialogBoxGlobal->searchForAuditorId("TessellateLevel");
  tessellateSlider->addAuditor(new TessellateSliderAuditor(this));
  tessellatorCheck->addAuditor(new CheckSwitchAuditor(v_tessellatorSwitch));
  tessellatorCheck->enable = TRUE;
  tessellatorCheck->state =  v_userTessellator == NULL ? FALSE : TRUE;    
  if (v_edgeErrorMetricGeometry == NULL)
    tessellateSlider->enable = FALSE;
  tessellateSlider->applyDlgCptAuditor();
  tessellatorCheck->applyDlgCptAuditor();

  SoDialogCheckBox* progressCheck = (SoDialogCheckBox *)v_DialogBoxGlobal->searchForAuditorId("ProgressCheck");
  progressCheck->addAuditor(new CheckProgressAuditor(v_Mesh3DScalarViewer));
  if (v_MeshStreamViewer) progressCheck->addAuditor(new CheckProgressAuditor(v_MeshStreamViewer));
  progressCheck->state = FALSE;
  progressCheck->applyDlgCptAuditor();

}/*---------------------------------------------------------------------------*/

void MeshViewer::setDraggerPos (const SbVec3f& pos) {
  // set the dragger position
  v_DraggerPos = pos;
  v_Dragger->translation.setValue(pos);
  v_Dragger->rotation = SbRotation(SbVec3f(0,1,0),v_DraggerNormal);
  v_Dragger->scaleFactor.setValue(v_ScaleFactor,v_ScaleFactor,v_ScaleFactor);

  //  updateAllConnectedToDragger(v_Dragger);
  if (v_Mesh3DScalarViewer) v_Mesh3DScalarViewer->updateAllConnectedToDragger(v_Dragger,v_DraggerNormal);
  if (v_MeshProbeViewer) v_MeshProbeViewer->updateAllConnectedToDragger(v_Dragger,v_DraggerNormal);
  if (v_MeshStreamViewer) v_MeshStreamViewer->updateAllConnectedToDragger(v_Dragger,v_DraggerNormal);
  if (v_Mesh3DVecViewer) v_Mesh3DVecViewer->updateAllConnectedToDragger(v_Dragger,v_DraggerNormal);
}/*---------------------------------------------------------------------------*/


void MeshViewer::motionCallback(void *user_data, SoJackDragger *dragger) {
  MeshViewer *viewer = (MeshViewer *)user_data;
  viewer->v_DraggerPos = dragger->translation.getValue();
  if (viewer->v_Mesh3DScalarViewer) viewer->v_Mesh3DScalarViewer->updateAllConnectedToDragger(dragger,viewer->v_DraggerNormal);
  if (viewer->v_MeshProbeViewer) viewer->v_MeshProbeViewer->updateAllConnectedToDragger(dragger,viewer->v_DraggerNormal);
  if (viewer->v_MeshStreamViewer) viewer->v_MeshStreamViewer->updateAllConnectedToDragger(dragger,viewer->v_DraggerNormal);
  if (viewer->v_Mesh3DVecViewer) viewer->v_Mesh3DVecViewer->updateAllConnectedToDragger(dragger,viewer->v_DraggerNormal);
}/*---------------------------------------------------------------------------*/

void MeshViewer::preWriteAction() {
  if (v_Mesh3DScalarViewer) v_Mesh3DScalarViewer->preWriteAction();
  if (v_MeshProbeViewer) v_MeshProbeViewer->preWriteAction();
  if (v_MeshStreamViewer) v_MeshStreamViewer->preWriteAction();
  if (v_Mesh3DVecViewer) v_Mesh3DVecViewer->preWriteAction();
}/*---------------------------------------------------------------------------*/

void MeshViewer::postWriteAction() {
  // re-enable the connection of the fields of the viewers. See previous comments.
  if (v_Mesh3DScalarViewer) v_Mesh3DScalarViewer->postWriteAction();
  if (v_MeshProbeViewer) v_MeshProbeViewer->postWriteAction();
  if (v_MeshStreamViewer) v_MeshStreamViewer->postWriteAction();
  if (v_Mesh3DVecViewer) v_Mesh3DVecViewer->postWriteAction();
}/*---------------------------------------------------------------------------*/

void MeshViewer::setParallelMode(bool mode) 
{  
  v_parallel = mode;
  if (v_Mesh3DScalarViewer) v_Mesh3DScalarViewer->setParallelMode(mode);
  if (v_Mesh3DVecViewer) v_Mesh3DVecViewer->setParallelMode(mode);
  if (v_MeshStreamViewer) v_MeshStreamViewer->setParallelMode(mode);
  if (v_MeshProbeViewer) v_MeshProbeViewer->setParallelMode(mode);
  if (v_MeshFilterViewer) v_MeshFilterViewer->setParallelMode(mode);
}


