#ifndef _MyMesh_h
#define _MyMesh_h

#include <vector>
#include <ostream>

#include <MeshVizXLM/mesh/MiMesh.h>
#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>
#include <MeshVizXLM/mesh/data/MiDataSetI.h>

class MiTessellator;

/**
* Defines a Mesh and its data sets.
* Internal usage for the mesh viewer
*/
class MyMesh
{
public:
  MyMesh(const MiVolumeMeshUnstructured& _mesh, 
    std::vector<const MiScalardSetI*>& _scalarSets,
    std::vector<const MiVec3dSetI*>& _vecSets,
    MiTessellator* _tesselator);

  const MiVolumeMeshUnstructured& mesh;
  std::vector<const MiScalardSetI*> scalarSets;
  std::vector<const MiVec3dSetI*> vecSets;
  MiTessellator* tessellator;

  SbBox3f getBoundingBoxf() const;

  float getScalarMin(size_t dataId) const
  {
    return m_minScalarSet[dataId]; 
  }

  float getScalarMax(size_t dataId) const
  {
    return m_maxScalarSet[dataId];
  }

private:
  SbBox3f m_bbox;
  std::vector<float> m_minScalarSet;
  std::vector<float> m_maxScalarSet;
};

//-----------------------------------------------------------------------------
inline 
MyMesh::MyMesh(const MiVolumeMeshUnstructured& _mesh, 
               std::vector<const MiScalardSetI*>& _scalarSets,
               std::vector<const MiVec3dSetI*>& _vecSets,
               MiTessellator* _tesselator) :
mesh(_mesh),
scalarSets(_scalarSets),
vecSets(_vecSets),
tessellator(_tesselator)
{
  const MiGeometryI& geometry = _mesh.getGeometry();
  size_t beginNodeId = _mesh.getTopology().getBeginNodeId();
  size_t endNodeId = _mesh.getTopology().getEndNodeId();

  // compute bbox of the mesh
  for (size_t i = beginNodeId; i < endNodeId; ++i)
    m_bbox.extendBy((SbVec3f)geometry.getCoord(i));

  // compute and store min and max of each scalar set
  double val, maxval, minval;
  size_t beginId, endId;

  for (size_t i=0; i!=scalarSets.size(); ++i)
  {
    const MiScalardSetI* mydataset = scalarSets[i];
    minval = std::numeric_limits<double>::max();
    maxval = -std::numeric_limits<double>::max();

    if (mydataset->getBinding() == MiDataSet::PER_NODE)
    {
      beginId = beginNodeId;
      endId = endNodeId;
    } else
    {
      beginId = 0;
      endId = _mesh.getTopology().getNumCells();  
    }

    for (size_t j = beginId; j < endId; ++j)
    {
      val = mydataset->get(j);
      if (val > maxval) maxval = val;
      if (val < minval) minval = val;
    }
    m_minScalarSet.push_back((float)minval);
    m_maxScalarSet.push_back((float)maxval);
  }

}

//-----------------------------------------------------------------------------
inline SbBox3f
MyMesh::getBoundingBoxf() const
{
  return m_bbox;
}


#endif

