////////////////////////////////
//
////////////////////////////////

#include <Inventor/STL/vector>
#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>

#include "../MeshViewer/MeshViewer.h"
#include <mesh/volumes/MbVolumeMeshHexahedron.h>
#include <data/MbScalarSetI.h>
#include <data/MbVec3SetI.h>

using std::vector;
using std::string;
using std::endl;
using std::cout;

MbScalarSetI<> ds0;
MbScalarSetI<> ds1;
MbScalarSetI<> ds2;
MbScalarSetI<> ds3;
MbScalarSetI<> ds4;
MbVec3SetI<> dv;

//-----------------------------------------------------------------------------

void readMeshFile(string fileName, MbVolumeMeshHexahedron<>& mesh) 
{
  SbString path = SoPreferences::getString("OIVHOME",".") + "/examples/data/MeshViz/";

  string filePath = path.getString() + fileName;
  int i, numHexahedrons, numNodes;

  cout << "Reading file " << filePath << endl;

  std::ifstream fileStream(filePath.c_str());
  if (fileStream.fail()) 
  {
    cout << "Cannot open file " << filePath << endl;
    exit(-1);
  }

  fileStream >> numHexahedrons >> numNodes;
  cout << numHexahedrons << " hexahedrons, " << numNodes << " nodes" << endl;

  // reading the mesh topology
  vector<int> hexahedronNodes(numHexahedrons*8);
  for (i=0; i<numHexahedrons*8; i++) 
    fileStream >> hexahedronNodes[i];
  mesh.setTopology(hexahedronNodes.begin(),hexahedronNodes.end());

  // reading the mesh geometry
  vector<MbVec3d> coords(numNodes);
  for (i=0; i<numNodes; i++) 
    fileStream >> coords[i][0] >> coords[i][1] >> coords[i][2];
  mesh.setGeometry(coords.begin(),coords.end());

  // reading the mesh datasets
  vector<double> vds0(numNodes);
  vector<double> vds1(numNodes);
  vector<double> vds2(numNodes);
  vector<double> vds3(numNodes);
  vector<double> vds4(numNodes);
  for (i=0; i<numNodes; i++) 
    fileStream >> vds0[i] >> vds1[i] >> vds2[i] >> vds3[i] >> vds4[i];

  ds0.assign(vds0.begin(),vds0.end());
  ds0.setName( "density");
  mesh.addScalarSet(&ds0);

  ds1.assign(vds1.begin(),vds1.end());
  ds1.setName( "momentum U");
  mesh.addScalarSet(&ds1);

  ds2.assign(vds2.begin(),vds2.end());
  ds2.setName( "momentum V");
  mesh.addScalarSet(&ds2);

  ds3.assign(vds3.begin(),vds3.end());
  ds3.setName( "momentum W");
  mesh.addScalarSet(&ds3);

  ds4.assign(vds4.begin(),vds4.end());
  ds4.setName( "enthalpy");
  mesh.addScalarSet(&ds4);

  fileStream.close();

  vector<MbVec3d> vecs(numNodes);
  for (i=0; i<numNodes; i++) vecs[i].setValue(vds1[i],vds2[i],vds3[i]);
  dv.assign(vecs.begin(),vecs.end());
  dv.setName( "UVW");
  mesh.addVec3Set(&dv);

  cout << "End reading file " << filePath << endl;
}

//-----------------------------------------------------------------------------
int main(int, char **)
{
  MbVolumeMeshHexahedron<> mesh;
  readMeshFile("TURBO.DAT",mesh);  

  MeshViewer *mesh_viewer = new MeshViewer();
  mesh_viewer->setWindowTitle("Turbine");
  mesh_viewer->setCourtesyTitle("by courtesy of Turbomeca");

  vector<const MiScalardSetI*> dss(mesh.getNumScalarSets(),NULL);
  for (size_t i=0; i<dss.size(); ++i)
    dss[i] = mesh.getScalarSet(i);

  vector<const MiVec3dSetI*> dsv(mesh.getNumVec3Sets(),NULL);
  for (size_t i=0; i<dsv.size(); ++i)
    dsv[i] = mesh.getVec3Set(i);

  mesh_viewer->show(mesh,dss,dsv);

  delete mesh_viewer;
  return 0;
}


