#ifndef  _Bench_h
#define  _Bench_h

// define MBMESH_DONT_USE_ASSERT to prevent assert and misc checks (like isDead inside getCellNodeIds) to speed up the bench
#define MBMESH_DONT_USE_ASSERT

// define MBMESH_DONT_USE_STL to implement mesh without STL to speed up the bench on MinSizeRel windows platform.
#define MBMESH_DONT_USE_STL

#define MBMESH_DATA_ACCESS_MICROSLEEP 1

#include <Inventor/lock/SoLicensesInfo.h>

#include <MeshVizXLM/MiMeshViz.h>
#include <MeshVizXLM/extractors/MiExtractorCallback.h>

#include <Inventor/SoDB.h>
#include <Inventor/STL/iostream>
#include <Inventor/STL/sstream>
#include <Inventor/STL/iomanip>
#include <Inventor/STL/fstream>
#include <Inventor/STL/list>

#include "../Common.h"
#include "BenchParameters.h"
#include "BenchExtractorCallback.h"
#include "Gauge.h"

class MiTessellator;

template < typename _ExtractorT, typename _MeshT, typename _FilterT = CellFilter>
class Bench
{
public:
  typedef _MeshT MeshType;
  typedef _FilterT FilterType;

  Bench(std::string extractname, std::string meshtype, int argc, char* argv[]);
  ~Bench();

  void run(size_t numCase, BenchParameters& params);
  static void setNumThreads( int numThreads );

private:
  /**      Bench methods to be overloaded or specialized       **/
  // write report header (overload to add more results)
  virtual void writeHeader(std::ostringstream& result) const;
  // write case info in report (overload to add more to results)
  virtual void writeCaseInfo(std::ostringstream& /*result*/) const {}
  // return the a new extractor
  _ExtractorT* getNewExtractor(_MeshT& mesh, bool parallel)
  { return _ExtractorT::getNewInstance(mesh,parallel,m_parameters->tessellator); }
  // return the mesh corresponding to given number
  virtual _MeshT* getMesh(size_t num, bool deadCells) = 0;
  virtual size_t getNumCells() const = 0;
  // overload if microsleep must be added to the mesh
  virtual void setUSleep(size_t /*us*/) {}
  // touch mesh at each iteration (specify to empty if no touch in mesh)
  void touch();
  virtual void touchFilters();
  // return number of filters to run
  virtual size_t getNumFilters() const;
  // extractor to bench
  virtual void    extract(_ExtractorT* extractor, const _FilterT* filter) const = 0;
  /************************************/

  void profile(_MeshT& mesh, size_t usleep, size_t filterId = -1);
  void profileFilters(_MeshT& mesh);
  void startExtraction( std::ostringstream& result, _ExtractorT* extract, const _FilterT* cellFilter = NULL );

  BenchExtractorCallback m_timersCB;

  std::string m_name;
  std::string m_meshtype;
  std::string m_filename;
  std::ofstream::openmode m_mode;
  std::ostringstream m_results;

  static int s_numThreads;
  static bool s_reportMemoryInfo;
  
protected:
  virtual void runCase(size_t caseNum);
  virtual _FilterT* getFilter(size_t /*i*/, std::string& /*fname*/, size_t /*usleep*/);

  size_t* m_dims;
  BenchParameters* m_parameters;
  size_t m_currentCase;
  const char*    m_sep;
  _MeshT*  m_currentMesh;
  Gauge m_gauge;

};

template <typename _ExtractT, typename _MeshT, typename _FilterT>
int Bench<_ExtractT,_MeshT,_FilterT>::s_numThreads = -1;

template <typename _ExtractT, typename _MeshT, typename _FilterT>
bool Bench<_ExtractT, _MeshT, _FilterT>::s_reportMemoryInfo = true;

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
Bench<_ExtractT,_MeshT,_FilterT>::Bench(std::string extractname, std::string meshtype, int argc, char* argv[])
: m_name(extractname), m_meshtype(meshtype), m_currentMesh(NULL)
{
  m_sep = "";
  int optc = 0;

  if (argc>1 || s_reportMemoryInfo)
  {
    if (s_reportMemoryInfo || strcmp(argv[0], "-f") == 0)
    {
      ++optc;
      m_filename = "bench.csv";
      m_mode = std::ofstream::out;
      if (argc-1 > optc)
      {
        if (strcmp(argv[2],"-a")==0)
        {
          ++optc;
          m_mode = std::ofstream::app | std::ofstream::ate;
        }
        if (argc-1 > optc)
          m_filename = argv[optc+1];
      }
      m_sep = ";";
    }
    else
    {
      std::cout << "bench [-f -a [filename] ]: print result in file .csv" << std::endl;
      exit(1);
    }
  }

  m_results << SoLicensesInfo::getInstance().getVersion() << std::endl;
  std::cout << SoLicensesInfo::getInstance().getVersion() << std::endl;
  std::cout << "MeshVizXLM " << m_name << " Bench"<< std::endl << std::endl;
  
  MiMeshViz::init(s_numThreads);
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
Bench<_ExtractT,_MeshT,_FilterT>::~Bench()
{
  MiMeshViz::finish();
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void Bench<_ExtractT, _MeshT, _FilterT>::setNumThreads( int numThreads )
{
  s_numThreads = numThreads;
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void 
  Bench<_ExtractT,_MeshT,_FilterT>::run(size_t numCase, BenchParameters& params)
{
  m_parameters = &params;
  m_timersCB.includeProgressCallback = params.includeProgressCallback;
  m_timersCB.displayPhaseTimings = params.displayPhaseTimings;
  writeHeader(m_results);
  m_results << std::endl;

  for (size_t i = 0; i < numCase; ++i)
    runCase(i);

  if (strcmp(m_sep,"") == 0)
    std::cout << m_results.str();
  else // write to file
  {
    std::ofstream out(m_filename.c_str(),m_mode);
    if ((out.tellp() == (std::streampos) 0))
      out << "sep=" << m_sep << std::endl;
    out << m_results.str();
    out.close();
    std::cout << "report in " << m_filename << std::endl;
  }
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void Bench<_ExtractT,_MeshT,_FilterT>::writeHeader(std::ostringstream &results) const
{
  results << m_name << "Extract from " << m_meshtype << std::endl;
  results.precision(3);
  results.setf ( std::ios::showbase );

  if (m_parameters->includeParallel)
  {
    if (s_numThreads == -1)
      results << "num Threads = max" << std::endl;
    else
      results << "num Threads = " << s_numThreads << std::endl;
  }

  results << std::setw(6) << "Cells" << m_sep << std::setw(7) << "Dead"<< m_sep;
  results  << std::setw(8) << "Filter" << m_sep;
  if (m_parameters->includeSequential) 
  {
    results << std::setw(7) << "Seq(s)" << m_sep << std::setw(5) << "1st" << m_sep << std::setw(5) << "avg" << m_sep << std::setw(5) << "min" << m_sep;
    results << "1st(MB) " << m_sep << "next avg(MB)" << m_sep;
  }
  if (m_parameters->includeParallel)
  {
    results << std::setw(7) << "Par(s)" << m_sep << std::setw(5) << "1st" << m_sep << std::setw(5) << "avg" << m_sep << std::setw(5) << "min" << m_sep;
    results << "1st(MB) " << m_sep << "next avg(MB)" << m_sep;
  }
  if (m_parameters->includeMicroSleep)
    results << std::setw(8) << "Sleep(us)" << m_sep ;
  results << std::setw(8) << "ExCells";
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void 
Bench<_ExtractT,_MeshT,_FilterT>::runCase(size_t caseNum)
{
  size_t maxMicroSleep = m_parameters->includeMicroSleep ? MBMESH_DATA_ACCESS_MICROSLEEP : 0; 

  try 
  {
    size_t numPass = m_parameters->includeDeadCells ? 2 : 1;
    bool useDeadCells = false;
    for (size_t n = 0; n < numPass; ++n)
    {
      m_currentMesh = getMesh(caseNum, useDeadCells);
      m_currentCase = caseNum;
      if (m_currentMesh)
      {
        if (!m_parameters->includeOnlyFiltering)
          for (size_t usleep = 0; usleep < maxMicroSleep + 1; ++usleep)
            profile(*m_currentMesh, usleep);

        profileFilters(*m_currentMesh);
      }
      useDeadCells = true;
    }
  }

  catch (std::exception exception)
  {
    std::cout << exception.what() << std::endl;
  }

  catch (...)
  {
    // all other exceptions
    std::cout << "unknown exception" << std::endl;
  }


}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void 
Bench<_ExtractT,_MeshT,_FilterT>::profile(_MeshT& mesh, size_t usleep, size_t filterId)
{
  setUSleep(usleep);
  std::cout << std::endl << "Starting " << m_name << " extraction:";
  std::string fname = "";
  const _FilterT* cellFilter = NULL;
  if (filterId != -1)
  {
    cellFilter = getFilter(filterId,fname,usleep);
    std::cout << " using " << fname << " cell filter ";
  }
  std::cout << std::endl;
  const char* milli = "";
  size_t numCells = getNumCells();
  if (numCells >= 1e6) 
  {
    numCells = (size_t) (numCells * 1e-6);
    milli = "m"; 
  }
  m_results << std::setw(6) << numCells << milli;
  m_results << m_sep  << std::setw(7) << (mesh.getTopology().hasDeadCells()? "true" : "false");
  m_results << m_sep << std::setw(8) << (cellFilter? fname : "  --  ");

  std::ostringstream secondResult;
  secondResult.precision( 3 );
  secondResult.setf( std::ios::showbase );

  _ExtractT* extract;
  size_t numExtrCells = 0;
  if (m_parameters->includeSequential)
  {
    std::cout << std::endl << "Sequential" << std::endl;
    // Get an instance of an sequential extractor according to the mesh type
    extract = getNewExtractor(mesh,false);
    extract->setExtractorCallback(&m_timersCB);
    startExtraction( m_results, extract, cellFilter );
    numExtrCells = extract->getExtract().getTopology().getNumCells();

    if ( m_parameters->includeSecondExtractor )
    {
      _ExtractT* newExtract = getNewExtractor( mesh, false );
      newExtract->setExtractorCallback( &m_timersCB );
      startExtraction( secondResult, newExtract, cellFilter );
      if ( numExtrCells != extract->getExtract().getTopology().getNumCells() )
        std::cout << "WARNING! Number of extracted cells does not match previous extraction: " << extract->getExtract().getTopology().getNumCells() << std::endl;
      delete newExtract;
    }

    delete extract;
  }

  if (m_parameters->includeParallel)
  {
    std::cout << std::endl << "Parallel" << std::endl;
    // Get an instance of an parallel extractor according to the mesh type
    extract = getNewExtractor(mesh,true);
    extract->setExtractorCallback(&m_timersCB);

    startExtraction( m_results, extract, cellFilter );
    if (!m_parameters->includeSequential)
       numExtrCells = extract->getExtract().getTopology().getNumCells();
    else if (numExtrCells != extract->getExtract().getTopology().getNumCells())
      std::cout << "WARNING! Number of extracted cells does not match sequential mode: " << extract->getExtract().getTopology().getNumCells()<< std::endl;
    std::cout << std::endl;

    if ( m_parameters->includeSecondExtractor )
    {
      _ExtractT* newExtract = getNewExtractor( mesh, true );
      newExtract->setExtractorCallback( &m_timersCB );
      startExtraction( secondResult, newExtract, cellFilter );
      if ( numExtrCells != extract->getExtract().getTopology().getNumCells() )
        std::cout << "WARNING! Number of extracted cells does not match previous extraction: " << extract->getExtract().getTopology().getNumCells() << std::endl;
      delete newExtract;
    }

    delete extract;
  }
  if (m_parameters->includeMicroSleep)
    m_results << m_sep << std::setw(8) <<  usleep;
  m_results << m_sep << std::setw(8) << numExtrCells;
  writeCaseInfo(m_results);
  m_results << std::endl;

  if ( m_parameters->includeSecondExtractor )
  {
    m_results << m_sep << m_sep << secondResult.str();
    m_results << std::endl;
  }
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void 
Bench<_ExtractT,_MeshT,_FilterT>::startExtraction( std::ostringstream& result, _ExtractT* extractor, const _FilterT* cellFilter )
{
  m_gauge.reset();
  // first extract not benched to ignore initial memory alloc
  extract(extractor,cellFilter);
  m_gauge.sample("extraction");
  std::cout << ".";

  for (size_t i =0; i < m_parameters->numIteration; ++i)
  {
    if (m_parameters->includeTouchMesh) touch();
    if (cellFilter) touchFilters();
    m_gauge.sample();
    extract(extractor,cellFilter);
    m_gauge.sample("extraction");
    std::cout << ".";
  }
  std::cout << std::endl;
  result << m_sep << std::setw(5);
  std::cout << m_gauge.reportStat( result, m_sep ) << std::endl;
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void 
Bench<_ExtractT,_MeshT,_FilterT>::touch()
{
  m_currentMesh->touch();
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
void 
Bench<_ExtractT,_MeshT,_FilterT>::profileFilters(_MeshT& mesh)
{
  size_t maxMicroSleep = m_parameters->includeMicroSleep ? MBMESH_DATA_ACCESS_MICROSLEEP : 0; 
  size_t numFilters = getNumFilters();

  for (size_t f = 0; f < numFilters; ++f)
  {
    for (size_t usleep = 0; usleep < maxMicroSleep+1; ++usleep)
      profile(mesh,usleep,f);
  }
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
inline size_t
Bench<_ExtractT,_MeshT,_FilterT>::getNumFilters() const
{
  return m_parameters->getNumFilters();
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
inline _FilterT*
Bench<_ExtractT,_MeshT,_FilterT>::getFilter(size_t f, std::string& fname, size_t /*usleep*/)
{
  _FilterT* filter = NULL;
  if (f < m_parameters->getNumFilters())
  {
    filter = m_parameters->getFilter(f);
    filter->dimension = m_dims[m_currentCase];
    fname = filter->getName();
    filter->touch();
  }
  return filter;
}

//-----------------------------------------------------------------------------
template <typename _ExtractT, typename _MeshT, typename _FilterT>
inline void
Bench<_ExtractT,_MeshT,_FilterT>::touchFilters()
{
  m_parameters->touchFilters();
}


#endif 


