#ifndef  _BenchExtractorCallback_h
#define  _BenchExtractorCallback_h

#include <MeshVizXLM/extractors/MiExtractorCallback.h>

#include <Inventor/SbElapsedTime.h>
#include <Inventor/STL/iostream>
#include <Inventor/STL/sstream>
#include <Inventor/STL/iomanip>
#include <Inventor/STL/fstream>
#include <Inventor/STL/list>

#include "../Common.h"
#include "Gauge.h"

class BenchExtractorCallback : public MiExtractorCallback
{
public:
  bool includeProgressCallback;
  bool displayPhaseTimings;

  BenchExtractorCallback() : includeProgressCallback(false), displayPhaseTimings(false), m_numExtractor(0), m_progressStarted(false) {}
  virtual void beginExtract(const std::string extractorName, 
                            bool /*geomChanged*/, bool /*topoChanged*/, bool /*dataSetChanged*/, 
                            size_t /*numPhases*/) 
  {
    if (displayPhaseTimings)
    {
      offsetDisplay();
      std::cout << "starting " << extractorName << std::endl;
      if (m_numExtractor == 0)
        m_gauge.reset();
    }
    ++m_numExtractor;
    m_phaseNames.resize(m_numExtractor);
    m_starts.resize(m_numExtractor);
    m_numCurrentPhaseSteps.resize(m_numExtractor);
  }
  virtual void endExtract() 
  { 
    --m_numExtractor; 
    if (m_numExtractor == 0 && displayPhaseTimings)
      m_gauge.sample("finishing extraction", true);
  }
  virtual bool beginPhase(size_t /*phase*/, std::string phaseName, size_t numSteps ) 
  {
    m_phaseNames[m_numExtractor-1] = phaseName;
    m_numCurrentPhaseSteps[m_numExtractor-1] = numSteps;
    m_starts[m_numExtractor-1] = m_localTime.getElapsed();
    return true;
  }
  virtual bool endPhase() 
  {
    if (displayPhaseTimings)
    {
      offsetDisplay();
      m_gauge.sample(m_phaseNames[m_numExtractor - 1], true);
    }
    m_progressStarted = false;
    return true; 
  }
  virtual bool endStep(size_t numIterationDone) 
  {
    if (includeProgressCallback)
    {
      if (!m_progressStarted)
      {
        offsetDisplay();
        std::cout << "start " << m_phaseNames[m_numExtractor-1] << std::endl;
      }
      offsetDisplay();
      std::streamsize prec = std::cout.precision();
      std::cout.precision(0);
      std::cout << std::fixed << 100 *numIterationDone / (double) m_numCurrentPhaseSteps[m_numExtractor-1] << "% done " << std::endl;
      std::cout.precision(prec);
      std::cout.unsetf(std::ios_base::floatfield);
      m_progressStarted = true;
    }
    return true; 
  }
  double getEndStepCallPeriod() { return includeProgressCallback ? 0.05 : 1; }

private:
  void offsetDisplay() { for( size_t i = 0; i < m_numExtractor; ++i) std::cout << "  "; }
  Gauge m_gauge;
  size_t m_numExtractor;
  SbElapsedTime m_localTime;
  std::vector<double> m_starts;
  std::vector<std::string> m_phaseNames;
  std::vector<size_t> m_numCurrentPhaseSteps;
  bool m_progressStarted;
};

#endif 


