#ifndef _COMMON_H
#define _COMMON_H

#include <Inventor/STL/algorithm>
#include <Inventor/SbTime.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterI.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterIj.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterIjk.h>
#include <MeshVizXLM/mesh/data/MiDataSetI.h>
#include <MeshVizXLM/mesh/geometry/MiGeometryI.h>
#include <MbSampleMeshFunctor.h>

//-----------------------------------------------------------------------------
class MyCellFilter : public MiCellFilterI
{
  virtual bool acceptCell(size_t /*cellIndex*/) const { return true; }
  virtual size_t getTimeStamp() const { return 0; }
};

//-----------------------------------------------------------------------------
class NoCellFilterI : public MiCellFilterI
{
  size_t m_ts;
public:
  size_t microsleep;
  NoCellFilterI() : m_ts(1), microsleep(0) {}
  virtual bool acceptCell(size_t) const 
  { 
    if (microsleep>0)
      SbTime::usleep(microsleep);
    return true; 
  }
  void touch() {++m_ts;}
  virtual size_t getTimeStamp() const { return m_ts; }
  std::string getName() const { return "NoCell"; }
};

//-----------------------------------------------------------------------------
class BlockCellFilterI : public MiCellFilterI
{
public:
  size_t microsleep;
  BlockCellFilterI() : microsleep(0) {}
  virtual bool acceptCell(size_t i) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return (i/10)%2 == 0; 
  }
  virtual size_t getTimeStamp() const { return 1; }
  std::string getName() const { return "Block"; }
};


class MyCellFilterIj : public MiCellFilterIj
{
  virtual bool acceptCell(size_t /*i*/, size_t /*j*/) const { return true; }
  virtual size_t getTimeStamp() const { return 1; }
};

class CellFilter : public MiCellFilterIjk, public MiCellFilterI
{
  size_t m_ts;
public:
  CellFilter() : m_ts(1), microsleep(0) {}
  MbVec3ui dimension;
  size_t microsleep;
  using MiCellFilterIjk::acceptCell;
  virtual bool acceptCell(size_t id) const 
  { 
    size_t k = id / (dimension[0]*dimension[1]);
    size_t ij = id % (dimension[0]*dimension[1]);
    size_t j = ij / dimension[0];
    size_t i = ij % dimension[0];
    return acceptCell(i,j,k);
  }
  void touch() {++m_ts;}
  virtual std::string getName() const = 0;
  virtual size_t getTimeStamp() const { return m_ts; }
};

class MyCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t j, size_t k) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return !(i==j && j==k); 
  }
};

class HalfCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t, size_t) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return (i < dimension[0] / 2); 
  }
  std::string getName() const { return "Half"; }
};

class ThirdCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t, size_t) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return (i < size_t( 0.65 * dimension[0])); 
  }
  std::string getName() const { return "Third"; }
};

class SpreadHalfCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t, size_t) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return (i/10)%2 == 0; 
  }
  std::string getName() const { return "SHalf"; }
};

class Every2CellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t j , size_t k) const
  {
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return (i % 2 && j % 2 && k % 2);
  }
  std::string getName() const { return "Every2Cells"; }
};

class SliceICellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t, size_t) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return i==0; 
  }
  std::string getName() const { return "SliceI"; }
};

class SliceKCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t, size_t, size_t k) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return k==0; 
  }
  std::string getName() const { return "SliceK"; }
};

class CrossCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t j, size_t) const 
  { 
    if (microsleep > 0)
      SbTime::usleep(microsleep);
    return i==j || i == dimension[1] - j; 
  }
  std::string getName() const { return "Cross"; }
};

class NoCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t, size_t, size_t) const 
  { 
    if (microsleep>0)
      SbTime::usleep(microsleep);
    return true; 
  }
  std::string getName() const { return "NoCell"; }
};

class BorderCellFilterIjk : public CellFilter
{
public:
  virtual bool acceptCell(size_t i, size_t, size_t) const 
  { 
    if (microsleep>0)
      SbTime::usleep(microsleep);
    return i != 0; 
  }
  std::string getName() const { return "Border"; }
};

class MyDeadCellIjk : public MbDeadCellFctorIjk
{
public:
  bool operator()(size_t /*i*/, size_t j, size_t k) const
  {
    return j+1==k;
  }
};

class NoDeadCellI : public MbDeadCellFctorI
{
public:
  bool operator()(size_t /*i*/) const
  {
    return false;
  }
};

class NoDeadCellIjk : public MbDeadCellFctorIjk
{
public:
  bool operator()(size_t i, size_t j, size_t k) const
  {
    return i == 0 && j == 0 && k == 0;
  }
};

class MyDeadCellIj : public MbDeadCellFctorIj
{
public:
  bool operator()(size_t i, size_t j) const
  {
    return i==j;
  }
};

class MyDeadCellI : public MbDeadCellFctorI
{
public:
  bool operator()(size_t i) const
  {
    return i%3==0;
  }
};

class XScalarSet : public MiScalardSetI
{
public:
  XScalarSet(const MiGeometryI& geom) : m_geom(geom) {}
  double get(size_t i) const { return m_geom.getCoord(i)[0]; }
  size_t getTimeStamp() const { return m_geom.getTimeStamp(); }
  double getMin() const { return m_geom.getMin()[0]; }
  double getMax() const { return m_geom.getMax()[0]; }
  std::string getName() const { return "x values"; }
  MiDataSet::DataBinding getBinding() const { return MiDataSet::PER_NODE; }

private:
  const MiGeometryI& m_geom;
};

class SphereScalarSet : public MiScalardSetI
{
public:
  SphereScalarSet(const MiGeometryI& geom) : m_geom(geom) 
  {
  }
  double get(size_t i) const 
  { 
    double x = m_geom.getCoord(i)[0];
    double y = m_geom.getCoord(i)[1];
    double z = m_geom.getCoord(i)[2];
    return x*x + y*y + z*z; 
  }
  size_t getTimeStamp() const { return m_geom.getTimeStamp(); }
  double getMin() const 
  { 
    MbVec3d cmin = m_geom.getMin();
    return cmin[0]*cmin[0] + cmin[1]*cmin[1] + cmin[2]*cmin[2];
  }
  double getMax() const 
  { 
    MbVec3d cmax = m_geom.getMax();
    return cmax[0]*cmax[0] + cmax[1]*cmax[1] + cmax[2]*cmax[2];
  }
  std::string getName() const { return "sphere values"; }
  MiDataSet::DataBinding getBinding() const { return MiDataSet::PER_NODE; }

private:
  const MiGeometryI& m_geom;
};

class AltitudeFctor1 : public MbScalarFunctor<float>
{
public:
  AltitudeFctor1(size_t numCellI, size_t numCellJ, float step):
      m_numCellI(numCellI), m_numCellJ(numCellJ), m_step(step) {}

  float operator()(float x, float y, float z) const
  {
    float value;
    float y1 = y>0 ? y : z;
    size_t i = size_t (x/m_step);
    size_t j = size_t (y1/m_step);
    if( i == 1 && j==1) value = 0;
    else if( i==m_numCellI-1 && j==m_numCellJ-1) value = m_step*std::max(m_numCellI,m_numCellJ);
    else value = std::max(x,y1);
    return value;
  }
private:
  size_t m_numCellI;
  size_t m_numCellJ;
  float m_step;
};

class AltitudeFctor2 : public MbScalarFunctor<float>
{
public:
  AltitudeFctor2(size_t numCellI, size_t numCellJ, float step):
      m_numCellI(numCellI), m_numCellJ(numCellJ), m_step(step) {}
  
  float operator()(float x, float y, float z) const
  {
    float value;
    float y1 = y>0 ? y : z;
    size_t i = size_t (x/m_step);
    size_t j = size_t (y1/m_step);
    if(j<=3-i)
      value = std::max(x,y1);
    else
      value = std::max(m_numCellI-i,m_numCellJ-j)*m_step + 1;
    return value;
  }
private:
  size_t m_numCellI;
  size_t m_numCellJ;
  float m_step;
};

#endif


