#include <fstream>
#include <iostream>
#include <iomanip>

#include <MeshVizXLM/MiMeshViz.h>
#include <MeshVizXLM/extractors/MiGridPlaneSliceExtractUnstructured.h>

#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>

#include <MbSampleMeshBuilder.h>

#include <Inventor/SbElapsedTime.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/vector>
using namespace std;

#define NAMESTR "MeshVizXLM grid plane slice extraction bench"

enum MeshType{ HEXA, TETRA };

void run(MeshType type, vector<size_t>& dims, ostringstream& results, const char* sep = " ");
double extract(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB, ostringstream& result, const char* sep = " ");

#if defined(_MSC_VER)
#pragma warning( push )
#pragma warning(disable:4250)
#endif

SbElapsedTime localTime;

class MeshBuilder 
{
public:
  const MiVolumeMeshUnstructured& getMesh(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB);
  string getMeshType() { return m_meshType; }
private:
  MbSampleMeshBuilder<MbVec3d,double> m_meshBuilder;
  string m_meshType;
};
MeshBuilder meshBuilder;


const MiVolumeMeshUnstructured& MeshBuilder::getMesh(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB)
{
  switch (type)
  {
  case HEXA:
    m_meshType = "Hexa";
    return m_meshBuilder.getMeshHexahedron(MbVec3<size_t>(dim),minBB,maxBB);
    break;
  case TETRA:
    m_meshType = "Tetra";
    return m_meshBuilder.getMeshTetrahedron(MbVec3<size_t>(dim),minBB,maxBB);
    break;
  default:
    cout << " error mesh type " << endl;
    exit(1);
  };
}

//-----------------------------------------------------------------------------
int
main(int argc, char* argv[])
{
  ostream* output = &cout;
  const char* sep = "";

  if(argc>1)
  {
    if (strcmp(argv[1],"-f")==0)
    {
      const char* filename = "bench_gridslice.csv";
      if (argc == 3)
         filename = argv[2];
      output = new ofstream(filename);
      sep = ";";
    }
    else
    {
      cout << "bench [-f [filename] ]: print result in file .csv" << endl;
      exit(1);
    }
  }

  MiMeshViz::init();

  ostringstream results;
  results << setw(8) << "MeshType" << sep  << setw(10) << "NumCells" << sep << setw(10) << "NumSlices" << sep ;
  results << setw(12) << "Pts/Slice" << sep;
  results << setw(18) << "OverallTime(sec)" << sep <<setw(19) << "Time/Slice(ms)" << endl;
  results.precision(4);
  results.setf ( ios::showbase );

  //run bench for hexahedon mesh
  size_t dims1[4] = {40,80,120,160};
  vector<size_t> dimensions(dims1,dims1+4);
  run(HEXA,dimensions,results,sep);

  results << endl;

  //run bench for tetrahedron mesh
  size_t dims2[4] = {30,60,90,120};
  dimensions.assign(dims2,dims2+4);
  run(TETRA,dimensions,results,sep);

  (*output) << results.str();

  if (argc>1)
    delete output;

  MiMeshViz::finish();
  return 0;
}

void run(MeshType type, vector<size_t>& dims, ostringstream& results, const char* sep)
{
  for (size_t d = 0; d < dims.size(); d++)
  {
    extract(type,dims[d],MbVec3d(0),MbVec3d(100),results, sep);
    cout << endl;
  }
}

double
extract(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB, ostringstream& result, const char* sep)
{
  const MiVolumeMeshUnstructured& mesh = meshBuilder.getMesh(type,dim,minBB,maxBB);
  cout << meshBuilder.getMeshType() << " Mesh" << endl;
  cout << "mesh built, num cells=" << mesh.getTopology().getNumCells() 
    << ", num nodes=" << mesh.getTopology().getEndNodeId() << endl;

  double minV = minBB[0];
  double maxV = maxBB[0];
  size_t numExtract = 80;
  double deltaV = (maxV-minV)/double(numExtract);
  double distance = minV;
  MbVec3<double> normal (1,0,0);
  double step = (maxV-minV) / double(100);

  // Get an instance of a slice extractor according to the mesh type
  MiGridPlaneSliceExtractUnstructured* extractor = MiGridPlaneSliceExtractUnstructured::getNewInstance(mesh);
  // first extract not benched to ignore initial memory alloc
  const MiTopologyIj& grid = extractor->extractGrid(normal,distance,step).getTopology();
  size_t count = 0;
  if(grid.hasDeadCells()) 
  {
    for( size_t l=0;l<grid.getNumCellsI();++l)
      for( size_t m=0;m<grid.getNumCellsJ();++m)
        if(grid.isDead(l,m)) ++count;
  }
  size_t totalNumPoints = grid.getNumCellsI()*grid.getNumCellsJ();

  distance = minV + deltaV*0.5;
  cout << "start extracting " << numExtract << " slices" << endl;
  double t1 = localTime.getElapsed();
  for (size_t i=0; i<numExtract; ++i)
  {
    extractor->extractGrid(normal,distance,step);
    cout << ".";
    distance += deltaV;
  }
  cout << endl;
  double t2 = localTime.getElapsed();

  result << setw(8) << meshBuilder.getMeshType() << sep << setw(10) << mesh.getTopology().getNumCells()  << sep  << setw(10)<< numExtract <<  sep;
  result << setw(12) << (totalNumPoints - count) <<  sep;
  result << setw(18) << t2-t1 << sep  <<setw(19) << 1000*(t2-t1)/float(numExtract) << endl;

  delete extractor;
  return t2-t1;
  }

#if defined(_MSC_VER)
#pragma warning( pop ) 
#endif
