#include <fstream>
#include <iostream>
#include <iomanip>
#include <Inventor/SoPreferences.h> 
#include <MeshVizXLM/MiMeshViz.h>
#include <MeshVizXLM/extractors/MiPointProbeUnstructured.h>

#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>

#include <MbSampleMeshBuilder.h>

#include <Inventor/SbElapsedTime.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/vector>
using namespace std;

#define NAMESTR "MeshVizXLM pointprobe extraction bench"
//#define INITONLY

enum MeshType{ HEXA, TETRA };

void run(MeshType type, vector<size_t>& dims, ostringstream& results, const char* sep = " ");
double probeMesh(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB, double rotation, ostringstream& result, const char* sep = " ");

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4250)
#endif

SbElapsedTime localTime;

class MeshBuilder 
{
public:
  const MiVolumeMeshUnstructured& getMesh(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB, double rotations[3]);
  string getMeshType() { return m_meshType; }
private:
  MbSampleMeshBuilder<MbVec3d,double> m_meshBuilder;
  string m_meshType;
};
MeshBuilder meshBuilder;


const MiVolumeMeshUnstructured& MeshBuilder::getMesh(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB, double rotations[3])
{
  switch (type)
  {
  case HEXA:
    m_meshType = "Hexa";
    return m_meshBuilder.getMeshHexahedron(MbVec3<size_t>(dim),minBB,maxBB,rotations);
    break;
  case TETRA:
    m_meshType = "Tetra";
    return m_meshBuilder.getMeshTetrahedron(MbVec3<size_t>(dim),minBB,maxBB,rotations);
    break;
  default:
    cout << " error mesh type " << endl;
    exit(1);
  };
}

//-----------------------------------------------------------------------------
int
main(int argc, char* argv[])
{
  ostream* output = &cout;
  const char* sep = "";

  if(argc>1)
  {
    if (strcmp(argv[1],"-f")==0)
    {
      const char* filename = "bench_pointprobe.csv";
      if (argc == 3)
         filename = argv[2];
      output = new ofstream(filename);
      sep = ";";
    }
    else
    {
      cout << "bench [-f [filename] ]: print result in file .csv" << endl;
      exit(1);
    }
  }

  MiMeshViz::init();

  ostringstream results;
  results << setw(10) << "MeshType" << sep  << setw(10) << "NumCells" << sep << setw(15) << "ProbedPoints" << sep ;
  results << setw(14) << "InitTime(sec)";
#ifndef INITONLY
  results << sep << setw(18) << "OverallTime(sec)" << sep <<setw(25) << "Time for 10000 pts(sec)" << endl;
#else
  results << endl;
#endif

  results.precision(4);
  results.setf ( ios::showbase );

  SoPreferences::setBool("MESHVIZ_OCTREE_CACHE_CELLBBOX",true);
  size_t maxdepth[4] = {5,6,7,8};
  size_t maxcellptile[3] = {100,40,10};
  for( size_t d=0; d < 4; d++)
    for( size_t c=0; c < 3; c++)
    {
      SoPreferences::setInt("MESHVIZ_OCTREE_MAX_DEPTH",(int)maxdepth[d]);
      SoPreferences::setInt("MESHVIZ_OCTREE_MAX_CELL_PER_TILE",(int)maxcellptile[c]);
      cout << "Max Depth " << maxdepth[d] << endl << "Max Cell Per Tile " << maxcellptile[c]<< endl;
      //run bench for hexahedon mesh
      size_t hexadims[4] = {40,80,128,160};
      vector<size_t> dimensions(hexadims,hexadims+4);
      run(HEXA,dimensions,results,sep);

      //run bench for tetrahedron mesh
      size_t tetradims[4] = {22,44,70,88};
      dimensions.assign(tetradims,tetradims+4);
      run(TETRA,dimensions,results,sep);

      results <<"Max Depth " << sep << maxdepth[d] << sep << " Max Cell Per Tile " << sep << maxcellptile[c]<< endl;

      results << endl;
    }

  (*output) << results.str();

  if (argc>1)
    delete output;

  MiMeshViz::finish();
  return 0;
}

void run(MeshType type, vector<size_t>& dims, ostringstream& results, const char* sep)
{
  for (size_t d = 0; d < dims.size(); d++)
  {
    probeMesh(type,dims[d],MbVec3d(0),MbVec3d(100),0.0,results, sep);
    cout << endl;
  }
}

double
probeMesh(MeshType type, size_t dim, const MbVec3d& minBB, const MbVec3d& maxBB, double rotation, ostringstream& result, const char* sep)
{
  double rotations[4] = {0, 0, 0, 0};
  rotations[3] = rotation;
  const MiVolumeMeshUnstructured& mesh = meshBuilder.getMesh(type,dim,minBB,maxBB,rotations);
  cout << meshBuilder.getMeshType() << " Mesh" << endl;
  cout << "mesh built, num cells=" << mesh.getTopology().getNumCells() 
    << ", num nodes=" << mesh.getTopology().getEndNodeId() << endl;
  
  MbVec3d startP = mesh.getGeometry().getCoord(0);
  size_t numExtract = dim*dim*dim;
  MbVec3d delta = (maxBB-minBB)/double(dim);
  MbVec3d u(delta[0],0,0), v(0,delta[1],0), w(0,0,delta[3]);
  if(rotation>0.0)
  {
    double cosa = cos(rotation);
    double sina = sin(rotation);
    u[0] = delta[0]*cosa;
    u[1] = delta[0]*sina;
    v[0] = - delta[1]*sina;
    v[1] = delta[1]*cosa;
    delta[0] = u[0] + v[0];
    delta[1] = u[1] + v[1];
  }
  MbVec3d point, spoint;
  point = spoint = startP + delta*0.5;

  double tb = localTime.getElapsed();
  // Get an instance of a probe according to the mesh type
  MiPointProbeUnstructured* probe = MiPointProbeUnstructured::getNewInstance(mesh);
  double tc = localTime.getElapsed();

  // first probe not benched to ignore initial memory alloc
  probe->setLocation(point);
  
  double t1 = localTime.getElapsed();
#ifndef INITONLY
  cout << "start probing " << numExtract << " points" << endl;
  for (size_t i=0; i<dim; ++i)
  {
    for (size_t j=0; j<dim; ++j)
    {
      for (size_t k=0; k<dim; ++k)
      {
        probe->setLocation(point);
        point[2] +=  delta[2];
      }
      point.setValue(point[0]+v[0],point[1]+v[1],spoint[2]);
    }
    spoint += u;
    point = spoint;
    cout << ".";
  }
  cout << endl;
#endif
  double t2 = localTime.getElapsed();

  result << setw(10) << meshBuilder.getMeshType() << sep << setw(10) << mesh.getTopology().getNumCells()  << sep  << setw(15)<< numExtract <<  sep;
  result << setw(14) << tc-tb;
#ifndef INITONLY
  result << sep << setw(18) << t2-t1 << sep  <<setw(25) << 10000*(t2-t1)/float(numExtract) << endl;
#else
  result << endl;  
#endif

  delete probe;
  return t2-t1;
  }


#ifdef _MSC_VER
#pragma warning( pop )
#endif
