#ifndef  _QuadraticMesh_h
#define  _QuadraticMesh_h

#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>
#include <MeshVizXLM/mesh/data/MiDataSetI.h>
#include <MeshVizXLM/MxTimeStamp.h>

#include "QuadraticHexahedronCell.h"

#include <Inventor/STL/vector>

static double s_radiusMin = 5;
static double s_radiusMax = 8;

//-----------------------------------------------------------------------------
class QuadraticGeometry : public MiGeometryI
{
public:
  QuadraticGeometry(double width)
  {
    updateCoords(width);
  }

  void updateCoords(double width)
  {
    double c1 = s_radiusMin * sqrt(2.)/2.;
    double c3 = s_radiusMax * sqrt(2.)/2.;

    m_coords.assign(48,MbVec3d(0,0,0));

    m_coords[0] = MbVec3d(  s_radiusMin, 0, -width);     
    m_coords[1] = MbVec3d(  s_radiusMax, 0, -width); 
    m_coords[2] = MbVec3d(  s_radiusMax, 0, width); 
    m_coords[3] = MbVec3d(  s_radiusMin, 0, width);

    m_coords[4] = MbVec3d(  0, s_radiusMin, -width);
    m_coords[5] = MbVec3d(  0, s_radiusMax, -width);
    m_coords[6] = MbVec3d(  0, s_radiusMax, width);
    m_coords[7] = MbVec3d(  0, s_radiusMin, width);

    m_coords[8] =  (m_coords[0]+m_coords[1])/2;    m_coords[8][2] *= 1.25;
    m_coords[9] =  (m_coords[1]+m_coords[2])/2;    m_coords[9][0] *= 1.1;
    m_coords[10] =  (m_coords[2]+m_coords[3])/2;   m_coords[10][2] *= 1.25;
    m_coords[11] =  (m_coords[0]+m_coords[3])/2;   m_coords[11][0] *= 1.05;

    m_coords[12] =  (m_coords[4]+m_coords[5])/2;   m_coords[12][2] *= 1.25;
    m_coords[13] =  (m_coords[5]+m_coords[6])/2;   m_coords[13][1] *= 1.1;
    m_coords[14] =  (m_coords[6]+m_coords[7])/2;   m_coords[14][2] *= 1.25;
    m_coords[15] =  (m_coords[7]+m_coords[4])/2;   m_coords[15][1] *= 1.05;

    m_coords[16] = MbVec3d(c1,c1,-width); 
    m_coords[17] = MbVec3d(c3,c3,-width);
    m_coords[18] = MbVec3d(c3,c3,width); 
    m_coords[19] = MbVec3d(c1,c1,width); 

    // add symetric points
    m_coords[20] = getSymetricToXPlane(0) ;
    m_coords[21] = getSymetricToXPlane(1) ; 
    m_coords[22] = getSymetricToXPlane(2) ; 
    m_coords[23] = getSymetricToXPlane(3) ; 

    m_coords[24] = getSymetricToXPlane(8) ; 
    m_coords[25] = getSymetricToXPlane(9) ; 
    m_coords[26] = getSymetricToXPlane(10); 
    m_coords[27] = getSymetricToXPlane(11);

    m_coords[28] = getSymetricToXPlane(16); 
    m_coords[29] = getSymetricToXPlane(17); 
    m_coords[30] = getSymetricToXPlane(18); 
    m_coords[31] = getSymetricToXPlane(19);

    m_coords[32] = getSymetricToYPlane(4);
    m_coords[33] = getSymetricToYPlane(5); 
    m_coords[34] = getSymetricToYPlane(6); 
    m_coords[35] = getSymetricToYPlane(7); 

    m_coords[36] = getSymetricToYPlane(12);
    m_coords[37] = getSymetricToYPlane(13); 
    m_coords[38] = getSymetricToYPlane(14); 
    m_coords[39] = getSymetricToYPlane(15); 

    m_coords[40] = getSymetricToYPlane(28);
    m_coords[41] = getSymetricToYPlane(29); 
    m_coords[42] = getSymetricToYPlane(30); 
    m_coords[43] = getSymetricToYPlane(31); 

    m_coords[44] = getSymetricToYPlane(16);
    m_coords[45] = getSymetricToYPlane(17); 
    m_coords[46] = getSymetricToYPlane(18); 
    m_coords[47] = getSymetricToYPlane(19); 

    m_timeStamp = MxTimeStamp::getTimeStamp();
  }

  const std::vector<MbVec3d>& getCoords() const
  {
    return m_coords;
  }

  virtual MbVec3d getCoord(size_t i) const 
  {
    return m_coords[i];
  }

  virtual size_t getTimeStamp() const 
  {
    return m_timeStamp;
  }

  MbVec3d getSymetricToXPlane(size_t nodeId) const
  {
    MbVec3d pt = m_coords[nodeId];
    return MbVec3d(-pt[0],pt[1],pt[2]);
  }

  MbVec3d getSymetricToYPlane(size_t nodeId) const
  {
    MbVec3d pt = m_coords[nodeId];
    return MbVec3d(pt[0],-pt[1],pt[2]);
  }

private:
  std::vector<MbVec3d> m_coords;
  size_t m_timeStamp;
};

//-----------------------------------------------------------------------------
class QuadraticTopology : public MiVolumeTopologyExplicitI
{
public:
  QuadraticTopology() 
    : m_cell0(0,1,2,3,4,5,6,7, 8,9,10,11,12,13,14,15,16,17,18,19),
      m_cell1(4,5,6,7, 20,21,22,23, 12,13,14,15, 24,25,26,27, 28,29,30,31),
      m_cell2(20,21,22,23, 32,33,34,35, 24,25,26,27, 36,37,38,39, 40,41,42,43),
      m_cell3(32,33,34,35, 0,1,2,3, 36,37,38,39, 8,9,10,11, 44,45,46,47)      
  {
  }

  virtual const MiVolumeCell* getCell(size_t id) const
  {
    switch (id)
    {
    default:
    case 0: return &m_cell0;
    case 1: return &m_cell1;
    case 2: return &m_cell2;
    case 3: return &m_cell3;
    }
  }

  virtual size_t getEndNodeId() const
  {
    return 48;
  }

  virtual size_t getNumCells() const
  {
    return 4;
  }

  virtual size_t getTimeStamp() const 
  {
    return 1;
  }

private:
  QuadraticHexahedronCell m_cell0;
  QuadraticHexahedronCell m_cell1;
  QuadraticHexahedronCell m_cell2;
  QuadraticHexahedronCell m_cell3;
};


//-----------------------------------------------------------------------------
class QuadraticMesh : public MiVolumeMeshUnstructured
{
public:
  QuadraticMesh(double width) : m_geometry(width) {}

  virtual const MiVolumeTopologyExplicitI& getTopology() const 
  {
    return m_topology;
  }

  virtual const MiGeometryI& getGeometry() const 
  {
    return m_geometry;
  }

  const std::vector<MbVec3d>& getCoords() const
  {
    return m_geometry.getCoords();
  }

  void update(double width)
  {
    m_geometry.updateCoords(width);
  }

private:
  QuadraticGeometry m_geometry;
  QuadraticTopology m_topology;
};

//-----------------------------------------------------------------------------
class ScalarSet : public MiScalardSetI, public MbScalardSet
{
public:
  ScalarSet(const MiVolumeMeshUnstructured& mesh) 
    : m_geometry(mesh.getGeometry())
  {
    m_vmin = 1E30;
    m_vmax = -1E30;
    for (size_t i=mesh.getTopology().getBeginNodeId(); i!=mesh.getTopology().getEndNodeId(); ++i)
    {
      double v = get(i);
      if (v < m_vmin) m_vmin = v;
      if (v > m_vmax) m_vmax = v;
    }
  }
  double get(size_t i) const 
  { 
    // value = distance of point i to Z-axis
    MbVec3d coord = m_geometry.getCoord(i);
    return  sqrt(coord[0]*coord[0] + coord[1]*coord[1]);
  }

  double getMin () const { return m_vmin; }
  double getMax () const { return m_vmax; }
  size_t  getTimeStamp () const  { return 1; }
  std::string getName () const  { return "testData"; }
  DataBinding  getBinding () const { return PER_NODE; }

private:
  const MiGeometryI& m_geometry;
  double m_vmin,m_vmax;
};


#endif 


