#include <MeshVizXLM/MiMeshViz.h>

#include <MeshVizXLM/extractors/MiSkinExtractUnstructured.h>
#include <MeshVizXLM/tessellator/MxEdgeErrorMetricGeometry.h>

#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSetI.h>
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSurface.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoMaterial.h>

#include <Inventor/events/SoKeyboardEvent.h> 
#include <Inventor/nodes/SoEventCallback.h>

#include <data/MbDataSet.h>
#include "QuadraticMesh.h"

using namespace std;

#define NAMESTR "MeshVizXLM quadratic skin extraction"

void viewMesh(const MbScalardSet& volumeScalarSet, 
              const MiSurfaceMeshUnstructured& basicSkin, const MeXScalardSetI& basicDataSet, 
              const MiSurfaceMeshUnstructured& tessellatedSkin, const MeXScalardSetI& tessellatedDataSet);
void keyboardCallback(void *userData, SoEventCallback *eventCB);

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4250)
#endif

SbElapsedTime localTime;
SoSwitch* s_basicSwitch;
SoSwitch* s_tessellatedSwitch;

MoMesh* s_tessellatedMeshNode;

double s_maxError = 0.2;
double s_minError = 0.0005;
double s_incrError = (s_maxError-s_minError)/30;

double s_metricError = (s_maxError+s_minError)/2;

MxEdgeErrorMetricGeometry s_edgeErrorMetricGeometry(s_metricError);
MiTessellator* s_tessellator;
MiSkinExtractUnstructured* s_basicSkinExtract;
MiSkinExtractUnstructured* s_skinExtract;
MoScalarSetI* s_tessellatedScalarSetNode;

static double s_width = 2;
QuadraticMesh s_volumeMesh(s_width);
ScalarSet s_volumeScalarSet(s_volumeMesh);

//-----------------------------------------------------------------------------
void updateTessellation(double metricError)
{
  s_basicSkinExtract->extractSkin();
  s_basicSkinExtract->extractScalarSet(s_volumeScalarSet);
  cout << "updateTessellation, tolerance = " << metricError;
  double t1 = localTime.getElapsed();
  const MiSurfaceMeshUnstructured& tessellatedSkin = s_skinExtract->extractSkin();
  double t2 = localTime.getElapsed();
  s_tessellatedMeshNode->setMesh(&tessellatedSkin); 
  const MeXScalardSetI& skinScalarSet = s_skinExtract->extractScalarSet(s_volumeScalarSet);
  s_tessellatedScalarSetNode->setScalarSet(&skinScalarSet);
  cout << ", num generated polygones " << tessellatedSkin.getTopology().getNumCells() ;
  cout << ", in " << (t2-t1)*1000 << " ms" << endl;
}

//-----------------------------------------------------------------------------
void
keyboardCallback(void*, SoEventCallback *eventCB)
{
  bool handled = false;

  const SoEvent *ev = eventCB->getEvent();

  if (SO_KEY_PRESS_EVENT(ev, T) )
  {
    s_basicSwitch->whichChild = (s_basicSwitch->whichChild.getValue() == SO_SWITCH_NONE) 
      ? SO_SWITCH_ALL 
      : SO_SWITCH_NONE;
    
    s_tessellatedSwitch->whichChild = (s_tessellatedSwitch->whichChild.getValue() == SO_SWITCH_NONE) 
      ? SO_SWITCH_ALL 
      : SO_SWITCH_NONE;
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, D)) 
  {
    if (s_metricError+s_incrError < s_maxError)
    {
      s_metricError += s_incrError;
      s_edgeErrorMetricGeometry.setMaxError(s_metricError);
      updateTessellation(s_metricError);
    }
    handled = true;
  } 
  else if (SO_KEY_PRESS_EVENT(ev, I))
  {
    if (s_metricError-s_incrError > s_minError)
    {
      s_metricError -= s_incrError;
      s_edgeErrorMetricGeometry.setMaxError(s_metricError);
      updateTessellation(s_metricError);
    }
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, P))
  {
    ++s_width;
    s_volumeMesh.update(s_width);
    updateTessellation(s_metricError);
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, M))
  {
    --s_width;
    s_volumeMesh.update(s_width);
    updateTessellation(s_metricError);
    handled = true;
  }

  if (handled) 
    eventCB->setHandled();

}

//-----------------------------------------------------------------------------
int
main(int, char **)
{
  MiMeshViz::init();

  cout << "T to toggle quadratic tesselation " << std::endl;
  cout << "I to increase tesselation (only when quadratic tesselation is on) " << std::endl;
  cout << "D to decrease tesselation (only when quadratic tesselation is on)mesh" << std::endl;

  MiTessellator* basicTessellator = MiTessellator::getNewTessellatorBasic();
  s_basicSkinExtract = MiSkinExtractUnstructured::getNewInstance(s_volumeMesh,false,basicTessellator);
  const MiSurfaceMeshUnstructured& basicSkin = s_basicSkinExtract->extractSkin();
  const MeXScalardSetI& basicSkinScalarSet = s_basicSkinExtract->extractScalarSet(s_volumeScalarSet);


  s_tessellator = MiTessellator::getNewTessellatorGeometry(s_edgeErrorMetricGeometry);

  double t1 = localTime.getElapsed();
  s_skinExtract = MiSkinExtractUnstructured::getNewInstance(s_volumeMesh,false,s_tessellator);
  const MiSurfaceMeshUnstructured& skin = s_skinExtract->extractSkin();
  const MeXScalardSetI& skinScalarSet = s_skinExtract->extractScalarSet(s_volumeScalarSet);

  double t2 = localTime.getElapsed();

  cout << "Time to extract skin:" << (t2-t1) << endl;

  viewMesh(s_volumeScalarSet, basicSkin,basicSkinScalarSet, skin,skinScalarSet);
  //viewMesh(s_volumeScalarSet, skin,skinScalarSet, skin,skinScalarSet);

  MiMeshViz::finish();

  return 0;
}


//-----------------------------------------------------------------------------
void viewMesh(const MbScalardSet& volumeScalarSet, 
              const MiSurfaceMeshUnstructured& basicSkin, const MeXScalardSetI& basicDataSet, 
              const MiSurfaceMeshUnstructured& tessellatedSkin, const MeXScalardSetI& tessellatedDataSet) 
{
  // Init viewer
  Widget my_window = SoXt::init(NAMESTR) ;
  if (my_window == NULL) exit(1) ;

  MoMeshViz::init();

  //  ----- basic skin -------
  MoMesh* basicMeshNode = new MoMesh;
  basicMeshNode->setMesh(&basicSkin); 

  MoScalarSetI* basicScalarSetNode = new MoScalarSetI;
  basicScalarSetNode->setScalarSet(&basicDataSet);

  SoMaterial* basicMaterial = new SoMaterial;
  basicMaterial->diffuseColor = SbColor(0,0,1);

  MoMeshSurface *basicSkinSurface = new MoMeshSurface;

  SoSeparator* basicSep = new SoSeparator;

  s_basicSwitch = new SoSwitch;
  s_basicSwitch->whichChild = SO_SWITCH_ALL;

  // ----- tessellated skin -----
  s_tessellatedMeshNode = new MoMesh;
  s_tessellatedMeshNode->setMesh(&tessellatedSkin); 

  s_tessellatedScalarSetNode = new MoScalarSetI;
  s_tessellatedScalarSetNode->setScalarSet(&tessellatedDataSet);

  SoMaterial* tessellatedMaterial = new SoMaterial;
  tessellatedMaterial->diffuseColor = SbColor(1,0,0);

  MoMeshSurface *tessellateSkinSurface = new MoMeshSurface;

  SoSeparator* tessellatedSep = new SoSeparator;

  s_tessellatedSwitch = new SoSwitch;
  s_tessellatedSwitch->whichChild = SO_SWITCH_NONE;

  // ----- color mapping -----
  MoPredefinedColorMapping* colorMapping = new MoPredefinedColorMapping;
  colorMapping->predefColorMap = MoPredefinedColorMapping::STANDARD;
  colorMapping->minValue = float(volumeScalarSet.getMin());
  colorMapping->maxValue = float(volumeScalarSet.getMax());

  SoShapeHints* shapeHints = new SoShapeHints;
  shapeHints->creaseAngle = 1.0;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE  ;

  SoEventCallback *myCallbackNode = new SoEventCallback;
  myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(),keyboardCallback,NULL); 

  SoSeparator* sep = new SoSeparator;
  sep->ref();
  {
    sep->addChild(shapeHints);
    sep->addChild(colorMapping);

    sep->addChild(basicSep);
    {
      basicSep->addChild(s_basicSwitch);
      {
        s_basicSwitch->addChild(basicScalarSetNode);
        s_basicSwitch->addChild(basicMeshNode);
        s_basicSwitch->addChild(basicSkinSurface);
      }
    }

    sep->addChild(tessellatedSep);
    {
      tessellatedSep->addChild(s_tessellatedSwitch);
      {
        s_tessellatedSwitch->addChild(s_tessellatedScalarSetNode);
        s_tessellatedSwitch->addChild(tessellatedMaterial);
        s_tessellatedSwitch->addChild(s_tessellatedMeshNode);
        s_tessellatedSwitch->addChild(tessellateSkinSurface);
      }
    }
    sep->addChild(myCallbackNode);
  }

  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(my_window);
  viewer->setSceneGraph(sep);
  viewer->show();
  viewer->viewAll();

  // dialog
  SoXt::show(my_window);
  SoXt::mainLoop();

  delete viewer;
  sep->unref();
  MoMeshViz::finish();
  SoXt::finish();
}

#ifdef _MSC_VER
#pragma warning( pop ) 
#endif

