#include "ClipLine.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/details/SoCubeDetail.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDepthOffset.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTranslation.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <MeshVizXLM/extrmesh/MeXSurfaceMeshUnstructured.h>
#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshClipLine.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSurface.h>
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSet.h>

//---------------------------------------------------------------------
ClipLine::ClipLine()
{
  m_move = SbVec3f( .1f, .1f, .0f );
}

//---------------------------------------------------------------------
ClipLine::~ClipLine()
{
  delete m_mesh;
}

//---------------------------------------------------------------------
bool
ClipLine::displayInstructions() const
{
  std::cout << "Press left mouse button to add a new point to the polyline" << std::endl;
  std::cout << "Double click to clear the current polyline and start a new one" << std::endl;
  std::cout << "Press k/l to move the clip line direction" << std::endl;
  std::cout << "Press e to toggle surface edges visibility" << std::endl;
  std::cout << "Press v to toggle surface faces visibility" << std::endl;
  std::cout << "Press c to enable/disable surface faces coloring" << std::endl;
  return true;
}

//------------------------------------------------------------------------
void
ClipLine::onMousePressed( SoEventCallback* eventCB )
{
  bool isDoubleClick = SoMouseButtonEvent::isButtonDoubleClickEvent( eventCB->getEvent(), SoMouseButtonEvent::BUTTON1 );
  if ( isDoubleClick )
  {
    // clear the existing clip line
    m_clipline->polyline.setNum( 0 );
  }
  else if ( SoMouseButtonEvent::isButtonPressEvent( eventCB->getEvent(), SoMouseButtonEvent::BUTTON1 ) )
  {
    const SoPickedPoint* pickedPoint = eventCB->getPickedPoint();
    if ( pickedPoint != NULL )
    {
      const SoDetail* detail = pickedPoint->getDetail();
      // check if the transparent cube has been picked
      if ( detail->isOfType( SoCubeDetail::getClassTypeId() ) )
      {
        SbVec3f pickedPointCoord = pickedPoint->getPoint();
        if ( m_clipline->polyline.getNum() == 0 )
        {
          // the first point has been picked: we use the normal of the first
          // picked face on the bounding box as the direction of the clip line.
          SoCubeDetail* cubedetail = ( SoCubeDetail* )detail;
          SbVec3f lineDirection;
          switch ( cubedetail->getPart() )
          {
          case 0:
          case 1:
            lineDirection.setValue( 0, 0, 1 );
            break;
          case 2:
          case 3:
            lineDirection.setValue( 1, 0, 0 );
            break;
          case 4:
          case 5:
            lineDirection.setValue( 0, 1, 0 );
            break;
          }
          m_clipline->direction = lineDirection;
        }
        m_clipline->polyline.set1Value( m_clipline->polyline.getNum(), pickedPointCoord );
      }
    }
  }
}

//---------------------------------------------------------------------
void
ClipLine::onKeyPressed( SoEventCallback* eventCB )
{
  const SoEvent* ev = eventCB->getEvent();

  if ( SO_KEY_PRESS_EVENT( ev, E ) )
  {
    m_ds->displayEdges = !m_ds->displayEdges.getValue();
    eventCB->setHandled();
  }
  if ( SO_KEY_PRESS_EVENT( ev, V ) )
  {
    m_ds->displayFaces = !m_ds->displayFaces.getValue();
    eventCB->setHandled();
  }
  if ( SO_KEY_PRESS_EVENT( ev, C ) )
  {
    m_mat->faceColoring = m_mat->faceColoring.getValue() == MoMaterial::COLOR ? MoMaterial::CONTOURING : MoMaterial::COLOR;
    eventCB->setHandled();
  }
  if ( SO_KEY_PRESS_EVENT( ev, L ) )
  {
    m_clipline->direction = m_clipline->direction.getValue() + m_move;
    eventCB->setHandled();
  }
  if ( SO_KEY_PRESS_EVENT( ev, K ) )
  {
    m_clipline->direction = m_clipline->direction.getValue() - m_move;
    eventCB->setHandled();
  }
}

//---------------------------------------------------------------------
SoSeparator*
ClipLine::buildSceneGraph()
{
  SoSeparator* root = new SoSeparator;

  SoSeparator* scene3D = new SoSeparator;
  scene3D->setName( "Scene_3D" );
  root->addChild( scene3D );

  SoPerspectiveCamera* camera3D = new SoPerspectiveCamera;
  scene3D->addChild( camera3D );

  MoMesh* mesh = new MoMesh;
  mesh->setMesh( m_mesh );
  scene3D->addChild( mesh );

  MoPredefinedColorMapping* colorMapNode = new MoPredefinedColorMapping;
  colorMapNode->predefColorMap = MoPredefinedColorMapping::STANDARD;
  scene3D->addChild( colorMapNode );

  if ( m_mesh->getNumScalarSets() > 0 )
  {
    const MbScalardSetIj* scalarSet = m_mesh->getScalarSet( 0 );
    colorMapNode->minValue = ( float )scalarSet->getMin();
    colorMapNode->maxValue = ( float )scalarSet->getMax();
    MoScalarSet* sset = new MoScalarSet;
    sset->setScalarSet( scalarSet );
    scene3D->addChild( sset );
  }

  SoShapeHints* shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  scene3D->addChild( shapeHints );

  m_ds = new MoDrawStyle;
  m_ds->displayFaces = true;
  m_ds->displayEdges = true;
  scene3D->addChild( m_ds );

  m_mat = new MoMaterial;
  m_mat->faceColor = SbColor( 0.8f, 0.8f, 0.8f );
  m_mat->lineColor = SbColor( 0.f, 0.f, 0.f );
  m_mat->faceColoring = MoMaterial::COLOR;
  m_mat->lineColoring = MoMaterial::COLOR;
  scene3D->addChild( m_mat );

  MoMeshSurface* surface = new MoMeshSurface;
  surface->colorScalarSetId = 0;
  scene3D->addChild( surface );

  // Transparent cube
  SoSeparator* cubeSep = new SoSeparator;
  SoMaterial* cubeMaterial = new SoMaterial;
  cubeMaterial->transparency = 0.9f;
  cubeMaterial->diffuseColor = SbColor( 0, 1, 1 );
  SoTranslation* cubePos = new SoTranslation;
  const MiGeometry& geometry = m_mesh->getGeometry();
  MbVec3d size = geometry.getMax() - geometry.getMin();
  MbVec3d center = size / 2.0 + geometry.getMin();
  cubePos->translation.setValue( SbVec3f( ( float )center[0], ( float )center[1], ( float )center[2] ) );
  float width = ( float )size[0];
  float height = ( float )size[1];
  float depth = size[2] != 0 ? ( float )size[2] : 2.f;
  SoCube* cube = new SoCube;
  cube->width = width;
  cube->height = height;
  cube->depth = depth;
  cubeSep->addChild( cubeMaterial );
  cubeSep->addChild( cubePos );
  cubeSep->addChild( cube );

  SoDrawStyle* cubeDrawStyle = new SoDrawStyle;
  cubeDrawStyle->style = SoDrawStyle::LINES;
  cubeDrawStyle->lineWidth = 5;
  cubeSep->addChild( cubeDrawStyle );
  cubeSep->addChild( cube );

  scene3D->addChild( cubeSep );

  // default clip line
  float topZcoord = ( float )center[2] + depth / 2.f;
  SoDepthOffset* lineOffset = new SoDepthOffset;
  SoDrawStyle* lineDS = new SoDrawStyle;
  lineDS->lineWidth = 2;
  MoMaterial* mat = new MoMaterial;
  mat->lineColoring = MoMaterial::CONTOURING;
  m_cliplineDS = new MoDrawStyle;
  m_cliplineDS->displayEdges = true;
  m_clipline = new MoMeshClipLine;
  m_clipline->polyline.setNum( 6 );
  m_clipline->polyline.set1Value( 0, SbVec3f( 0.2f * ( float )size[0], 0.2f * ( float )size[1], topZcoord ) );
  m_clipline->polyline.set1Value( 1, SbVec3f( 0.2f * ( float )size[0], 0.3f * ( float )size[1], topZcoord ) );
  m_clipline->polyline.set1Value( 2, SbVec3f( 0.3f * ( float )size[0], 0.4f * ( float )size[1], topZcoord ) );
  m_clipline->polyline.set1Value( 3, SbVec3f( 0.5f * ( float )size[0], 0.3f * ( float )size[1], topZcoord ) );
  m_clipline->polyline.set1Value( 4, SbVec3f( 0.7f * ( float )size[0], 0.8f * ( float )size[1], topZcoord ) );
  m_clipline->polyline.set1Value( 5, SbVec3f( 0.9f * ( float )size[0], 0.5f * ( float )size[1], topZcoord ) );
  m_clipline->direction.setValue( SbVec3f( 0, 0, 1 ) );

  scene3D->addChild( mat );
  scene3D->addChild( lineOffset );
  scene3D->addChild( lineDS );
  scene3D->addChild( m_cliplineDS );
  scene3D->addChild( m_clipline );

  // polyline
  SoVertexProperty* vertexProp = new SoVertexProperty;
  vertexProp->vertex.connectFrom( &m_clipline->polyline );

  SoLineSet* polyline = new SoLineSet;
  polyline->vertexProperty.setValue( vertexProp );

  SoMaterial* polylineMaterial = new SoMaterial;
  polylineMaterial->diffuseColor = SbColor( 1, 0, 0 );

  scene3D->addChild( polylineMaterial );
  scene3D->addChild( polyline );

  // Mouse Pressed
  SoEventCallback* mousePressedCallbackNode = new SoEventCallback;
  mousePressedCallbackNode->addEventCallback<SoMouseButtonEvent>( []( void* _this, SoEventCallback* eventCB )
  {
    static_cast<ClipLine*>( _this )->onMousePressed( eventCB );
  }, this );
  root->addChild( mousePressedCallbackNode );

  // Key Pressed
  SoEventCallback* keyPressedCallbackNode = new SoEventCallback;
  keyPressedCallbackNode->addEventCallback<SoKeyboardEvent>( []( void* _this, SoEventCallback* eventCB )
  {
    static_cast<ClipLine*>( _this )->onKeyPressed( eventCB );
  }, this );
  root->addChild( keyPressedCallbackNode );

  return root;
}

//---------------------------------------------------------------------
bool
ClipLine::readMesh( const SbString& topoFilename, const SbString& altitudeFilename, const SbString& scalarsetFilename )
{
  int nx, ny;
  float xmin, xmax;
  float ymin, ymax;

  // read topology
  std::ifstream topoFile( topoFilename.toStdString(), std::ifstream::in );
  if ( !topoFile )
  {
    std::cerr << "Fail to open topology file." << std::endl;
    return false;
  }

  try
  {
    std::string line;
    // read surface dimensions
    {
      std::getline( topoFile, line );
      std::stringstream stream( line );
      stream >> nx >> ny;
    }
    // read xmin and xmax
    {
      std::getline( topoFile, line );
      std::stringstream stream( line );
      stream >> xmin >> xmax;
    }
    // read ymin and ymax
    {
      std::getline( topoFile, line );
      std::stringstream stream( line );
      stream >> ymin >> ymax;
    }
  }
  catch ( const std::exception& exc )
  {
    std::cerr << "Fail to read topology file: " << exc.what() << std::endl;
    return false;
  }

  // read altitudes
  std::vector<float> altitudeValues;
  std::ifstream altitudeFile( altitudeFilename.toStdString(), std::ifstream::in );
  if ( altitudeFile )
  {
    std::string line;
    size_t i = 0;
    size_t numVal = nx * ny;
    float scale = 0.5f;
    altitudeValues.resize( numVal );
    try
    {
      while ( i < numVal && std::getline( altitudeFile, line, ' ' ) )
      {
        if ( !line.empty() )
          altitudeValues[i++] = std::stof( line ) * scale;
      }
    }
    catch ( const std::exception& exc )
    {
      std::cerr << "Fail to read altitudes file: " << exc.what() << std::endl;
      return false;
    }
  }

  // read scalar set
  MbScalardSetIj* sset = nullptr;
  std::ifstream ssetFile( scalarsetFilename.toStdString(), std::ifstream::in );
  if ( ssetFile )
  {
    std::string line;
    size_t i = 0;
    size_t numVal = nx * ny;
    sset = new MbScalardSetIj( nx, ny, "ScalarSet", MiDataSet::PER_NODE );
    try
    {
      while ( i < numVal && std::getline( ssetFile, line, ' ' ) )
      {
        if ( !line.empty() )
        {
          sset->set( i % nx, i / nx, std::stof( line ) );
          i++;
        }
      }
    }
    catch ( const std::exception& exc )
    {
      std::cerr << "Fail to read scalar set file: " << exc.what() << std::endl;
      return false;
    }
  }

  m_mesh = new MbSurfaceMeshCurvilinear<MbVec3f, double, MbVec3f>( nx - 1, ny - 1 );
  MbVec3f origin( xmin, ymin, 0 );
  MbVec3f stepI( ( xmax - xmin ) / ( nx - 1 ), 0, 0 );
  MbVec3f stepJ( 0, ( ymax - ymin ) / ( ny - 1 ), 0 );
  MbVec3f coord;
  for ( size_t j = 0; j < ny; j++ )
    for ( size_t i = 0; i < nx; i++ )
    {
      coord = origin + ( float )i * stepI + ( float )j * stepJ;
      if ( !altitudeValues.empty() )
        coord[2] = altitudeValues[i + j * nx];
      m_mesh->setCoord( i, j, coord );
    }

  if ( sset != nullptr )
    m_mesh->addScalarSet( sset );

  return true;
}

//---------------------------------------------------------------------
int
ClipLine::run()
{
  Widget my_window = SoXt::init( "MeshVizXLM ClipLine" );
  if ( my_window == NULL )
    exit( 1 );

  displayInstructions();

  MoMeshViz::init();

  SbString dataFilePath = SbFileHelper::expandString( "$OIVHOME/examples/data/MeshViz/" );
  SbString topologyFilename = dataFilePath + "REGULARGRID.TOPO";
  SbString altitudesFilename = dataFilePath + "GRID3.DAT";
  SbString datasetFilename = dataFilePath + "GRID1.DAT";
  SoSeparator* root;

  if ( readMesh( topologyFilename, altitudesFilename, datasetFilename ) )
    root = buildSceneGraph();
  else
    root = new SoSeparator;

  SoXtExaminerViewer* viewer = new SoXtExaminerViewer( my_window );
  viewer->setSize( SbVec2s( 1024, 768 ) );
  viewer->setSceneGraph( root );
  viewer->setTitle( "MeshVizXLM ClipLine" );
  viewer->show();
  viewer->viewAll();
  viewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );

  // dialog
  SoXt::show( my_window );
  SoXt::mainLoop();

  delete viewer;

  MoMeshViz::finish();
  SoXt::finish();
  return 0;
}
