#include "Demo.h"
#include "MeshSceneEditor.h"

#include <Inventor/lock/SoLicensesInfo.h>

#include <MeshVizXLM/mapping/MoMeshViz.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoKeyboardEvent.h> 
#include <Inventor/events/SoLocation2Event.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
using namespace std;

//---------------------------------------------------------------------
void
keyboardCallback(void *_this, SoEventCallback *eventCB)
{
  if (SO_KEY_PRESS_EVENT(eventCB->getEvent(), M))
  {
    ((Demo*)_this)->displayMenu();
     eventCB->setHandled();
  }
  else
  {
    ((Demo*)_this)->editInputMesh(eventCB);
    ((Demo*)_this)->keyPressed(eventCB);
  }
}

//------------------------------------------------------------------------
// Function called each time the mouse is moved in the viewer window
void mouseMovedCallback(void *_this, SoEventCallback *eventCB) 
{
  ((Demo*)_this)->mouseMoved(eventCB);
}

//------------------------------------------------------------------------
// Function called each time the mouse button is pressed in the viewer window
void mousePressedCallback(void *_this, SoEventCallback *eventCB) 
{
  ((Demo*)_this)->mousePressed(eventCB);
}

//---------------------------------------------------------------------
Demo::Demo(const string& demoTitle, MeshType meshType, SwitchMeshType switchMeshType, 
           bool toggleFiltering, MiDataSet::DataBinding binding, const MbVec3<size_t>& meshSize) :
 MeshData(meshType,meshSize,binding),
 m_demoTitle(demoTitle),
 m_meshEditor(NULL),
 m_switchMeshType(switchMeshType),
 m_toggleFiltering(toggleFiltering)
{
}

 //---------------------------------------------------------------------
Demo::Demo(const string& demoTitle) :
 MeshData(UNDEFINED_MESH,s_defaultMeshSize,MiDataSet::PER_NODE),
 m_demoTitle(demoTitle),
 m_meshEditor(NULL),
 m_switchMeshType(NONE),
 m_toggleFiltering(false)
{
}

//---------------------------------------------------------------------
Demo::~Demo()
{
  delete m_meshEditor;
}

//-----------------------------------------------------------------------------
int
Demo::run()
{
  Widget my_window = SoXt::init(m_demoTitle.c_str()) ;
  if (my_window == NULL) exit(1) ;

  std::cout << SoLicensesInfo::getInstance().getVersion() << std::endl;
  displayMenu();

  MoMeshViz::init();

  SoNode* root = buildSceneGraph();

  m_viewer = new SoXtExaminerViewer(my_window);
  m_viewer->setSize(SbVec2s(1024,768));
  m_viewer->setSceneGraph(root);
  m_viewer->setTitle(m_demoTitle.c_str());
  m_viewer->show();
  m_viewer->viewAll();
  m_viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);

  // dialog
  SoXt::show(my_window);
  SoXt::mainLoop();

  delete m_viewer;

  MoMeshViz::finish();
  SoXt::finish();
  return 0;
}

//-----------------------------------------------------------------------------
SoNode* 
Demo::buildSceneGraph()
{
  SoSeparator* root = new SoSeparator;
  
  // 3D view
  SoSeparator* scene3D = new SoSeparator;
  scene3D->setName("Scene_3D");
  root->addChild(scene3D);

  SoPerspectiveCamera *camera3D = new SoPerspectiveCamera ;
  scene3D->addChild(camera3D);
  
  insertInputMesh(scene3D);
  insertRepresentations(scene3D);

  // 2D view
  SoSeparator* scene2D = new SoSeparator;
  scene2D->setName("Scene_2D");
  root->addChild(scene2D);

  SoOrthographicCamera *camera2D = new SoOrthographicCamera ;
  camera2D->viewportMapping = SoCamera::LEAVE_ALONE;
  scene2D->addChild(camera2D);
  
  insertScene2D(scene2D);

  // Add Callbacks
  // Mouse Moved
  SoEventCallback *mouseMovedCallbackNode= new SoEventCallback;
  mouseMovedCallbackNode->setName("Mouse_Moved");
  mouseMovedCallbackNode->addEventCallback(SoLocation2Event::getClassTypeId(), mouseMovedCallback, this); 
  root->addChild(mouseMovedCallbackNode);

  // Mouse Pressed
  SoEventCallback *mousePressedCallbackNode= new SoEventCallback;
  mousePressedCallbackNode->addEventCallback(SoMouseButtonEvent::getClassTypeId(),mousePressedCallback, this); 
  root->addChild(mousePressedCallbackNode);

  // Key Pressed
  SoEventCallback *keyPressedCallbackNode = new SoEventCallback;
  keyPressedCallbackNode->setName("Key_Pressed");
  keyPressedCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(),keyboardCallback,this); 
  root->addChild(keyPressedCallbackNode);



  return root;
}

//---------------------------------------------------------------------
void Demo::insertInputMesh(SoSeparator* sep)
{
  MoDrawStyle* ds = NULL;
  MoMaterial* mat = NULL;
  SoDrawStyle* sods = NULL;
  SoPickStyle* ps = NULL;
  this->getMeshAttributes(ds,mat,sods,ps);
  MiExtractorCallback* cb = this->getExtractorCallback();
  m_meshEditor = new MeshSceneEditor(sep,ds,mat,sods,ps,cb);

  m_meshEditor->edit(this);
}

//---------------------------------------------------------------------
void Demo::displayMenu() const
{
  displayMeshSwitch();
  if (displayInstructions())
    cout << "Press M to display this menu" << endl;
  cout << endl;
}

//---------------------------------------------------------------------
void Demo::displayMeshSwitch() const
{
  if (m_switchMeshType != NONE)
  {
    bool displayAll = (m_switchMeshType == ALL_TYPES || m_switchMeshType == ALL_TYPES_NO_TETRA);
    if (displayAll || m_switchMeshType == ALL_VOLUME_MESHES || m_switchMeshType == ALL_VOLUME_MESHES_NO_TETRA 
        || m_switchMeshType == ALL_SURFACE_VOLUME_MESHES || m_switchMeshType == ALL_IJK_MESHES)
    {
      if (m_switchMeshType != ALL_IJK_MESHES)
      {
        if (m_switchMeshType != ALL_TYPES_NO_TETRA && m_switchMeshType != ALL_VOLUME_MESHES_NO_TETRA)
          cout << "Press F1 for Tetrahedon mesh" << endl;
        cout << "Press F2 for Hexahedron mesh" << endl;
      }
      cout << "Press F3 for Regular mesh" << endl;
      cout << "Press F4 for Rectilinear mesh" << endl;
      cout << "Press F5 for Curvilinear mesh" << endl;
      cout << "Press F6 for Hexahedron IJK mesh (twice for Vertex Hexahedron IJK mesh)" << endl;
    }
    if (displayAll || m_switchMeshType == ALL_SURFACE_MESHES || m_switchMeshType == ALL_SURFACE_VOLUME_MESHES)
    {
      cout << "Press F7 for Quadrangle mesh" << endl;
      cout << "Press F8 for Regular Surface mesh" << endl;
      cout << "Press F9 for Rectilinear Surface mesh" << endl;
      cout << "Press F10 for Curvilinear Surface mesh" << endl;
    }
    if (displayAll || m_switchMeshType == ALL_LINE_MESHES)
    {
      cout << "Press F11 for unstructured line mesh" << endl;
      cout << "Press F12 for curvilinear mesh" << endl;
    }
  }
  if (m_toggleFiltering)
  {
    cout << "Press F to enable/disable the cell filter";
    if (m_switchMeshType == ALL_TYPES || m_switchMeshType == ALL_VOLUME_MESHES) 
      cout << " (except for Tetrahedron mesh)" << endl;
    else
      cout << endl;
  }

}

//---------------------------------------------------------------------
void Demo::setMeshData(bool setMesh, bool setTess)
{
  m_meshEditor->edit(this,setMesh,setTess);
}

//---------------------------------------------------------------------
void Demo::viewAll()
{
  m_viewer->viewAll();
}

//---------------------------------------------------------------------
void Demo::connectToMeshRepresentation(MoMesh* mesh) const
{
  m_meshEditor->connectTo(mesh);
}

//---------------------------------------------------------------------
void Demo::editInputMesh(SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();

  bool handled = false;
  bool noTetra = (m_switchMeshType == ALL_VOLUME_MESHES_NO_TETRA || m_switchMeshType == ALL_TYPES_NO_TETRA || m_switchMeshType ==  ALL_IJK_MESHES);
  if (m_switchMeshType != NONE && m_meshEditor != NULL) {
    bool enableAllMeshes = (m_switchMeshType == ALL_TYPES || m_switchMeshType == ALL_TYPES_NO_TETRA);
    bool enableVolumeMesh = (enableAllMeshes || m_switchMeshType == ALL_VOLUME_MESHES || m_switchMeshType ==  ALL_IJK_MESHES 
                            || m_switchMeshType == ALL_VOLUME_MESHES_NO_TETRA || m_switchMeshType == ALL_SURFACE_VOLUME_MESHES);
    if (enableAllMeshes || enableVolumeMesh)
    {
      if (!noTetra && SO_KEY_PRESS_EVENT(ev, F1))
      {
        m_meshType = UNDEFINED_MESH;
        m_meshEditor->disableFiltering();
        handled = true;
      }
      else if (m_switchMeshType !=  ALL_IJK_MESHES && SO_KEY_PRESS_EVENT(ev, F2))
      {
        m_meshType = VOLUME_MESH_UNSTRUCTURED;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F3))
      {
        m_meshType = VOLUME_MESH_REGULAR;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F4))
      {
        m_meshType = VOLUME_MESH_RECTILINEAR;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F5))
      {
        m_meshType = VOLUME_MESH_CURVILINEAR;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F6))
      {
        m_meshType = (m_meshType == VOLUME_MESH_HEXAHEDRON_IJK) ? HEXAHEDRON_MESH_IJK : VOLUME_MESH_HEXAHEDRON_IJK;
        handled = true;
      }
    }
    if (!handled && (enableAllMeshes || m_switchMeshType == ALL_SURFACE_MESHES || m_switchMeshType == ALL_SURFACE_VOLUME_MESHES))
    {
      if (SO_KEY_PRESS_EVENT(ev, F7))
      {
        m_meshType = SURFACE_MESH_UNSTRUCTURED;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F8))
      {
        m_meshType = SURFACE_MESH_REGULAR;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F9))
      {
        m_meshType = SURFACE_MESH_RECTILINEAR;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F10))
      {
        m_meshType = SURFACE_MESH_CURVILINEAR;
        handled = true;
      }
    }
    if (!handled && (enableAllMeshes || m_switchMeshType == ALL_LINE_MESHES))
    {
      if (SO_KEY_PRESS_EVENT(ev, F11))
      {
        m_meshType = LINE_MESH_UNSTRUCTURED;
        handled = true;
      }
      else if (SO_KEY_PRESS_EVENT(ev, F12))
      {
        m_meshType = LINE_MESH_CURVILINEAR;
        handled = true;
      }
    }

    if (handled)
      m_meshEditor->edit(this);
  }
  if ( !handled && m_toggleFiltering && m_meshType != UNDEFINED_MESH &&  SO_KEY_PRESS_EVENT(ev, F))
  {
    m_meshEditor->toggleFiltering();
    handled = true;
  }
 
  if (handled)
    eventCB->setHandled();
}


