#include "MeshSceneEditor.h"
#include "MeshData.h"

#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>

#include <MeshVizXLM/mapping/nodes/MoDataBinding.h>
#include <MeshVizXLM/mapping/nodes/MoTessellator.h>
#include <MeshVizXLM/mapping/nodes/MoCellFilter.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSurface.h>
#include <MeshVizXLM/mapping/nodes/MoMeshLine.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterI.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterIj.h>
#include <MeshVizXLM/mesh/cell/MiCellFilterIjk.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/sstream>
using namespace std;

//------------------------------------------------------------------------
MeshSceneEditor::MeshSceneEditor(SoSeparator* mainScene, MoDrawStyle* ds, MoMaterial* mat, SoDrawStyle* sods, SoPickStyle* ps, MiExtractorCallback* cb) :
 m_meshCenter(0.0)
{
  m_meshNode = new MoMesh;
  mainScene->addChild(m_meshNode);

  // Define color map
  m_colorMapNode = new MoPredefinedColorMapping;
  m_colorMapNode->predefColorMap = MoPredefinedColorMapping::STANDARD;
  mainScene->addChild(m_colorMapNode);

  // data set binding for unstructured ijk meshes
  m_binding = new MoDataBinding;
  m_binding->dataBinding = MoDataBinding::PER_NODE;
  mainScene->addChild(m_binding);

  // Scalar Value
  SoGroup* scalarGroup = new SoGroup;
  mainScene->addChild(scalarGroup);
  scalarGroup->setName("ScalarSets");
  {
    m_scalarSetINode = new MoScalarSetI;
    scalarGroup->addChild(m_scalarSetINode);
    m_scalarSetIjNode = new MoScalarSetIj;
    scalarGroup->addChild(m_scalarSetIjNode);
    m_scalarSetIjkNode = new MoScalarSetIjk;
    scalarGroup->addChild(m_scalarSetIjkNode);
  }

  // Vector Value
  SoGroup* vectorGroup = new SoGroup;
  mainScene->addChild(vectorGroup);
  vectorGroup->setName("VectorSets");
  {
    m_vec3SetINode = new MoVec3SetI;
    vectorGroup->addChild(m_vec3SetINode);
    m_vec3SetIjNode = new MoVec3SetIj;
    vectorGroup->addChild(m_vec3SetIjNode);
    m_vec3SetIjkNode = new MoVec3SetIjk;
    vectorGroup->addChild(m_vec3SetIjkNode);
  }

  SoShapeHints* shapeHints = new SoShapeHints;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  mainScene->addChild(shapeHints);

  m_tessellator = new MoTessellator;
  mainScene->addChild(m_tessellator);

  m_filterSwitch = new SoSwitch;
  m_filterSwitch->whichChild = SO_SWITCH_NONE;
  m_cellFilter = new MoCellFilter;
  mainScene->addChild(m_filterSwitch);
  m_filterSwitch->addChild(m_cellFilter);

  // Mesh Skin - Surface
  SoSeparator* sep = new SoSeparator;
  mainScene->addChild(sep);
  {
    if (ds == NULL)
    {
      ds = new MoDrawStyle;
      ds->displayEdges = true;
    }
    sep->addChild(ds);
    if (mat == NULL)
    {
      mat = new MoMaterial;
      mat->lineColoring = MoMaterial::COLOR;
      mat->lineColor.setValue(SbColor(0,0,0));
    }
    sep->addChild(mat);
    if (sods)
      sep->addChild(sods);
    if(ps)
      sep->addChild(ps);
    m_displaySwitch = new SoSwitch;
    sep->addChild(m_displaySwitch);
    {
      MoMeshSkin *skin = new MoMeshSkin;
      skin->setExtractorCallback(cb);
      m_displaySwitch->addChild(skin);
      MoMeshSurface *surface = new MoMeshSurface;
      surface->setExtractorCallback(cb);
      m_displaySwitch->addChild(surface);
      MoMeshLine *line = new MoMeshLine;
      line->setExtractorCallback(cb);
      m_displaySwitch->addChild(line);
    }
  }
}

 //------------------------------------------------------------------------
 void MeshSceneEditor::disableFiltering()
 {
   m_filterSwitch->whichChild = SO_SWITCH_NONE;
 }

 //------------------------------------------------------------------------
 void MeshSceneEditor::toggleFiltering()
 {
   m_filterSwitch->whichChild = (m_filterSwitch->whichChild.getValue() == SO_SWITCH_ALL) ? SO_SWITCH_NONE : SO_SWITCH_ALL;
 }

//------------------------------------------------------------------------
void MeshSceneEditor::clear()
{
  m_tessellator->setTessellator(NULL);
  m_cellFilter->reset();
  m_meshNode->reset();
  m_scalarSetINode->setScalarSet(NULL);
  m_scalarSetIjNode->setScalarSet(NULL);
  m_scalarSetIjkNode->setScalarSet(NULL);
  m_vec3SetINode->setVec3Set(NULL);
  m_vec3SetIjNode->setVec3Set(NULL);
  m_vec3SetIjkNode->setVec3Set(NULL);
}

//------------------------------------------------------------------------
void MeshSceneEditor::edit(MeshData* data, bool setMesh, bool setTess)
{
  if ( setMesh && setTess)
    clear();

  if ( setTess )
    m_tessellator->setTessellator(data->getTessellator());

  MeshType meshType;
  const MiMesh* mesh = data->getInputMesh(meshType);
  if (mesh == NULL || !setMesh)
    return;

  switch(meshType)
  {
  case UNDEFINED_MESH:
  case VOLUME_MESH_UNSTRUCTURED:
    {
      const MiVolumeMeshUnstructured* realMesh = dynamic_cast<const MiVolumeMeshUnstructured*> (mesh);
      this->set(realMesh,data->getScalarSetI(),m_scalarSetINode,data->getVec3dSetI(),m_vec3SetINode,data->getCellFilterI());
      break;
    }
  case VOLUME_MESH_REGULAR:
    {
      const MiVolumeMeshRegular* realMesh = dynamic_cast<const MiVolumeMeshRegular*> (mesh);
      this->set(realMesh,data->getScalarSetIjk(),m_scalarSetIjkNode,data->getVec3dSetIjk(),m_vec3SetIjkNode,data->getCellFilterIjk());
      break;
    }
  case VOLUME_MESH_RECTILINEAR:
    {
      const MiVolumeMeshRectilinear* realMesh = dynamic_cast<const MiVolumeMeshRectilinear*> (mesh);
      this->set(realMesh,data->getScalarSetIjk(),m_scalarSetIjkNode,data->getVec3dSetIjk(),m_vec3SetIjkNode,data->getCellFilterIjk());
      break;
    }
  case VOLUME_MESH_CURVILINEAR:
    {
      const MiVolumeMeshCurvilinear* realMesh = dynamic_cast<const MiVolumeMeshCurvilinear*> (mesh);
      this->set(realMesh,data->getScalarSetIjk(),m_scalarSetIjkNode,data->getVec3dSetIjk(),m_vec3SetIjkNode,data->getCellFilterIjk());
      break;
    }
  case HEXAHEDRON_MESH_IJK:
    {
      const MiVolumeMeshVertexHexahedronIjk* realMesh = dynamic_cast<const MiVolumeMeshVertexHexahedronIjk*> (mesh);
      this->set(realMesh,data->getScalarSetIjk(),m_scalarSetIjkNode,data->getVec3dSetIjk(),m_vec3SetIjkNode,data->getCellFilterIjk());
      break;
    }
  case VOLUME_MESH_HEXAHEDRON_IJK:
    {
      const MiVolumeMeshHexahedronIjk* realMesh = dynamic_cast<const MiVolumeMeshHexahedronIjk*> (mesh);
      if (data->getScalarSetI() != NULL)
      {
        this->set(realMesh,data->getScalarSetI(),m_scalarSetINode,data->getVec3dSetI(),m_vec3SetINode,data->getCellFilterIjk());
        m_binding->dataBinding = MoDataBinding::PER_NODE;
      }
      else
      {
        this->set(realMesh,data->getScalarSetIjk(),m_scalarSetIjkNode,data->getVec3dSetIjk(),m_vec3SetIjkNode,data->getCellFilterIjk());
        m_binding->dataBinding = MoDataBinding::PER_CELL;
      }
      break;
    }
  case SURFACE_MESH_UNSTRUCTURED:
    {
      const MiSurfaceMeshUnstructured* realMesh = dynamic_cast<const MiSurfaceMeshUnstructured*> (mesh);
      this->set(realMesh,data->getScalarSetI(),m_scalarSetINode,data->getVec3dSetI(),m_vec3SetINode,data->getCellFilterI());
      break;
    }
  case SURFACE_MESH_REGULAR:
    {
      const MiSurfaceMeshRegular* realMesh = dynamic_cast<const MiSurfaceMeshRegular*> (mesh);
      this->set(realMesh,data->getScalarSetIj(),m_scalarSetIjNode,data->getVec3dSetIj(),m_vec3SetIjNode,data->getCellFilterIj());
      break;
    }
  case SURFACE_MESH_RECTILINEAR:
    {
      const MiSurfaceMeshRectilinear* realMesh = dynamic_cast<const MiSurfaceMeshRectilinear*> (mesh);
      this->set(realMesh,data->getScalarSetIj(),m_scalarSetIjNode,data->getVec3dSetIj(),m_vec3SetIjNode,data->getCellFilterIj());
      break;
    }
  case SURFACE_MESH_CURVILINEAR :
    {
      const MiSurfaceMeshCurvilinear* realMesh = dynamic_cast<const MiSurfaceMeshCurvilinear*> (mesh);
      this->set(realMesh,data->getScalarSetIj(),m_scalarSetIjNode,data->getVec3dSetIj(),m_vec3SetIjNode,data->getCellFilterIj());
      break;
    }
  case LINE_MESH_UNSTRUCTURED:
    {
      const MiLineMeshUnstructured* realMesh = dynamic_cast<const MiLineMeshUnstructured*> (mesh);
      this->set(realMesh,data->getScalarSetI(),m_scalarSetINode,data->getVec3dSetI(),m_vec3SetINode,data->getCellFilterI());
      break;
    }
  case LINE_MESH_CURVILINEAR:
    {
      const MiLineMeshCurvilinear* realMesh = dynamic_cast<const MiLineMeshCurvilinear*> (mesh);
      this->set(realMesh,data->getScalarSetI(),m_scalarSetINode,data->getVec3dSetI(),m_vec3SetINode,data->getCellFilterI());
      break;
    }
  default:
    {
      clear();
      break;
    }
  }
  setDisplay(meshType);
}


//------------------------------------------------------------------------
void MeshSceneEditor::setDisplay(MeshType meshType)
{
  if (meshType == UNDEFINED_MESH)
    m_displaySwitch->whichChild = 0;
  else if (meshType >= LINE_MESH_UNSTRUCTURED)
    m_displaySwitch->whichChild = 2;
  else if (meshType >= SURFACE_MESH_UNSTRUCTURED)
    m_displaySwitch->whichChild = 1;
  else
    m_displaySwitch->whichChild = 0;
}

//------------------------------------------------------------------------
void MeshSceneEditor::connectTo(MoMesh* mesh) const
{
  int32_t activeChild = m_displaySwitch->whichChild.getValue();
  MoMeshRepresentation* meshRep = (MoMeshRepresentation*)m_displaySwitch->getChild(activeChild);
  mesh->connectFrom(meshRep);
}
