#ifndef _Vec3Set_H
#define _Vec3Set_H

#include <data/MbVec3SetI.h>
#include <data/MbVec3SetIj.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/vector>
#include <cmath>
using namespace std;

class Vec3Set : public MbVec3SetI<MbVec3d>, public MiVec3dSetIj, public MiVec3dSetIjk
{
friend class Vec3SetIj;
friend class Vec3SetIjk;
public:
  Vec3Set(size_t meshDim, DataBinding binding) : m_dimI(meshDim), m_dimJ(meshDim), m_dimK(meshDim), m_normalized(false), m_random(false)
  {
    setBinding(binding);
  }

  Vec3Set(const MbVec3<size_t>& meshDim, DataBinding binding) : m_dimI(meshDim[0]), m_dimJ(meshDim[1]), m_dimK(meshDim[2]),
    m_normalized(false), m_random(false)
  {
    setBinding(binding);
  }

  using MbVec3SetI<MbVec3d>::get;
  MbVec3d get(size_t i, size_t j) const 
  { 
    return get(j * m_dimI  + i);
  }

  MbVec3d get(size_t i, size_t j,size_t k) const 
  { 
    return get(k*m_dimI*m_dimJ + j *m_dimI  + i);
  }

  void randomVec(bool normalize = true)
  {
    size_t d = getSize();
    m_coords.resize(d);
    for (size_t i=0; i<d ; ++i) {
      m_coords[i] = MbVec3d(2*(rand()%m_dimI)/double(m_dimI)-1,2*(rand()%m_dimJ)/double(m_dimJ)-1 , 2*(rand()%m_dimK)/double(m_dimK)-1);
      if(normalize) 
        m_coords[i].normalize();
    }
    m_timeStamp++;
    m_normalized = normalize;
    m_random = true;
  }

  void regularVec(bool normalize = true)
  {
    size_t d = getSize();
    m_coords.resize(d);
    for (size_t i=0; i<d ; ++i) {
      m_coords[i] = computeValue(i,d);
      if(normalize) 
        m_coords[i].normalize();
    }
    m_timeStamp++;
    m_normalized = normalize;
    m_random = false;
  }

  void setBinding(DataBinding binding) 
  {
    if (binding != m_binding || m_coords.empty())
    {
      if (binding == MiDataSet::PER_NODE) {
        m_dimI++; m_dimJ++; m_dimK++; 
      }
      else if (!m_coords.empty())
      {
        m_dimI--; m_dimJ--; m_dimK--;
      }
      m_binding = binding;
      if (m_random)
        randomVec(m_normalized);
      else
        regularVec(m_normalized);
      m_timeStamp++;
    }
  }

  virtual std::string getName() const { return "Vec3Set";}
  virtual DataBinding  getBinding() const {return m_binding;}

  protected:
    size_t getSize(){ 
      return m_dimI*m_dimJ*m_dimK;
    }
    MbVec3d computeValue(size_t i, size_t SO_UNUSED_PARAM(d))
    {
      double z = (i / (m_dimI*m_dimJ)) / (double) m_dimK;
      size_t n = i % (m_dimI*m_dimJ);
      double y = (n / m_dimI) / (double) m_dimJ;
      double x = (n % m_dimI) / (double) m_dimI;
      return MbVec3d(x,(1-y),cos(3.14*x)*cos(z));
    }

    size_t m_dimI, m_dimJ, m_dimK;
    bool m_normalized;
    bool m_random;
};

class Magnitude : public MiDataSetI<double>, public MiDataSetIj<double>, public MiDataSetIjk<double>
{
public:
  Magnitude(const Vec3Set& vectorSet) : m_vecSet(vectorSet) {}

  double get(size_t i) const { 
    return m_vecSet.get(i).length();
  }
  double get(size_t i,size_t j) const { 
    return m_vecSet.get(i,j).length();
  }
  double get(size_t i,size_t j,size_t k) const { 
    return m_vecSet.get(i,j,k).length();
  }
  double getMin() const { 
    return m_vecSet.getMin().length();
  }
  double getMax() const { 
    return m_vecSet.getMax().length();
  }
  size_t getTimeStamp() const { return m_vecSet.getTimeStamp();}
  std::string getName() const { return "Magnitude";}
  DataBinding  getBinding() const {return m_vecSet.getBinding();}

private:
  const Vec3Set& m_vecSet;
};

class MagnitudeI : public MbScalardSetI
{
public:
  MagnitudeI(const MbVec3dSetI& vectorSet) : m_vecSet(vectorSet) {}

  double get(size_t i) const { 
    return m_vecSet.get(i).length();
  }
  double getMin() const { 
    return m_vecSet.getMin().length();
  }
  double getMax() const { 
    return m_vecSet.getMax().length();
  }
  size_t getTimeStamp() const { return m_vecSet.getTimeStamp();}
  std::string getName() const { return "MagnitudeI";}
  DataBinding  getBinding() const {return m_vecSet.getBinding();}

private:
  const MbVec3dSetI& m_vecSet;
};

class MagnitudeIj : public MbScalardSetIj
{
public:
  MagnitudeIj(const MbVec3dSetIj& vectorSet) : m_vecSet(vectorSet) {}

  double get(size_t i,size_t j) const { 
    return m_vecSet.get(i,j).length();
  }
  double getMin() const { 
    return m_vecSet.getMin().length();
  }
  double getMax() const { 
    return m_vecSet.getMax().length();
  }
  size_t getTimeStamp() const { return m_vecSet.getTimeStamp();}
  std::string getName() const { return "MagnitudeIj";}
  DataBinding  getBinding() const {return m_vecSet.getBinding();}

private:
  const MbVec3dSetIj& m_vecSet;
};

class MagnitudeIjk : public MiDataSetIjk<double>
{
public:
  MagnitudeIjk(const MbVec3dSetIjk& vectorSet) : m_vecSet(vectorSet) {}

  double get(size_t i,size_t j,size_t k) const { 
    return m_vecSet.get(i,j,k).length();
  }
  double getMin() const { 
    return m_vecSet.getMin().length();
  }
  double getMax() const { 
    return m_vecSet.getMax().length();
  }
  size_t getTimeStamp() const { return m_vecSet.getTimeStamp();}
  std::string getName() const { return "MagnitudeIjk";}
  DataBinding  getBinding() const {return m_vecSet.getBinding();}

private:
  const MbVec3dSetIjk& m_vecSet;
};

#endif

