#include <MbSampleMeshBuilder.h>

#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSetI.h>
#include <MeshVizXLM/mapping/nodes/MoLinearColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoLevelColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoCustomColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoMeshPlaneSlice.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoLegend.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/manips/SoClipPlaneManip.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/events/SoKeyboardEvent.h> 

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/vector>
using namespace std;

#define NAMESTR "MeshVizXLM Colormap test program"

#if defined(_MSC_VER)
#pragma warning( push )
#pragma warning(disable:4250)
#endif

static SbVec2s s_wSize(1024,768);

class myColorMapping : public MiColorMapping<double,SbColorRGBA>
{
public:
  myColorMapping(double min, double max) : m_min(min), m_max(max){}
  
  virtual SbColorRGBA getColor(double value) const
  {
    return SbColorRGBA(float((value-m_min)/(m_max-m_min)),0,0,1);
  }
  virtual void getRange(double& min, double& max) const 
  {
    min = m_min;
    max = m_max;
  }
  virtual size_t getTimeStamp() const {return 0;}
  double m_min;
  double m_max;

};

MbSampleMeshBuilder<MbVec3f,float> meshBuilder;
MbVolumeMeshTetrahedron<MbVec3f,float,MbVec3f>* mesh = &meshBuilder.getMeshTetrahedron();

void
editLegend(void *userData, SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();
  MoLegend* legend = (MoLegend*) userData;
  if (legend)
  {
    if (SO_KEY_PRESS_EVENT(ev, F1)) 
    {
      legend->vertical = (legend->vertical.getValue() == TRUE) ? FALSE : TRUE; 
      SbVec2f bl = legend->bottomLeft.getValue();
      SbVec2f tr = legend->topRight.getValue();
      // Adjust the width to the initial window ratio (does no work if resized)
      float r = float(s_wSize[0])/float(s_wSize[1]);
      float tmp;
      if (legend->vertical.getValue() == TRUE)
      {
        tmp = bl[0];
        bl[0] = bl[1]* r;
        bl[1] = tmp; 
        legend->bottomLeft.setValue(bl);
        tmp = tr[0];
        tr[0] = tr[1]* r;
        tr[1] = tmp; 
        legend->topRight.setValue(tr);
      }
      else
      {
        tmp = bl[0];
        bl[0] = bl[1];
        bl[1] = tmp  / r; 
        legend->bottomLeft.setValue(bl);
        tmp = tr[0];
        tr[0] = tr[1];
        tr[1] = tmp / r; 
        legend->topRight.setValue(SbVec2f(tr));
      }
    }
    if (SO_KEY_PRESS_EVENT(ev, F2)) 
    {
      int pos = legend->titlePosition.getValue()+1;
      if (pos > MoLegend::POS_LEFT) 
        pos = MoLegend::POS_BOTTOM;
      legend->titlePosition = pos; 
    }
    if (SO_KEY_PRESS_EVENT(ev, F3)) 
    {
      legend->displayValues = (legend->displayValues.getValue() == TRUE) ? FALSE : TRUE; 
    }
    if (SO_KEY_PRESS_EVENT(ev, PAGE_UP))
    {
      legend->numValues = legend->numValues.getValue() + 1;
    }
    if (SO_KEY_PRESS_EVENT(ev, PAGE_DOWN))
    {
      legend->numValues = legend->numValues.getValue() - 1;
    }
    if (SO_KEY_PRESS_EVENT(ev, F4))
    {
      legend->valueNotation = MoLegend::NOTATION_DEFAULT;
      std::cout<< "Default notation :  " << std::endl;
    }
    if (SO_KEY_PRESS_EVENT(ev, F5))
    {
      legend->valueNotation = MoLegend::NOTATION_FIXED;
      std::cout<< "Fixed notation :  " << std::endl;
    }
    if (SO_KEY_PRESS_EVENT(ev, F6))
    {
      legend->valueNotation = MoLegend::NOTATION_SCIENTIFIC;
      std::cout<< "Scientific notation :  " << std::endl;
    }
    if (SO_KEY_PRESS_EVENT(ev, O))
    {
      if (legend->valuePrecision.getValue() > 0) {
        legend->valuePrecision = legend->valuePrecision.getValue() - 1;
        std::cout<< "precision : " << legend->valuePrecision.getValue() << std::endl;
      }
    }
    if (SO_KEY_PRESS_EVENT(ev, P))
    {
      legend->valuePrecision = legend->valuePrecision.getValue() + 1;
      std::cout<< "precision : " << legend->valuePrecision.getValue() << std::endl;
    }
  }
}

void
editLevelColorMap(void *userData, SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();
  MoLevelColorMapping* levelColorMap = (MoLevelColorMapping*)userData;
  if (levelColorMap && !eventCB->isHandled())
  {
    if (SO_KEY_PRESS_EVENT(ev, F7))
    {
      eventCB->setHandled();
      levelColorMap->minThresholdEnable = !levelColorMap->minThresholdEnable.getValue();
    }
    else if (SO_KEY_PRESS_EVENT(ev, F8))
    {
      eventCB->setHandled();
      levelColorMap->maxThresholdEnable = !levelColorMap->maxThresholdEnable.getValue();
    }
    else if (SO_KEY_PRESS_EVENT(ev, L))
    {
      eventCB->setHandled();
      levelColorMap->minThresholdValue = levelColorMap->minThresholdValue.getValue() - 1;
      levelColorMap->maxThresholdValue = levelColorMap->maxThresholdValue.getValue() + 1;
    }
    else if (SO_KEY_PRESS_EVENT(ev, M))
    {
      eventCB->setHandled();
      levelColorMap->minThresholdValue = levelColorMap->minThresholdValue.getValue() + 1;
      levelColorMap->maxThresholdValue = levelColorMap->maxThresholdValue.getValue() - 1;
    }
  }
}

//-----------------------------------------------------------------------------
int
main(int, char **)
{

  cout << "Press F1 for vertical/horizontal " << endl;
  cout << "Press F2 to move the legend title position to the four possible values" << endl;
  cout << "Press F3 to toggle the visibility of min and max values" << endl;
  cout << "Press F4 to toggle the notation to default" << endl;
  cout << "Press F5 to toggle the notation to fixed" << endl;
  cout << "Press F6 to toggle the notation to scientific" << endl;
  cout << "Press F7 to toggle min thresholding on Level ColorMap" << endl;
  cout << "Press F8 to toggle max thresholding on Level ColorMap" << endl;
  cout << "Press L/M to change threshold values from Level ColorMap" << endl;
  cout << "Press O to decrement precision of values" << endl;
  cout << "Press P to increment precision of values" << endl;
  cout << "Press Page-down to decrement the number of values" << endl;
  cout << "Press Page-up to increment the number of values" << endl;

  float min = 2;
  float max = 15;
  float values[4] ;
  SbVec4f colors[4];

  SbVec2f bl(-0.9f,-0.9f);
  SbVec2f tr(-0.75f,0.9f);
  SbVec2f incr(0.6f,0.f);

  // Init viewer
  Widget my_window = SoXt::init(NAMESTR) ;
  if (my_window == NULL) exit(1) ;

  MoMeshViz::init();

  SoSeparator* sep = new SoSeparator;
  {
    // predefined color map
    MoPredefinedColorMapping* predefMap = new MoPredefinedColorMapping;
    predefMap->minValue = min;
    predefMap->maxValue = max;
    predefMap->predefColorMap = MoPredefinedColorMapping::PHYSICS;
    sep->addChild(predefMap);

    MoLegend* legendPredef = new MoLegend;
    legendPredef->title = "Predefined ColorMap";
    legendPredef->titleFontSize = 16;
    legendPredef->valuesFontSize = 15;
    legendPredef->vertical = TRUE;
    legendPredef->bottomLeft = bl;
    legendPredef->topRight   = tr;
    legendPredef->minValue = (float)min;
    legendPredef->maxValue = (float)max;
    legendPredef->displayValues = true;
    legendPredef->numValues = 5;

    sep->addChild(legendPredef);

    bl += incr;
    tr += incr;
 
    SoEventCallback *myCallbackNode = new SoEventCallback;
    myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(), editLegend, legendPredef); 
    sep->addChild(myCallbackNode);

  }
  {
    // custom color map
    MoCustomColorMapping* customCmap = new MoCustomColorMapping;
    customCmap->setColorMapping(new myColorMapping(min,max));
    sep->addChild(customCmap);

    MoLegend* legendCustom = new MoLegend;
    legendCustom->title = "Custom ColorMap";
    legendCustom->titleFontSize = 16;
    legendCustom->valuesFontSize = 15;
    legendCustom->vertical = TRUE;
    legendCustom->bottomLeft = bl;
    legendCustom->topRight   = tr;
    legendCustom->minValue = (float)min;
    legendCustom->maxValue = (float)max;
    legendCustom->displayValues = true;
    legendCustom->numValues = 4;

    sep->addChild(legendCustom);

    bl += incr;
    tr += incr;

    SoEventCallback *myCallbackNode = new SoEventCallback;
    myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(), editLegend, legendCustom); 
    sep->addChild(myCallbackNode);

  }
  {
    // linear color map
    MoLinearColorMapping* linearMap = new MoLinearColorMapping;
    values[0] = min;
    values[1] = (max+min)/3;
    values[2] = max;
    linearMap->values.setValues(0,3,values);
    colors[0].setValue(1,0,0,1); 
    colors[1].setValue(1,1,1,1); 
    colors[2].setValue(0,0,1,1); 
    linearMap->colors.setValues(0,3,colors);
    sep->addChild(linearMap);

    MoLegend* legendLinear = new MoLegend;
    legendLinear->title = "Linear ColorMap";
    legendLinear->titleFontSize = 16;
    legendLinear->valuesFontSize = 15;
    legendLinear->vertical = TRUE;
    legendLinear->bottomLeft = bl;
    legendLinear->topRight   = tr;
    legendLinear->minValue = (float)min;
    legendLinear->maxValue = (float)max;
    legendLinear->displayValues = true;
    legendLinear->numValues = 2;

    sep->addChild(legendLinear);

    bl += incr;
    tr += incr;

    SoEventCallback *myCallbackNode = new SoEventCallback;
    myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(), editLegend, legendLinear); 
    sep->addChild(myCallbackNode);

  }
  {
    // level color map
    MoLevelColorMapping* levelMap = new MoLevelColorMapping;
    values[0] = min;
    values[1] = (max + min)/3.0f;
    values[2] = (max + min)/2.0f;
    values[3] = max;
    levelMap->values.setValues(0,4,values);
    colors[0].setValue(1,0,0,1);
    colors[1].setValue(0,0,1,1);
    colors[2].setValue(0,1,0,1);
    levelMap->colors.setValues(0,3,colors);
    levelMap->minThresholdEnable = false;
    levelMap->minThresholdValue = (max + min) / 4.0f;
    levelMap->minThresholdColor.setValue(1, 1, 1, 1);
    levelMap->maxThresholdEnable = false;
    levelMap->maxThresholdValue = 3 * (max + min) / 4.0f;
    levelMap->maxThresholdColor.setValue(1, 1, 1, 1);
    sep->addChild(levelMap);

    MoLegend* legendLevel = new MoLegend;
    legendLevel->title = "Level ColorMap";
    legendLevel->titleFontSize = 16;
    legendLevel->valuesFontSize = 15;
    legendLevel->vertical = TRUE;
    legendLevel->bottomLeft = bl;
    legendLevel->topRight   = tr;
    legendLevel->displayValues = true;
    legendLevel->numValues = 11;

    sep->addChild(legendLevel);

    bl += incr;
    tr += incr;

    SoEventCallback *myCallbackNode = new SoEventCallback;
    myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(), editLegend, legendLevel); 
    sep->addChild(myCallbackNode);

    SoEventCallback *myCallbackNode2 = new SoEventCallback;
    myCallbackNode2->addEventCallback(SoKeyboardEvent::getClassTypeId(), editLevelColorMap, levelMap);
    sep->addChild(myCallbackNode2);

  }
  
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(my_window);
  viewer->setSize(s_wSize);
  viewer->setSceneGraph(sep);
  viewer->show();
  viewer->viewAll();

  // dialog
  SoXt::show(my_window);
  SoXt::mainLoop();

  delete viewer;

  MoMeshViz::finish();
  SoXt::finish();
  return 0;
}

#if defined(_MSC_VER)
#pragma warning( pop ) 
#endif

