#include "DemoPickingAndProbing.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoKeyboardEvent.h> 
#include <Inventor/manips/SoClipPlaneManip.h>
#include <MeshVizXLM/mapping/nodes/MoMeshPointProbe.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/details/MoLineDetailI.h>
#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoFaceDetailIj.h>
#include <MeshVizXLM/mapping/details/MoFaceDetailIjk.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>

using namespace::std;

//---------------------------------------------------------------------
DemoPickingAndProbing::DemoPickingAndProbing() :
Demo("MeshVizXLM Picking And Probing",VOLUME_MESH_UNSTRUCTURED,ALL_TYPES,true),
m_probeSwitch(NULL), m_draggerEnabled(false)
{
}

//---------------------------------------------------------------------
DemoPickingAndProbing::~DemoPickingAndProbing()
{
  delete m_textEditor;
}

//---------------------------------------------------------------------
bool DemoPickingAndProbing::displayInstructions() const
{
  cout << "Press D to enable probe with dragger (not available for line meshes)" << endl;
  cout << "Press P to enable mouse picking" << endl;
  cout << "Click on the arrow icon then move the cursor over the mesh skin." << endl << endl;
  return true;
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::keyPressed(SoEventCallback *eventCB)
{
  bool handled = false;
  const SoEvent *ev = eventCB->getEvent();

  if (SO_KEY_PRESS_EVENT(ev, D)) 
  {
    if (m_meshType < LINE_MESH_UNSTRUCTURED || m_meshType == UNDEFINED_MESH)
    {
      cout << "Move the dragger inside the mesh." << endl;
      m_probe->setProbeCallback(*this);
      m_draggerEnabled = true;
      handled = true;
    }
  }
  else if (SO_KEY_PRESS_EVENT(ev, P))
  {
    cout << "Click on the arrow icon then move the cursor over the mesh skin." << endl;
    m_probe->removeProbeCallback();
    m_draggerEnabled = false;
    handled = true;
  }

  if (handled)
    eventCB->setHandled();

  setMeshAttibutes();
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::mouseMoved(SoEventCallback *eventCB)
{
  if(m_draggerEnabled)  return;

  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // something has been picked
    SbVec3f point = pickedPoint->getPoint();
    const SoDetail* detail = pickedPoint->getDetail();
    const MoMeshDetail* mdetail = NULL;
    size_t cellId[3] = {UNDEFINED_ID,UNDEFINED_ID,UNDEFINED_ID };
    size_t num = 1;
    double value = 0.0;
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // a face of an unstructured mesh representation has been picked
      // get detail from the picked face 
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;
      cellId[0] = fdetail->getCellIndex();
      value = fdetail->getValue(point);
      mdetail = fdetail->getMeshDetail();
    }
    else if (detail->isOfType(MoFaceDetailIj::getClassTypeId()))
    {
      // a face of an structured surface mesh representation has been picked
      // get detail from the picked face 
      MoFaceDetailIj* fdetail = (MoFaceDetailIj*) detail;
      cellId[0] = fdetail->getCellIndexI();
      cellId[1] = fdetail->getCellIndexJ();
      num = 2;
      value = fdetail->getValue(point);
      mdetail = fdetail->getMeshDetail();
    }
    else if (detail->isOfType(MoFaceDetailIjk::getClassTypeId()))
    {
      // a face of an structured volume mesh representation has been picked
      // get detail from the picked face 
      MoFaceDetailIjk* fdetail = (MoFaceDetailIjk*) detail;
      cellId[0] = fdetail->getCellIndexI();
      cellId[1] = fdetail->getCellIndexJ();
      cellId[2] = fdetail->getCellIndexK();
      num = 3;
      value = fdetail->getValue(point);
      mdetail = fdetail->getMeshDetail();
    }
    else if (detail->isOfType(MoLineDetailI::getClassTypeId()))
    {
      // a line of a line mesh representation has been picked
      // get detail from the picked face 
      MoLineDetailI* ldetail = (MoLineDetailI*) detail;
      cellId[0] = ldetail->getCellIndex();
      value = ldetail->getValue(point);
      mdetail = ldetail->getMeshDetail();
    }
    if (mdetail)
    {
      // get detail from the picked input mesh 
      MeshType meshType = mdetail->getMeshType();
      std::string colorName = mdetail->getColorScalarSet()->getName();
      // process the detail information retreived by picking
      m_textEditor->edit(cellId,num,value,meshType,colorName);
    }
    else
      // other Inventor shape
      m_textEditor->clear();
  }
  else 
    // nothing picked
    m_textEditor->clear();
  eventCB->setHandled();
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::motionCallback(size_t cellId, const MeXScalardSetI& scalars, const MeXVec3dSetI& vectors)
{
  double scalar = scalars.getSize() > 0 ? scalars.get(0) : 0;
  MbVec3d vector = vectors.getSize() > 0 ? vectors.get(0) : MbVec3d(0);
  m_textEditor->edit(&cellId,1,scalar,vector);
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::motionCallback(size_t cellIdI,size_t cellIdJ, const MeXScalardSetI& scalars, const MeXVec3dSetI& vectors)
{
  size_t cellIds[2];
  cellIds[0] = cellIdI;
  cellIds[1] = cellIdJ;
  double scalar = scalars.getSize() > 0 ? scalars.get(0) : 0;
  MbVec3d vector = vectors.getSize() > 0 ? vectors.get(0) : MbVec3d(0);
  m_textEditor->edit(cellIds,2,scalar,vector);
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::motionCallback(size_t cellIdI,size_t cellIdJ, size_t cellIdK, const MeXScalardSetI& scalars, const MeXVec3dSetI& vectors)
{
  size_t cellIds[3];
  cellIds[0] = cellIdI;
  cellIds[1] = cellIdJ;
  cellIds[2] = cellIdK;
  double scalar = scalars.getSize() > 0 ? scalars.get(0) : 0;
  MbVec3d vector = vectors.getSize() > 0 ? vectors.get(0) : MbVec3d(0);
  m_textEditor->edit(cellIds,3,scalar,vector);
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::getMeshAttributes(MoDrawStyle*& ds, MoMaterial*& mat, SoDrawStyle*& /*ids*/, SoPickStyle*& ps)
{
  ps = m_ps = new SoPickStyle;
  ds = m_ds = new MoDrawStyle;
  mat = m_mat = new MoMaterial;
  m_ds->displayEdges = true;
  m_mat->lineColor.setValue(SbColor(0,0,0));
  setMeshAttibutes();
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::setMeshAttibutes()
{
  bool displayingLines = m_meshType >= LINE_MESH_UNSTRUCTURED && m_meshType != UNDEFINED_MESH;
  if (m_draggerEnabled)
  {
    m_ps->style = SoPickStyle::UNPICKABLE;
    m_ds->displayFaces = false;
    m_mat->lineColoring = MoMaterial::CONTOURING;
    if (m_probeSwitch)
      m_probeSwitch->whichChild = displayingLines ? SO_SWITCH_NONE : SO_SWITCH_ALL;
  }
  else
  {
    m_ps->style = SoPickStyle::SHAPE;
    m_ds->displayFaces = true;
    m_mat->lineColoring =  displayingLines ? MoMaterial::CONTOURING : MoMaterial::COLOR;
    if (m_probeSwitch)
      m_probeSwitch->whichChild = SO_SWITCH_NONE;
  }
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::insertRepresentations(SoSeparator* sep)
{
  m_probeSwitch = new SoSwitch;
  m_probeSwitch->whichChild = SO_SWITCH_ALL;
  sep->addChild(m_probeSwitch);
  // Manipulator to move the plane slice plane (as a clip plane)
  SoClipPlaneManip *manip = new SoClipPlaneManip;
  manip->plane.setValue(SbPlane(SbVec3f(1,0,1),5.0f));
  // Turn off clip plane visibility to avoid flickering of plane slice faces.
  manip->on.setValue(FALSE); 
  manip->draggerPosition.setValue(this->getCenter());
  m_probeSwitch->addChild(manip);

    // PointProbe Section
  m_probe = new MoMeshPointProbe;
  m_probe->position.connectFrom(&manip->draggerPosition);
  m_probeSwitch->addChild(m_probe);
}

//---------------------------------------------------------------------
void DemoPickingAndProbing::insertScene2D(SoSeparator* sep)
{
  m_textEditor = new TextSceneEditor(sep);
}

