#include "DemoPolyhedralIsosurface.h"

#include <MeshVizXLM/tessellator/MiTessellator.h>

#include <MeshVizXLM/mapping/nodes/MoTessellator.h>
#include <MeshVizXLM/mapping/nodes/MoMeshIsosurface.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>

#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoKeyboardEvent.h> 
#include <Inventor/helpers/SbFileHelper.h>

using namespace::std;

//---------------------------------------------------------------------
DemoPolyhedralIsosurface::DemoPolyhedralIsosurface() :
Demo("MeshVizXLM Polyhedral Isosurface"), 
m_mesh(NULL),
m_meshFileName("cone.dat")
{
  m_tessellator = MiTessellator::getNewTessellatorPolyhedron();
  m_demoPath = (SoPreferences::getString("OIVHOME",".")).getString() 
                + std::string("/examples/data/MeshVizXLM/");
}

//---------------------------------------------------------------------
DemoPolyhedralIsosurface::~DemoPolyhedralIsosurface()
{
  delete m_tessellator;
}

//---------------------------------------------------------------------
bool DemoPolyhedralIsosurface::displayInstructions() const
{
  cout << "Press F1 for cone" << endl;
  cout << "Press F2 for sphere" << endl;
  cout << "Press F3 for smiley " << endl;
  cout << "Press i to enable/disable isosurface visibility" << endl;
  cout << "Press u/d to increase/decrease isovalue" << endl;
  cout << "Press g to tranform the geometry" << endl;
  return true;
}

//---------------------------------------------------------------------
void DemoPolyhedralIsosurface::keyPressed(SoEventCallback *eventCB)
{
  bool handled = false;
  const SoEvent *ev = eventCB->getEvent();

  if (SO_KEY_PRESS_EVENT(ev, U)) 
  {
    float isovalue = m_isosurface->isovalue.getValue();
    m_isosurface->isovalue.setValue(isovalue + (float)0.1);
    handled = true;
  } 
  else if (SO_KEY_PRESS_EVENT(ev, D)) 
  {
    float isovalue = m_isosurface->isovalue.getValue();
    m_isosurface->isovalue.setValue(isovalue - (float)0.1);
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, I)) 
  {
    m_isoSwitch->whichChild = ( m_isoSwitch->whichChild.getValue()==SO_SWITCH_ALL) ? SO_SWITCH_NONE : SO_SWITCH_ALL;
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, G)) 
  {
    m_mesh->getGeometry().transform(m_transformFunc);
    m_transformFunc.reverse = !m_transformFunc.reverse;
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, F1))
  {
    cout << "Displaying cone" << endl;
    setMesh("cone.dat");
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, F2))
  {
    cout << "Displaying sphere" << endl;
    setMesh("sphere.dat");
    handled = true;
  }
  else if (SO_KEY_PRESS_EVENT(ev, F3))
  {
    cout << "Displaying smiley" << endl;
    setMesh("smiley.dat");
    handled = true;
  }

  if (handled)
    eventCB->setHandled();
}

//---------------------------------------------------------------------
MbVec3d DemoPolyhedralIsosurface::Transform::operator()(const MbVec3d& coord) const
{
  double x = !reverse ? cos(coord[2])+coord[0] : -cos(coord[2])+coord[0];  
  return MbVec3d(x,coord[1],coord[2]);
}

//---------------------------------------------------------------------
void DemoPolyhedralIsosurface::setMesh(const char* filename)
{
  m_meshFileName = filename;
  setMeshData();
  m_isosurface->isovalue = (float) m_isoMidValue;
  viewAll();
}

//---------------------------------------------------------------------
const MiMesh* DemoPolyhedralIsosurface::getInputMesh(MeshType& meshType)
{
  meshType = VOLUME_MESH_UNSTRUCTURED;
  m_mesh = &m_reader.readMesh(m_demoPath + m_meshFileName);
  return m_mesh;
}

//---------------------------------------------------------------------
const MbScalardSetI* DemoPolyhedralIsosurface::getScalarSetI()
{
  const MbScalardSetI* scalar = m_mesh->getScalarSet(0);
  m_isoMidValue = (scalar->getMin() + scalar->getMax()) / 2.0;
  return scalar;
}

//---------------------------------------------------------------------
void DemoPolyhedralIsosurface::getMeshAttributes(MoDrawStyle*& ds, MoMaterial*& mat, SoDrawStyle*& /*ids*/, SoPickStyle*& /*ps*/)
{
  ds = new MoDrawStyle;
  mat = new MoMaterial;
  ds->displayFaces = false;
  ds->displayEdges = true;
  mat->lineColoring = MoMaterial::CONTOURING;
}

//---------------------------------------------------------------------
void DemoPolyhedralIsosurface::insertRepresentations(SoSeparator* sep)
{
  // Tessellator section
  MoTessellator* tessellatorNode = new MoTessellator;
  tessellatorNode->setTessellator(m_tessellator);
  sep->addChild(tessellatorNode);

  // Isosurface Section
  m_isoSwitch = new SoSwitch;
  m_isoSwitch->whichChild = SO_SWITCH_ALL;
  sep->addChild(m_isoSwitch);
  m_isosurface = new MoMeshIsosurface;
  m_isosurface->isovalue = (float) m_isoMidValue;
  m_isosurface->setExtractorCallback(&m_extractorCB);
  m_isoSwitch->addChild(m_isosurface);
}


