#include <MeshVizXLM/mesh/cell/MiCellFilterIjk.h>

#include <mesh/volumes/MbVolumeMeshPolyhedron.h>
#include <data/MbVec3SetI.h>

#include <colormapping/MbLinearColorMapping.h>
#include <colormapping/MbLevelColorMapping.h>

#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSetI.h>
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoCustomColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoMeshGridPlaneSlice.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMeshPointProbe.h>

#include <MbSampleMeshBuilder.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/SoPickedPoint.h>

#include <Inventor/manips/SoClipPlaneManip.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/vector>
using namespace std;

#define NAMESTR "MeshVizXLM sample program showing how to use a logical slice"

MbLinearColorMapping<double,SbColorRGBA,float> linearColorMap;
MbLevelColorMapping<double,SbColorRGBA> levelColorMapping;

SoXtExaminerViewer* viewer;
SoSwitch* m_sliceSwitch;
SoSwitch* m_skinSwitch;
SoClipPlaneManip *m_manip;

MbVolumeMeshPolyhedron<> *m_mesh;
const MbScalardSetI *m_colorSet;

#if defined(_MSC_VER)
#pragma warning( push )
#pragma warning(disable:4250)
#endif

MbVolumeMeshPolyhedron<>* buildDodecahedron()
{
  MbVec3d coords[] = {
    MbVec3d(0),
    MbVec3d(4.574530124664307e-001, 1.312415313720703e+001, 0.000000000000000e+000 ),
    MbVec3d(-3.600445032119751e+000, 1.312415313720703e+001, 0.000000000000000e+000 ),
    MbVec3d( -1.571496009826660e+000, 9.841244697570801e+000, 5.311857700347900e+000 ),
    MbVec3d( -1.571496009826660e+000, 5.783347129821777e+000, 5.311857700347900e+000 ),
    MbVec3d( -1.571496009826660e+000, 9.841244697570801e+000, -5.311857700347900e+000 ),
    MbVec3d( -1.571496009826660e+000, 5.783347129821777e+000, -5.311857700347900e+000 ),
    MbVec3d( 4.574530124664307e-001, 2.500438213348389e+000, 0.000000000000000e+000 ),
    MbVec3d( -3.600445032119751e+000, 2.500438213348389e+000, 0.000000000000000e+000 ),
    MbVec3d( -6.883353710174561e+000, 7.812295913696289e+000, 2.028949022293091e+000 ),
    MbVec3d( -6.883353710174561e+000, 7.812295913696289e+000, -2.028949022293091e+000 ),
    MbVec3d( 3.740361690521240e+000, 7.812295913696289e+000, -2.028949022293091e+000 ),
    MbVec3d( 3.740361690521240e+000, 7.812295913696289e+000, 2.028949022293091e+000 ),
    MbVec3d( -4.854404449462891e+000, 1.109520435333252e+001, 3.282908678054810e+000 ),
    MbVec3d( 1.711412668228149e+000, 1.109520435333252e+001, 3.282908678054810e+000 ),
    MbVec3d( 1.711412668228149e+000, 1.109520435333252e+001, -3.282908678054810e+000 ),
    MbVec3d( -4.854404449462891e+000, 1.109520435333252e+001, -3.282908678054810e+000 ),
    MbVec3d( -4.854404449462891e+000, 4.529387474060059e+000, -3.282908678054810e+000 ),
    MbVec3d( -4.854404449462891e+000, 4.529387474060059e+000, 3.282908678054810e+000 ),
    MbVec3d( 1.711412668228149e+000, 4.529387474060059e+000, -3.282908678054810e+000 ),
    MbVec3d( 1.711412668228149e+000, 4.529387474060059e+000, 3.282908678054810e+000 )
  };

  double color[21];
  for (size_t i=0; i < 21; ++i){
    color[i] = coords[i][0];
  }

  size_t numFacets[] = { 12 };
  size_t numNodes[] = { 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5 };
  size_t facets[] = { 15, 5, 16, 2, 1, 
                      1, 14, 12, 11, 15, 
                      16, 10, 9, 13, 2, 
                      2, 13, 3, 14, 1, 
                      18, 8, 7, 20, 4, 
                      9, 18, 4, 3, 13, 
                      14, 3, 4, 20, 12, 
                      8, 17, 6, 19, 7, 
                      15, 11, 19, 6, 5, 
                      5, 6, 17, 10, 16, 
                      12, 20, 7, 19, 11, 
                      10, 17, 8, 18, 9 } ;
  size_t facets2[60];
  for (size_t i=0; i < 60; ++i)
    facets2[i] = facets[59-i];

  MbVolumeMeshPolyhedron<> *mesh = new MbVolumeMeshPolyhedron<>(coords,coords+21,facets2,facets2+60,numNodes,numNodes+12,numFacets,numFacets+1);
  mesh->addScalarSet(new MbScalarSetI<double>(color,color+21,"colorSet"));
  return mesh;
}

MbVolumeMeshPolyhedron<>* buildOctogon()
{
  MbVec3d coords[] = {
    MbVec3d(0),
    MbVec3d( 1.325747108459473e+001, 2.500442504882813e+000, -6.793365478515625e-001 ),
    MbVec3d( 9.201438903808594e+000, 2.500442504882813e+000, -1.047146701812744e+001 ),
    MbVec3d( -5.906926393508911e-001, 2.500442504882813e+000, -1.452750015258789e+001 ),
    MbVec3d( -1.038282203674316e+001, 2.500442504882813e+000, -1.047146701812744e+001 ),
    MbVec3d( -1.443885612487793e+001, 2.500442504882813e+000, -6.793353557586670e-001 ),
    MbVec3d( -1.038282012939453e+001, 2.500442504882813e+000, 9.112794876098633e+000 ),
    MbVec3d( -5.906918644905090e-001, 2.500442504882813e+000, 1.316882705688477e+001 ),
    MbVec3d( 9.201440811157227e+000, 2.500442504882813e+000, 9.112790107727051e+000 ),
    MbVec3d( 1.325747108459473e+001, 2.500442504882813e+000, -6.793365478515625e-001 ),
    MbVec3d( 9.201438903808594e+000, 2.500442504882813e+000, -1.047146701812744e+001 ),
    MbVec3d( -5.906926393508911e-001, 2.500442504882813e+000, -1.452750015258789e+001 ),
    MbVec3d( -1.038282203674316e+001, 2.500442504882813e+000, -1.047146701812744e+001 ),
    MbVec3d( -1.443885612487793e+001, 2.500442504882813e+000, -6.793353557586670e-001 ),
    MbVec3d( -1.038282012939453e+001, 2.500442504882813e+000, 9.112794876098633e+000 ),
    MbVec3d( -5.906918644905090e-001, 2.500442504882813e+000, 1.316882705688477e+001 ),
    MbVec3d( 9.201440811157227e+000, 2.500442504882813e+000, 9.112790107727051e+000 ),
    MbVec3d( 1.325747108459473e+001, 1.694888305664063e+001, -6.793365478515625e-001 ),
    MbVec3d( 9.201438903808594e+000, 1.694888305664063e+001, -1.047146701812744e+001 ),
    MbVec3d( -5.906926393508911e-001, 1.694888305664063e+001, -1.452750015258789e+001 ),
    MbVec3d( -1.038282203674316e+001, 1.694888305664063e+001, -1.047146701812744e+001 ),
    MbVec3d( -1.443885612487793e+001, 1.694888305664063e+001, -6.793353557586670e-001 ),
    MbVec3d( -1.038282012939453e+001, 1.694888305664063e+001, 9.112794876098633e+000 ),
    MbVec3d( -5.906918644905090e-001, 1.694888305664063e+001, 1.316882705688477e+001 ),
    MbVec3d( 9.201440811157227e+000, 1.694888305664063e+001, 9.112790107727051e+000 )
  };

  double color[25];
  for (size_t i=0; i < 25; ++i)
    color[i] = coords[i][0];

  size_t numFacets[] = { 10 };
  size_t numNodes[] = { 4, 4, 4, 4, 4, 4, 4, 4, 8, 8 };
  size_t facets[] = { 9, 17, 18, 10, 16, 24, 17, 9, 10, 18, 19, 11, 11, 19,
    20, 12, 12, 20, 21, 13, 13, 21, 22, 14, 14, 22, 23, 15, 15, 23, 24, 16,
    18, 17, 24, 23, 22, 21, 20, 19, 9, 10, 11, 12, 13, 14, 15, 16 };

  MbVolumeMeshPolyhedron<> *mesh = new MbVolumeMeshPolyhedron<>(coords,coords+25,facets,facets+48,numNodes,numNodes+10,numFacets,numFacets+1);
  mesh->addScalarSet(new MbScalarSetI<double>(color,color+25,"colorSet"));
  return mesh;
}

MbVolumeMeshPolyhedron<>* buildNonPlanarPolyhedron()
{ // cell taken from petrel
    MbVec3d coords[] = 
  {
    MbVec3d(9840.5504526569275,-10849.854228217155,-1927.8735351562500),
    MbVec3d(9845.4567026569275,-10886.354228217155,-1924.6853027343750),
    MbVec3d(9818.2067026569275,-10899.854228217155,-1930.7694091796875),
    MbVec3d(9812.0504526569275,-10861.854228217155,-1934.7504882812500),

    MbVec3d(9843.4672272596508,-10850.299587312154,-1929.6202392578125),
    MbVec3d(9848.6088354458334,-10886.899006623775,-1926.5780029296875),
    MbVec3d(9821.3775410449598,-10900.405928235501,-1932.6950683593750),
    MbVec3d(9814.9312862149673,-10862.298144783825,-1936.4967041015625)
  };

  double color[8];
  for (size_t i=0; i < 8; ++i)
    color[i] = coords[i][0];

  size_t numFacets[] = { 6 };
  size_t facets[24] = { 0,1,2,3, 0,4,5,1, 3,2,6,7, 4,7,6,5 ,0,3,7,4 ,1,5,6,2  } ; 
  size_t numNodes[6] = { 4, 4, 4, 4, 4, 4 };
  MbVolumeMeshPolyhedron<> *mesh = new MbVolumeMeshPolyhedron<>(coords,coords+8,facets,facets+24,numNodes,numNodes+6,numFacets,numFacets+1);
  mesh->addScalarSet(new MbScalarSetI<double>(color,color+8,"colorSet"));
  return mesh;
}

MbVolumeMeshPolyhedron<>* buildPolyhedron(double a = 0.5)
{
  MbVec3d coords[] = 
  {
    MbVec3d(-a*10,-a*10,-10),
    MbVec3d(a*10,-a*10,-10),
    MbVec3d(a*10,a*10,-10),
    MbVec3d(-a*10,a*10,-10),

    MbVec3d(-10,-10,10),
    MbVec3d(10,-10,10),
    MbVec3d(10,10,10),
    MbVec3d(-10,10,10)
  };

  double color[8];
  for (size_t i=0; i < 8; ++i)
    color[i] = coords[i][0];
 
  size_t numFacets[] = { 6 };
  size_t facets[24] = { 0,1,2,3, 0,4,5,1, 3,2,6,7, 4,7,6,5 ,0,3,7,4 ,1,5,6,2  } ; 
  size_t numNodes[6] = { 4, 4, 4, 4, 4, 4 };
  MbVolumeMeshPolyhedron<> *mesh = new MbVolumeMeshPolyhedron<>(coords,coords+8,facets,facets+24,numNodes,numNodes+6,numFacets,numFacets+1);
  mesh->addScalarSet(new MbScalarSetI<double>(color,color+8,"colorSet"));
  return mesh;
}

class UserData 
{
public:
  UserData(MoMesh *moMesh, MoScalarSetI *moScalarSetI, MoPredefinedColorMapping* colMap, 
           MoMeshGridPlaneSlice *slice, MoDrawStyle *drawStyle, SoPickStyle* pickStyle) :
      m_drawStyle(drawStyle), m_pickStyle(pickStyle), m_moMesh(moMesh), m_moScalarSetI(moScalarSetI), m_colMap(colMap), m_slice(slice) {}
    
  MoMesh * getMoMesh() { return m_moMesh; }
  MoScalarSetI *getMoScalarSetI() { return m_moScalarSetI;}
  MoPredefinedColorMapping* getColorMap() { return m_colMap; }
  MoMeshGridPlaneSlice* getSlice() { return m_slice; }
  MoDrawStyle* const m_drawStyle;
  SoPickStyle* m_pickStyle;

private:
  MoMesh *m_moMesh;
  MoScalarSetI *m_moScalarSetI;
  MoPredefinedColorMapping* m_colMap;
  MoMeshGridPlaneSlice *m_slice;
};

void
editMesh(void *userData, SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();

  bool handled = false;

  UserData *data = (UserData*) userData;
  MoMesh *moMesh = data->getMoMesh();
  if (moMesh) {
    if (SO_KEY_PRESS_EVENT(ev, NUMBER_1))
    {
      delete m_mesh;
      delete m_colorSet;
      m_mesh = buildPolyhedron();
      handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, NUMBER_2))
    {
      delete m_mesh;
      delete m_colorSet;
      m_mesh = buildOctogon();
      handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, NUMBER_3))
    {
      delete m_mesh;
      delete m_colorSet;
      m_mesh = buildDodecahedron();
      handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, NUMBER_4))
    {
      delete m_mesh;
      delete m_colorSet;
      m_mesh = buildNonPlanarPolyhedron();
     handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, F))
    {
      if (data->m_drawStyle->displayFaces.getValue() == TRUE) 
      {
        data->m_pickStyle->style.setValue(SoPickStyle::UNPICKABLE);
        data->m_drawStyle->displayFaces = FALSE;
      }
      else
      {
        data->m_pickStyle->style.setValue(SoPickStyle::SHAPE);
        data->m_drawStyle->displayFaces = TRUE;
      }
      handled = true;
    }
  }

  if (handled) 
  {
    m_colorSet = m_mesh->getScalarSet("colorSet");
    moMesh->setMesh(m_mesh);
    MoScalarSetI *moScalarSetI = data->getMoScalarSetI();
    moScalarSetI->setScalarSet(m_colorSet);
    data->getColorMap()->minValue = (float)m_colorSet->getMin();
    data->getColorMap()->maxValue = (float)m_colorSet->getMax();
    MbVec3f center((m_mesh->getGeometry().getMax() + m_mesh->getGeometry().getMin()) / 2.0);
    m_manip->draggerPosition.setValue((SbVec3f((float)center[0],(float)center[1],(float)center[2])));
    eventCB->setHandled();
    viewer->viewAll();
  }
}

void
editGridPlaneSlice(void *userData, SoEventCallback *eventCB)
{
  bool handled = false;
  const SoEvent *ev = eventCB->getEvent();

  MoMeshGridPlaneSlice *slice = (MoMeshGridPlaneSlice*) (userData);
  if (slice) {
    if (SO_KEY_PRESS_EVENT(ev, U) )
    {
      slice->step.setValue(slice->step.getValue()/(float)1.5);
      handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, D) )
    {
      slice->step.setValue(slice->step.getValue()*(float)1.5);
      handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, M) )
    {
      m_skinSwitch->whichChild = ( m_skinSwitch->whichChild.getValue()==SO_SWITCH_ALL) ? SO_SWITCH_NONE : SO_SWITCH_ALL;
      handled = true;
    }
    else if (SO_KEY_PRESS_EVENT(ev, S) )
    {
      m_sliceSwitch->whichChild = ( m_sliceSwitch->whichChild.getValue()==SO_SWITCH_ALL) ? SO_SWITCH_NONE : SO_SWITCH_ALL;
      handled = true;
    }
  }

  if (handled) 
    eventCB->setHandled();
}

//------------------------------------------------------------------------
void mouseButtonCallback(void *user_data, SoEventCallback *cb_node) 
{
  MoMeshPointProbe *probe =  (MoMeshPointProbe *)user_data;
  const SoEvent *ev = cb_node->getEvent();
  if (SO_MOUSE_RELEASE_EVENT(ev,BUTTON1)) 
  {
    const SoPickedPoint *picked_point = cb_node->getPickedPoint();
    if (!picked_point) 
      probe->position.setValue(SbVec3f(FLT_MAX,FLT_MAX,FLT_MAX));
    else 
    {
      cout << "picked at " << picked_point->getPoint() << endl;
      probe->position.setValue(picked_point->getPoint()); 
    }
    cb_node->setHandled();
  }
}

//------------------------------------------------------------------------
class ProbeCallback : public MoMeshPointProbe::MoProbeCallback
{
  std::string m_name;
public:
  ProbeCallback(std::string name) : m_name(name) {}
  void motionCallback(size_t cellId, const MeXScalardSetI& scalars, const MeXVec3dSetI& vectors)
  {
   if (cellId != std::numeric_limits<size_t>::max()) {
      cout << m_name << " cell: " << cellId << " ";
      if ( scalars.getSize() > 0)
        cout << "value: " <<scalars.get(0) << "  ";
      if ( vectors.getSize() > 0)
        cout << "vector: " <<vectors.get(0);
      cout << endl;
    }
  }
};

//-----------------------------------------------------------------------------
int
main(int, char **)
{
  cout << "Press 1 for polyhedron mesh" << endl;
  cout << "Press 2 for octogon mesh" << endl;
  cout << "Press 3 for dodecahedron mesh" << endl;
  cout << "Press 4 for polyhedron mesh with nonplanar" << endl;
  cout << "Press u/d increase/decrease grid density" << endl;
  cout << "Press m to toggle cell visibility" << endl;
  cout << "Press s to toggle slice visibility" << endl;
  cout << "Press f to toggle facet visibility" << endl;

  // Init viewer
  Widget my_window = SoXt::init(NAMESTR) ;
  if (my_window == NULL) exit(1) ;

  MoMeshViz::init();

  SoSeparator* sep = new SoSeparator;
  
  m_mesh = buildPolyhedron();
  m_colorSet = m_mesh->getScalarSet("colorSet");

  MoMesh* moMesh = new MoMesh;
  moMesh->setMesh(m_mesh);
  sep->addChild(moMesh);

  MoScalarSetI *moScalarSetI = new MoScalarSetI;
  moScalarSetI->setScalarSet(m_colorSet);
  sep->addChild(moScalarSetI);
 
  MoPredefinedColorMapping* colMap = new MoPredefinedColorMapping;
  colMap->minValue = (float)m_colorSet->getMin();
  colMap->maxValue = (float)m_colorSet->getMax();
  colMap->predefColorMap = MoPredefinedColorMapping::STANDARD;
  sep->addChild(colMap);

  SoSeparator* sliceSep = new SoSeparator;
  sep->addChild(sliceSep);

  m_sliceSwitch = new SoSwitch;
  sliceSep->addChild(m_sliceSwitch);
  m_sliceSwitch->whichChild = SO_SWITCH_ALL;

  SoMaterial* red = new SoMaterial;
  m_sliceSwitch->addChild(red);
  red->diffuseColor = SbColor(1,0,0);

  MbVec3f center((m_mesh->getGeometry().getMax() + m_mesh->getGeometry().getMin()) / 2.0);
  SbPlane plane(SbVec3f(0,0,1),SbVec3f(center[0],center[1],center[2]));
  // Manipulator to move the plane slice plane (as a clip plane)
  m_manip = new SoClipPlaneManip;
  m_manip->plane.setValue(plane);
  // Turn off clip plane visibility to avoid flickering of plane slice faces.
  m_manip->on.setValue(FALSE); 
  m_manip->draggerPosition.setValue((SbVec3f((float)center[0],(float)center[1],(float)center[2])));
  m_sliceSwitch->addChild(m_manip);

  // Grid plane slice
  MoMeshGridPlaneSlice *gridPlaneSlice = new MoMeshGridPlaneSlice;
  m_sliceSwitch->addChild(gridPlaneSlice);
  gridPlaneSlice->step = .5;
  gridPlaneSlice->plane.connectFrom(&m_manip->plane);
  gridPlaneSlice->colorScalarSetId = 0;
  gridPlaneSlice->parallel = FALSE;

  // Transparent Mesh Skin
  m_skinSwitch = new SoSwitch;
  sep->addChild(m_skinSwitch);
  m_skinSwitch->whichChild = SO_SWITCH_ALL;

  MoMaterial* material = new MoMaterial;
  material->enhancedColoring = TRUE;
  m_skinSwitch->addChild(material);
  MoDrawStyle* style = new MoDrawStyle;
  style->displayEdges = TRUE;
  style->displayFaces = FALSE;
  m_skinSwitch->addChild(style);
  SoPickStyle* pickStyle = new SoPickStyle;
  pickStyle->style.setValue(SoPickStyle::UNPICKABLE);
  m_skinSwitch->addChild(pickStyle);
  SoLightModel* lModel = new SoLightModel;
  lModel->model.setValue(SoLightModel::BASE_COLOR);
  m_skinSwitch->addChild(lModel);

  MoMeshSkin *skin = new MoMeshSkin;
  skin->colorScalarSetId = 0;
  m_skinSwitch->addChild(skin);

  // Probe sectio
  MoMeshPointProbe* probe2 = new MoMeshPointProbe;
  ProbeCallback cb2("origin");
  probe2->setProbeCallback(cb2);
  sep->addChild(probe2);

  // This mesh is used to connect the grid generated mesh as input of the vectors
  MoMesh* gridMesh = new MoMesh;
  gridMesh->connectFrom(gridPlaneSlice);
  sep->addChild(gridMesh);

  SoEventCallback *myCallbackNode = new SoEventCallback;
  myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(), editGridPlaneSlice, gridPlaneSlice); 
  sep->addChild(myCallbackNode);

  SoEventCallback *myCallbackNode2 = new SoEventCallback;
  UserData data(moMesh,moScalarSetI,colMap,gridPlaneSlice,style,pickStyle);
  myCallbackNode2->addEventCallback(SoKeyboardEvent::getClassTypeId(), editMesh, &data); 
  sep->addChild(myCallbackNode2);

  SoEventCallback *myCallbackNode3= new SoEventCallback;
  myCallbackNode3->addEventCallback(SoMouseButtonEvent::getClassTypeId(), mouseButtonCallback, probe2);
  sep->addChild(myCallbackNode3);

  viewer = new SoXtExaminerViewer(my_window);
  viewer->setSceneGraph(sep);
  viewer->setSize(SbVec2s(1024,768));
  viewer->show();
  viewer->viewAll();

  // dialog
  SoXt::show(my_window);
  SoXt::mainLoop();

  delete viewer;

  MoMeshViz::finish();
  SoXt::finish();

  return 0;
}

#if defined(_MSC_VER)
#pragma warning( pop ) 
#endif


