#include "Demo_QuadraticSkin.h"

#include <MeshVizXLM/mapping/MoMeshViz.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSetI.h>
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoTessellator.h>
#include <MeshVizXLM/tessellator/MiTessellator.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/events/SoKeyboardEvent.h> 

#include <Inventor/STL/iostream>
#include <Inventor/STL/fstream>
#include <Inventor/STL/vector>
using namespace std;

void keyboardCallback(void *_this, SoEventCallback *eventCB);


//---------------------------------------------------------------------
Demo_QuadraticSkin::Demo_QuadraticSkin() :
m_width(2),
m_mesh(m_width),
m_dataset(m_mesh),
m_maxError(0.2),
m_minError(0.0005),
m_incrError((m_maxError-m_minError)/30),
m_metricError((m_maxError+m_minError)/2),
m_edgeErrorMetricGeometry(m_metricError)
{
  run(m_mesh,m_dataset);
}

//-----------------------------------------------------------------------------
int
Demo_QuadraticSkin::run(const MiVolumeMeshUnstructured& mesh, const MbScalardSetI& dataset)
{
  string demoTitle("MeshVizXLM sample program showing how to display quadratic cells");

  Widget my_window = SoXt::init(demoTitle.c_str()) ;
  if (my_window == NULL) exit(1) ;

  cout << "T to toggle quadratic tesselation " << std::endl;
  cout << "I to increase tesselation (only when quadratic tesselation is on) " << std::endl;
  cout << "D to decrease tesselation (only when quadratic tesselation is on)" << std::endl;
  cout << "P/M to modify the geometry " << std::endl;

  MoMeshViz::init();

  SoNode* root = buildSceneGraph(mesh,dataset);

  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(my_window);
  viewer->setSize(SbVec2s(1024,768));
  viewer->setSceneGraph(root);
  viewer->setTitle(demoTitle.c_str());
  viewer->show();
  viewer->viewAll();

  // dialog
  SoXt::show(my_window);
  SoXt::mainLoop();

  delete viewer;
  MoMeshViz::finish();
  SoXt::finish();
  return 0;
}

//-----------------------------------------------------------------------------
SoNode* 
Demo_QuadraticSkin::buildSceneGraph(const MiVolumeMeshUnstructured& mesh, const MbScalardSetI& dataset)
{
  //  ----- basic skin -------
  MiTessellator* basicTessellator = MiTessellator::getNewTessellatorBasic();
  MoTessellator* basicTessellatorNode = new MoTessellator;
  basicTessellatorNode->setTessellator(basicTessellator);

  MoMesh* meshNode = new MoMesh;
  meshNode->setMesh(&mesh); 

  MoScalarSetI* scalarSetNode = new MoScalarSetI;
  scalarSetNode->setScalarSet(&dataset);

  MoMeshSkin *basicSkinSurface = new MoMeshSkin;
  basicSkinSurface->parallel = FALSE;

  SoSeparator* basicSep = new SoSeparator;

  m_basicSwitch = new SoSwitch;
  m_basicSwitch->whichChild = SO_SWITCH_NONE;

  // ----- tessellated skin -----

  MiTessellator* geometryTessellator = MiTessellator::getNewTessellatorGeometry(m_edgeErrorMetricGeometry);
  m_geometryTessellatorNode = new MoTessellator;
  m_geometryTessellatorNode->setTessellator(geometryTessellator);

  MoMeshSkin *tessellateSkinSurface = new MoMeshSkin;

  SoSeparator* tessellatedSep = new SoSeparator;

  m_tessellatedSwitch = new SoSwitch;
  m_tessellatedSwitch->whichChild = SO_SWITCH_ALL;

  // ----- color mapping -----
  MoPredefinedColorMapping* colorMapping = new MoPredefinedColorMapping;
  colorMapping->predefColorMap = MoPredefinedColorMapping::STANDARD;
  colorMapping->minValue = float(dataset.getMin());
  colorMapping->maxValue = float(dataset.getMax());

  SoShapeHints* shapeHints = new SoShapeHints;
  shapeHints->creaseAngle = 1.0;
  shapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE  ;

  SoEventCallback *myCallbackNode = new SoEventCallback;
  myCallbackNode->addEventCallback(SoKeyboardEvent::getClassTypeId(),keyboardCallback,this); 

  MoMaterial * material = new MoMaterial;
  material->enhancedColoring = TRUE;

  SoSeparator* sep = new SoSeparator;
  {
    sep->addChild(shapeHints);
    sep->addChild(colorMapping);
    sep->addChild(material);

    sep->addChild(scalarSetNode);
    sep->addChild(meshNode);

    sep->addChild(basicSep);
    {
      basicSep->addChild(m_basicSwitch);
      {
        m_basicSwitch->addChild(basicTessellatorNode);
        m_basicSwitch->addChild(basicSkinSurface);
      }
    }

    sep->addChild(tessellatedSep);
    {
      tessellatedSep->addChild(m_tessellatedSwitch);
      {
        m_tessellatedSwitch->addChild(m_geometryTessellatorNode);
        m_tessellatedSwitch->addChild(tessellateSkinSurface);
      }
    }
    sep->addChild(myCallbackNode);
  }
  return sep;
}


//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::keyPressed(SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();
  bool handled = true;

  if (SO_KEY_PRESS_EVENT(ev, T) )
    keyPressedT();
  else if (SO_KEY_PRESS_EVENT(ev, D)) 
    keyPressedD();
  else if (SO_KEY_PRESS_EVENT(ev, I))
    keyPressedI();
  else if (SO_KEY_PRESS_EVENT(ev, M))
    keyPressedM();
  else if (SO_KEY_PRESS_EVENT(ev, P))
    keyPressedP();
  else
    handled = false;

  if (handled) 
    eventCB->setHandled();
}

//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::keyPressedT()
{
  m_basicSwitch->whichChild = (m_basicSwitch->whichChild.getValue() == SO_SWITCH_NONE) 
    ? SO_SWITCH_ALL 
    : SO_SWITCH_NONE;

  m_tessellatedSwitch->whichChild = (m_tessellatedSwitch->whichChild.getValue() == SO_SWITCH_NONE) 
    ? SO_SWITCH_ALL 
    : SO_SWITCH_NONE;

  if (m_tessellatedSwitch->whichChild.getValue() == SO_SWITCH_ALL)
    cout << "Displaying quadratic tesselation " << std::endl;
  else 
    cout << "Displaying without tesselation " << std::endl;
}

//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::keyPressedD()
{
  if (m_metricError+m_incrError < m_maxError)
  {
    m_metricError += m_incrError;
    updateTessellation(m_metricError);
  }
}

//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::keyPressedI()
{
  if (m_metricError-m_incrError > m_minError)
  {
    m_metricError -= m_incrError;
    updateTessellation(m_metricError);
  }
}

//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::keyPressedM()
{
  --m_width;
  m_mesh.update(m_width);
}

//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::keyPressedP()
{
  ++m_width;
  m_mesh.update(m_width);
}

//---------------------------------------------------------------------
void 
Demo_QuadraticSkin::updateTessellation(double metricError)
{
  cout << "updateTessellation, tolerance = " << metricError << std::endl;
  m_edgeErrorMetricGeometry.setMaxError(metricError);
  m_geometryTessellatorNode->touch();
}

//---------------------------------------------------------------------
void
keyboardCallback(void *_this, SoEventCallback *eventCB)
{
  ((Demo_QuadraticSkin*)_this)->keyPressed(eventCB);
}


