#include "DemoQuadraticSurface.h"

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/events/SoKeyboardEvent.h> 
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoCoordinate3.h>
#include <Inventor/nodes/SoMarkerSet.h>
#include <Inventor/manips/SoClipPlaneManip.h>
#include <Inventor/SoPickedPoint.h>

#include <MeshVizXLM/mapping/nodes/MoMeshOutline.h>
#include <MeshVizXLM/mapping/nodes/MoMeshClipLine.h>
#include <MeshVizXLM/mapping/nodes/MoMeshCellShape.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>

#include <MeshVizXLM/tessellator/MiTessellator.h>

using namespace::std;

//---------------------------------------------------------------------
DemoQuadraticSurface::DemoQuadraticSurface() :
Demo("MeshVizXLM Quadratic Surface"),
m_dataset(m_mesh),
m_maxError(0.2),
m_minError(0.0005),
m_incrError((m_maxError-m_minError)/30),
m_metricError((m_maxError+m_minError)/2),
m_enableTessellation(true),
m_edgeErrorMetricGeometry(m_metricError)
{
  m_basicTessellator = MiTessellator::getNewTessellatorBasic();
  m_geometryTessellator = MiTessellator::getNewTessellatorGeometry(m_edgeErrorMetricGeometry);
}

//---------------------------------------------------------------------
DemoQuadraticSurface::~DemoQuadraticSurface()
{
  delete m_basicTessellator;
  delete m_geometryTessellator;
}

//---------------------------------------------------------------------
const MiMesh* DemoQuadraticSurface::getInputMesh(MeshType& meshType)
{
  meshType = SURFACE_MESH_UNSTRUCTURED;
  return &m_mesh;
}

MiTessellator* DemoQuadraticSurface::getTessellator()
{
  if (m_enableTessellation)
    return m_geometryTessellator;
  else
    return m_basicTessellator;
}

//---------------------------------------------------------------------
const MbScalardSetI* DemoQuadraticSurface::getScalarSetI()
{
  return &m_dataset;
}

//---------------------------------------------------------------------
bool DemoQuadraticSurface::displayInstructions() const
{
  cout << "T to toggle quadratic tesselation " << endl;
  cout << "F to toggle faces visibility " << endl;
  cout << "E to toggle edges visibility " << endl;
  cout << "P to toggle points visibility " << endl;
  cout << "O to toggle outline visibility" << endl;
  cout << "C to toggle clipline visibility" << endl;
  cout << "I to increase tesselation (only when quadratic tesselation is on) " << endl;
  cout << "D to decrease tesselation (only when quadratic tesselation is on)" << endl << endl;

  return true;
}

//---------------------------------------------------------------------
// Method called each time the mouse is moved in the viewer window
void DemoQuadraticSurface::mouseMoved(SoEventCallback *eventCB)
{
  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell id.
      size_t cellId = fdetail->getCellIndex();
      m_cellShape->cellIndices.set1Value(0,(int)cellId);
    }
  }


  eventCB->setHandled();
}

//---------------------------------------------------------------------
void DemoQuadraticSurface::keyPressed(SoEventCallback *eventCB)
{
  const SoEvent *ev = eventCB->getEvent();
  bool handled = true;

  if (SO_KEY_PRESS_EVENT(ev, T) )
    keyPressedT();
  else if (SO_KEY_PRESS_EVENT(ev, P)) 
    keyPressedP();
  else if (SO_KEY_PRESS_EVENT(ev, E)) 
    keyPressedE();
  else if (SO_KEY_PRESS_EVENT(ev, F)) 
    keyPressedF();
  else if (SO_KEY_PRESS_EVENT(ev, O)) 
    keyPressedO();
  else if (SO_KEY_PRESS_EVENT(ev, C)) 
    keyPressedC();
  else if (SO_KEY_PRESS_EVENT(ev, D)) 
    keyPressedD();
  else if (SO_KEY_PRESS_EVENT(ev, I))
    keyPressedI();
  else
    handled = false;

  if (handled) 
    eventCB->setHandled();
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedT()
{
  m_enableTessellation = !m_enableTessellation;
  if (m_enableTessellation)
    cout << "Displaying quadratic tesselation " << std::endl;
  else 
    cout << "Displaying without tesselation " << std::endl;
  this->setMeshData(false,true);
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedP()
{
  m_ds->displayPoints = !m_ds->displayPoints.getValue();
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedE()
{
  m_ds->displayEdges = !m_ds->displayEdges.getValue();
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedF()
{
  m_ds->displayFaces = !m_ds->displayFaces.getValue();
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedO()
{
  m_outlineSwitch->whichChild = (m_outlineSwitch->whichChild.getValue() == SO_SWITCH_ALL) 
    ? SO_SWITCH_NONE : SO_SWITCH_ALL;
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedC()
{
  m_cliplineSwitch->whichChild = (m_cliplineSwitch->whichChild.getValue() == SO_SWITCH_ALL) 
    ? SO_SWITCH_NONE : SO_SWITCH_ALL;
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedD()
{
  if (m_metricError+m_incrError < m_maxError)
  {
    m_metricError += m_incrError;
    updateTessellation(m_metricError);
  }
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::keyPressedI()
{
  if (m_metricError-m_incrError > m_minError)
  {
    m_metricError -= m_incrError;
    updateTessellation(m_metricError);
  }
}

//---------------------------------------------------------------------
void 
DemoQuadraticSurface::updateTessellation(double metricError)
{
  cout << "updateTessellation, tolerance = " << metricError << std::endl;
  m_edgeErrorMetricGeometry.setMaxError(metricError);
}


//---------------------------------------------------------------------
void DemoQuadraticSurface::getMeshAttributes(MoDrawStyle*& ds, MoMaterial*& mat, SoDrawStyle*& ids, SoPickStyle*& /*ps*/)
{
  ds = m_ds = new MoDrawStyle;
  m_ds->displayPoints = true;
  mat = m_mat = new MoMaterial;
  m_mat->lineColoring = MoMaterial::CONTOURING;
  m_mat->pointColoring = MoMaterial::COLOR;
  m_mat->pointColor = SbColor(1,0,1);
  ids = new SoDrawStyle;
  ids->pointSize = 5.0f;
}

//---------------------------------------------------------------------
MiExtractorCallback* DemoQuadraticSurface::getExtractorCallback()
{
  return &m_extractorCB;
}

//---------------------------------------------------------------------
void DemoQuadraticSurface::insertRepresentations(SoSeparator* sep)
{
  SoMaterial* meshNodesMaterial = new SoMaterial;
  meshNodesMaterial->diffuseColor = SbColor(1,0,1);
  sep->addChild(meshNodesMaterial);

  SoDrawStyle* ids = new SoDrawStyle;
  ids->pointSize = 5.0f;
  sep->addChild(ids);

  m_outlineSwitch = new SoSwitch;
  m_outlineSwitch->whichChild = SO_SWITCH_ALL;
  sep->addChild(m_outlineSwitch);
  MoMeshOutline *outline = new MoMeshOutline;
  outline->colorScalarSetId = -1;
  m_outlineSwitch->addChild(outline);
  outline->setExtractorCallback(&m_extractorCB);

  MoDrawStyle *drawStyleCellShape = new MoDrawStyle;
  drawStyleCellShape->displayEdges = true;
  drawStyleCellShape->displayPoints = true;
  sep->addChild(drawStyleCellShape);

  MoMaterial *materialCellShape = new MoMaterial;
  materialCellShape->lineColoring = MoMaterial::COLOR;
  materialCellShape->lineColor = SbColor(0,0,1);
  materialCellShape->pointColoring = MoMaterial::COLOR;
  materialCellShape->pointColor = SbColor(0,0,1);
  sep->addChild(materialCellShape);

  m_cellShape = new MoMeshCellShape;
  sep->addChild(m_cellShape);

  m_cliplineSwitch = new SoSwitch;
  m_cliplineSwitch->whichChild = SO_SWITCH_NONE;
  sep->addChild(m_cliplineSwitch);

  MoDrawStyle *drawStyleClipLine = new MoDrawStyle;
  drawStyleClipLine->displayEdges = true;
  drawStyleClipLine->displayPoints = false;
  m_cliplineSwitch->addChild(drawStyleClipLine);

  MoMesh* surfaceMesh = new MoMesh;
  this->connectToMeshRepresentation(surfaceMesh);
  m_cliplineSwitch->addChild(surfaceMesh);
  
  MoMaterial* mat = new MoMaterial;
  mat->lineColoring = MoMaterial::COLOR;
  mat->lineColor.setValue(0,0,1);
  m_cliplineSwitch->addChild(mat);

  // Manipulator to move the clip line
  SoClipPlaneManip *manip = new SoClipPlaneManip;
  manip->plane.setValue(SbPlane(SbVec3f(1,0,1),5.0f));
  // Turn off clip plane visibility to avoid flickering of plane slice faces.
  manip->on.setValue(FALSE); 
  manip->draggerPosition.setValue(this->getCenter());
  m_cliplineSwitch->addChild(manip);

  MoMeshClipLine* clipline = new MoMeshClipLine;
  clipline->plane.connectFrom(&manip->plane);
  m_cliplineSwitch->addChild(clipline);

}


