#ifndef  _QuadraticMesh_h
#define  _QuadraticMesh_h

#include <MeshVizXLM/mesh/MiVolumeMeshUnstructured.h>
#include <MeshVizXLM/mesh/data/MiDataSetI.h>

#include "QuadraticHexahedronCell.h"

#include <Inventor/SbBox.h>

#include <Inventor/STL/vector>

static double s_radiusMin = 5;
static double c1 = s_radiusMin * sqrt(2.)/2.;
static double s_radiusMax = 8;
static double c3 = s_radiusMax * sqrt(2.)/2.;

//-----------------------------------------------------------------------------
class QuadraticGeometry : public MiGeometryI
{
public:
  QuadraticGeometry();
  const std::vector<MbVec3d>& getCoords() const
  {
    return m_coords;
  }

  virtual MbVec3d getCoord(size_t i) const 
  {
    return m_coords[i];
  }

  MbVec3d getCartCoord(const MbVec3d& cylCoord)
  {
    return MbVec3d(
      cylCoord[0] * cos(cylCoord[1]),
      cylCoord[0] * sin(cylCoord[1]),
      cylCoord[2]
    );
  }

  virtual MbVec3d getMin() const 
  {
    return MbVec3d(m_bbox.getMin());
  }
  /**
  * Returns the maximum value of this set.
  */
  virtual MbVec3d getMax() const 
  {
    return MbVec3d(m_bbox.getMax());
  }


  virtual size_t getTimeStamp() const 
  {
    return 1;
  }

  MbVec3d getSymetricToXPlane(size_t nodeId) const
  {
    MbVec3d pt = m_coords[nodeId];
    return MbVec3d(-pt[0],pt[1],pt[2]);
  }

  MbVec3d getSymetricToYPlane(size_t nodeId) const
  {
    MbVec3d pt = m_coords[nodeId];
    return MbVec3d(pt[0],-pt[1],pt[2]);
  }

private:
  std::vector<MbVec3d> computeQuadraticFace4(double alpha, double rmin, double rmax, double width) const;
  std::vector<MbVec3d> computeQuadraticFace8(double alpha, double rmin, double rmax, double width) const;

  std::vector<MbVec3d> m_coords;
  static MbVec3d s_cylcoords[20];

  SbBox3d m_bbox;
};

//-----------------------------------------------------------------------------
class QuadraticTopology : public MiVolumeTopologyExplicitI
{
public:
  QuadraticTopology() 
    : m_cell0(0,1,2,3,4,5,6,7, 8,9,10,11,12,13,14,15,16,17,18,19),
      m_cell1(4,5,6,7, 20,21,22,23, 12,13,14,15, 24,25,26,27, 28,29,30,31),
      m_cell2(20,21,22,23, 32,33,34,35, 24,25,26,27, 36,37,38,39, 40,41,42,43),
      m_cell3(32,33,34,35, 0,1,2,3, 36,37,38,39, 8,9,10,11, 44,45,46,47)      
  {
  }

  virtual const MiVolumeCell* getCell(size_t id) const
  {
    switch (id)
    {
    default:
    case 0: return &m_cell0;
    case 1: return &m_cell1;
    case 2: return &m_cell2;
    case 3: return &m_cell3;
    }
  }

  virtual size_t getEndNodeId() const
  {
    return 48;
  }

  virtual size_t getNumCells() const
  {
    return 4;
  }

  virtual size_t getTimeStamp() const 
  {
    return 1;
  }

private:
  QuadraticHexahedronCell m_cell0;
  QuadraticHexahedronCell m_cell1;
  QuadraticHexahedronCell m_cell2;
  QuadraticHexahedronCell m_cell3;
};


//-----------------------------------------------------------------------------
class QuadraticMesh : public MiVolumeMeshUnstructured
{
public:
  QuadraticMesh() {}

  virtual const MiVolumeTopologyExplicitI& getTopology() const 
  {
    return m_topology;
  }

  virtual const MiGeometryI& getGeometry() const 
  {
    return m_geometry;
  }

  const std::vector<MbVec3d>& getCoords() const
  {
    return m_geometry.getCoords();
  }


private:
  QuadraticGeometry m_geometry;
  QuadraticTopology m_topology;
};

//-----------------------------------------------------------------------------
class ScalarSet : public MiScalardSetI
{
public:
  ScalarSet(const MiVolumeMeshUnstructured& mesh) 
    : m_geometry(mesh.getGeometry())
  {
    m_vmin = 1E30;
    m_vmax = -1E30;
    for (size_t i=mesh.getTopology().getBeginNodeId(); i!=mesh.getTopology().getEndNodeId(); ++i)
    {
      double v = get(i);
      if (v < m_vmin) m_vmin = v;
      if (v > m_vmax) m_vmax = v;
    }
  }
  double get(size_t i) const 
  { 
    // value = distance of point i to Z-axis
    MbVec3d coord = m_geometry.getCoord(i);
    return  sqrt(coord[0]*coord[0] + coord[1]*coord[1]);
  }

  double getMin () const { return m_vmin; }
  double getMax () const { return m_vmax; }
  size_t  getTimeStamp () const  { return 1; }
  std::string getName () const  { return "testData"; }
  DataBinding  getBinding () const { return PER_NODE; }

private:
  const MiGeometryI& m_geometry;
  double m_vmin,m_vmax;
};


#endif 


